/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 05.05.2022
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.lang;

import java.util.AbstractCollection;
import java.util.Collection;
import java.util.Iterator;
import java.util.Objects;
import java.util.function.Function;

/**
 * A read-only view of a collection that maps its elements using mapper.
 * Example:
 * <pre>{@code
 * Pair<Integer, String> pair = new Pair<Integer, String>(1, "test");
 * Collection<Pair<Integer, String>> collection = new ArrayList<Pair<Integer, String>>();
 * collection.add(pair);
 * Collection<Integer> mappedCollection = MappedCollection.newInstance(collection, Pair::getFirst);
 * assert mappedCollection.contains(1);
 * }</pre>
 */
public class MappedCollection<P, T> extends AbstractCollection<T> {
    private Collection<P> underlying;
    private Function<P, T> mapper;

	/**
	 * Constructs a MappedCollection.
	 * @param underlying The underlying collection.
	 * @param mapper The mapper.
	 */
    public MappedCollection(Collection<P> underlying, Function<P, T> mapper) {
        Objects.requireNonNull(underlying);
        Objects.requireNonNull(mapper);

        this.underlying = underlying;
        this.mapper = mapper;
    }

	/**
	 * Same as {@link #MappedCollection(Collection, Function)} with nicer type inference.
     */
    public static <P, T> MappedCollection<P, T> newInstance(Collection<P> underlying, Function<P, T> mapper) {
        return new MappedCollection<P, T>(underlying, mapper);
    };

    @Override
    public Iterator<T> iterator() {
        return new MappedIterator<P, T>(this.underlying.iterator(),this.mapper);
    }

    @Override
    public int size() {
        return this.underlying.size();
    }

    public Collection<P> getUnderlying() {
        return this.underlying;
    }

}
