/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: Dec 27, 2006
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.io;

import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Static helper function for I/O functionalities.
 */
public class IOUtil {

    /**
     * Set to private, because this class defines no public non-static method.
     */
    private IOUtil() {
    }


    /**
     * Read one byte from the Stream.
     *
     * @param is
     * @return The next byte from the stream
     * @throws IOException On read errors.
     * @throws EOFException In situations, where {@link InputStream#read()} would
     *             return <code>-1</code>.
     */
    public static byte readByte(final InputStream is) throws IOException {
        int b = is.read();
        if (b == -1) {
            throw new EOFException("got an EOF");
        } else {
            return (byte) (b & 0xff);
        }
    }

    /**
     * Read a 0x00 terminated String
     *
     * @param is
     *            The Stream to read from.
     * @param maxLength
     *            The maximum amount of bytes to read.
     * @return The Byte array containing the read String.
     * @throws IOException
     *             if something goes wrong.
     */
    public static byte[] readZeroTerminatedString(final InputStream is,
            final int maxLength) throws IOException {

        // read the 0x00 terminated String:
        int i = -1;
        byte[] temp = new byte[maxLength];
        byte val;

        do {

            i++;
            val = readByte(is);

            temp[i] = val;

        } while (val != 0x00 && i < maxLength);

        byte[] result = new byte[i + 1];
        System.arraycopy(temp, 0, result, 0, result.length);

        return result;
    }

    /**
     * Reads a byte array from the given file
     *
     * @param baFile
     *            File to read from
     * @return A new byte array with all the data from the file.
     * @throws IOException
     *             in case of any I/O Problem
     */
    public static byte[] readByteArrayFromFile(File baFile) throws IOException {
        FileInputStream fis = new FileInputStream(baFile);
        try {

            long ll = baFile.length();
            if (ll > Integer.MAX_VALUE) {
                throw new IndexOutOfBoundsException("File ["+baFile+"] is too large for a byte array.");
            }

            int len = (int)ll;
            byte[] ba = new byte[len];
            fis.read(ba, 0, len);
            return ba;
        }
        finally {
            fis.close();
        }
    }


    /**
     * Write the byte array to the file with the given file name
     *
     * @param baFileName
     *            Name of the file to write the byte array to.
     * @param ba
     *            Byte Array to write to the file.
     * @throws IOException
     *             in case of any I/O Problem
     */
    public static void writeToFile(String baFileName, byte[] ba)
            throws IOException {
        writeToFile(new File(baFileName), ba);
    }

    /**
     * Write the byte array to the given file
     *
     * @param baFile
     *            File to write the byte array to.
     * @param ba
     *            Byte Array to write to the file.
     * @throws IOException
     *             in case of any I/O Problem
     */
    public static void writeToFile(File baFile, byte[] ba) throws IOException {
        FileOutputStream fos = new FileOutputStream(baFile);
        fos.write(ba);
        fos.close();
    }

    /**
     * Reads a byte array from the file with the given name
     *
     * @param baFileName
     *            Name of the file to read from
     * @return A new byte array with all the data from the file.
     * @throws IOException
     *             in case of any I/O Problem
     */
    public static byte[] readByteArrayFromFile(String baFileName)
            throws IOException {
        return readByteArrayFromFile(new File(baFileName));
    }

    /**
     * Copy all the data from an INputStream to an OutputStream.
     * The OutputStrema is flushed after the copy operation, but it is not closed
     * in order to allow writing additional data.
     *
     * @param is The source of the data.
     * @param os The destination of the data.
     * @throws IOException Upon I/O errors.
     */
    public static void copyStreams(InputStream is, OutputStream os) throws IOException
    {
        byte [] buf = new byte[4096];

        int n;

        while((n=is.read(buf))>0)
        {
            os.write(buf,0,n);
        }

        os.flush();
    }

    /**
     * @param fileName The name of a file possibly containing a file extension.
     * @return The portion of the <code>fileName</code> after the last '.' character or
     *         <code>null</code>, if <code>fileName</code> contains no '.'
     *         character.
     */
    public static String getFileExtension(String fileName) {

        int pos = fileName.lastIndexOf('.');

        if (pos < 0) return null;

        return fileName.substring(pos+1);
    }

    /**
     * @param fileName The name of a file possibly containing a file extension.
     * @return The portion of the file after up to and excluding the last '.'
     *         character or <code>fileName</code> as a whole, if <code>fileName</code>
     *         contains no '.' character.
     */
    public static String getFileBaseName(String fileName) {

        int pos = fileName.lastIndexOf('.');

        if (pos < 0) return fileName;

        return fileName.substring(0,pos);
    }
}
