/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 30.10.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.formula.ast;

/**
 * A visitor for formula nodes.
 */
public interface FormulaNodeVisitor {

    /**
     * @param constFormulaNode A constant to visit.
     */
    public void visit(ConstFormulaNode constFormulaNode);

    /**
     * @param symbolFormulaNode A symbol to visit.
     */
    public void visit(SymbolFormulaNode symbolFormulaNode);

    /**
     * Default operation for binary operators.
     * 
     * @param binaryOpFormulaNode A binary operation to visit.
     */
    public default void visitBinary(BinaryOpFormulaNode binaryOpFormulaNode) {
        throw new UnsupportedOperationException();
    }
    
    /**
     * @param plusFormulaNode The plus operation to visit.
     */
    public default void visit(PlusFormulaNode plusFormulaNode) {
        this.visitBinary(plusFormulaNode);
    }

    /**
     * @param minusFormulaNode The minus operation to visit.
     */
    public default void visit(MinusFormulaNode minusFormulaNode) {
        this.visitBinary(minusFormulaNode);
    }

    /**
     * @param multiplyFormulaNode The multiply operation to visit.
     */
    public default void visit(MultiplyFormulaNode multiplyFormulaNode) {
        this.visitBinary(multiplyFormulaNode);
    }

    /**
     * @param divideFormulaNode The division operation to visit.
     */
    public default void visit(DivideFormulaNode divideFormulaNode) {
        this.visitBinary(divideFormulaNode);
    }

    /**
     * @param powerFormulaNode The power operation to visit.
     */
    public default void visit(PowerFormulaNode powerFormulaNode) {
        this.visitBinary(powerFormulaNode);
    }

    /**
     * @param andFormulaNode The and operation to visit.
     */
    public default void visit(AndFormulaNode andFormulaNode) {
        this.visitBinary(andFormulaNode);
    }

    /**
     * @param orFormulaNode The or operation to visit.
     */
    public default  void visit(OrFormulaNode orFormulaNode) {
        this.visitBinary(orFormulaNode);
    }

    /**
     * @param lessFormulaNode The less operation to visit.
     */
    public default void visit(LessFormulaNode lessFormulaNode) {
        this.visitBinary(lessFormulaNode);
    }

    /**
     * @param leqFormulaNode The less than or equals operation to visit.
     */
    public default void visit(LeqFormulaNode leqFormulaNode) {
        this.visitBinary(leqFormulaNode);
    }

    /**
     * @param greaterFormulaNode The greater operation to visit.
     */
    public default void visit(GreaterFormulaNode greaterFormulaNode) {
        this.visitBinary(greaterFormulaNode);
    }

    /**
     * @param geqFormulaNode The greater than or equals operation to visit.
     */
    public default void visit(GeqFormulaNode geqFormulaNode) {
        this.visitBinary(geqFormulaNode);
    }

    /**
     * @param equalsFormulaNode The equals operation to visit.
     */
    public default void visit(EqualsFormulaNode equalsFormulaNode) {
        this.visitBinary(equalsFormulaNode);
    }

    /**
     * @param neqFormulaNode The not equals operation to visit.
     */
    public default void visit(NeqFormulaNode neqFormulaNode) {
        this.visitBinary(neqFormulaNode);
    }

    /**
     * @param functionFormulaNode The function call to visit.
     */
    public void visit(FunctionFormulaNode functionFormulaNode);

    /**
     * @param negFormulaNode Unary negation operation to visit.
     */
    public void visit(NegFormulaNode negFormulaNode);

    /**
     * @param notFormulaNode Unary logical not operation to visit.
     */
    public void visit(NotFormulaNode notFormulaNode);

    /**
     * @param switchFormulaNode Switch-case operation to visit.
     */
    public void visit(SwitchFormulaNode switchFormulaNode);
    
}
