/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 30.10.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.formula.ast;

import java.io.Serializable;

import org.clazzes.util.formula.SymbolValues;

/**
 * The abstract base node for formula nodes.
 */
public abstract class FormulaNode implements Serializable {

    private static final long serialVersionUID = 4079346188178220859L;

    /**
     * @return The operator name in C-style syntax.
     */
    abstract public String getOperator();
    
    /**
     * The precedence (higher numbers take precedence over lower numbers) is defined as:
     * <table>
     *     <caption>Detailed precedence, mostly known from math</caption>
     * <tr><td>-1</td><td>constants, symbols.</td></tr>
     * <tr><td>1</td><td>?:</td></tr>
     * <tr><td>2</td><td>||</td></tr>
     * <tr><td>3</td><td>&amp;&amp;</td></tr>
     * <tr><td>4</td><td>==,&ne;</td></tr>
     * <tr><td>5</td><td>&lt;,&lt;=,&gt;,&gt;=</td></tr>
     * <tr><td>6</td><td>+,-</td></tr>
     * <tr><td>7</td><td>*,/</td></tr>
     * <tr><td>8</td><td>^</td></tr>
     * <tr><td>9</td><td>!,- (unary)</td></tr>
     * <tr><td>10</td><td>function call</td></tr>
     * </table>
     *
     * @return The operators precedence.
     */
    abstract public int getPrecedence();
    
    /**
     * @param symbols The provided symbol values.
     * @return The value of the node after replacing the symbols.
     */
    abstract public double evaluate(SymbolValues symbols);
    
    /**
     * Accept a visitor.
     * @param visitor The visitor to execute.
     */
    abstract public void accept(FormulaNodeVisitor visitor);
}
