/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 31.10.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.formula.ast;

import java.text.Format;
import java.util.Locale;

import org.clazzes.util.formula.FormulaHelper;
import org.clazzes.util.formula.FormulaOptions;
import org.clazzes.util.ryu.RyuNumberFormat;

/**
 * A string-formatting visitor
 */
public class FormattingFormulaNodeVisitor implements FormulaNodeVisitor {

    private final Format numberFormat;
    private final int options;
    private final StringBuffer sb;

    /**
     * @param numberFormat A user-supplied number formatter or <code>null</code> to
     *    to use {@link Double#toString()} for number formatting.
     * @param options Options a bit mask composed of options in {@link FormulaOptions}.
     * @see FormulaOptions#LEGACY_BUILTINS
     */
    public FormattingFormulaNodeVisitor(Format numberFormat, int options) {
        
        this.sb = new StringBuffer();
        this.numberFormat = numberFormat;
        this.options = options;
    }

    /**
     * Instantiate a formula formatter with an instance of {@link RyuNumberFormat}
     * 
     * @param locale The locale used to format numbers.
     * @param options Options a bit mask composed of options in {@link FormulaOptions}.
     * @see FormulaOptions#LEGACY_BUILTINS
     */
    public FormattingFormulaNodeVisitor(Locale locale, int options) {
        
        this(new RyuNumberFormat(locale),options);
    }

    /**
     * Instantiate a formula formatter with an instance of {@link RyuNumberFormat}
     * and {@link FormulaOptions#DEFAULT}.
     * 
     * @param locale The locale used to format numbers.
     */
    public FormattingFormulaNodeVisitor(Locale locale) {
        
        this(new RyuNumberFormat(locale),FormulaOptions.DEFAULT);
    }

    /**
     * Instantiate a formula formatter using {@link Double#toString()} and
     * {@link FormulaOptions#DEFAULT}.
     */
    public FormattingFormulaNodeVisitor() {
        this((Format)null,FormulaOptions.DEFAULT);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.formula.ast.FormulaNodeVisitor#visit(org.clazzes.util.formula.ast.ConstFormulaNode)
     */
    @Override
    public void visit(ConstFormulaNode constFormulaNode) {
        
        if (this.numberFormat == null) {
            this.sb.append(constFormulaNode.getValue());
        }
        else {
            this.numberFormat.format(constFormulaNode.getValue(),this.sb,null);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.formula.ast.FormulaNodeVisitor#visit(org.clazzes.util.formula.ast.SymbolFormulaNode)
     */
    @Override
    public void visit(SymbolFormulaNode symbolFormulaNode) {
        this.sb.append(symbolFormulaNode.getSymbol());
    }

    @Override
    public void visitBinary(BinaryOpFormulaNode binaryOpFormulaNode) {
        
        FormulaNode lhs = binaryOpFormulaNode.getLhs();
        FormulaNode rhs = binaryOpFormulaNode.getRhs();
        
        if (FormulaHelper.needLeftBraces(binaryOpFormulaNode.getPrecedence(),lhs.getPrecedence())) {
            this.sb.append('(');
            lhs.accept(this);
            this.sb.append(')');
        }
        else {
            lhs.accept(this);
        }
        
        this.sb.append(binaryOpFormulaNode.getOperator());

        if (FormulaHelper.needRightBraces(binaryOpFormulaNode.getPrecedence(),rhs.getPrecedence())) {
            this.sb.append('(');
            rhs.accept(this);
            this.sb.append(')');
        }
        else {
            rhs.accept(this);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.formula.ast.FormulaNodeVisitor#visit(org.clazzes.util.formula.ast.FunctionFormulaNode)
     */
    @Override
    public void visit(FunctionFormulaNode functionFormulaNode) {
        
        String f = functionFormulaNode.getOperator();
        
        if ((this.options & FormulaOptions.LEGACY_BUILTINS) != 0) {
            switch(f) {
            case "log":
                this.sb.append("ln");
                break;
            case "log10":
                this.sb.append("log");
                break;
            case "signum":
                this.sb.append("sgn");
                break;
            default:
                this.sb.append(f);
            }
        }
        else {
            this.sb.append(f);
        }
        
        this.sb.append('(');
        functionFormulaNode.getRhs().accept(this);
        this.sb.append(')');
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.formula.ast.FormulaNodeVisitor#visit(org.clazzes.util.formula.ast.NegFormulaNode)
     */
    @Override
    public void visit(NegFormulaNode negFormulaNode) {
        
        FormulaNode rhs = negFormulaNode.getRhs();
        
        if (rhs.getPrecedence() >= 0 && rhs.getPrecedence() < negFormulaNode.getPrecedence()) {
            this.sb.append("-(");
            negFormulaNode.getRhs().accept(this);
            this.sb.append(')');      
        }
        else {
            this.sb.append('-');
            negFormulaNode.getRhs().accept(this);
        }   
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.formula.ast.FormulaNodeVisitor#visit(org.clazzes.util.formula.ast.NotFormulaNode)
     */
    @Override
    public void visit(NotFormulaNode notFormulaNode) {

        FormulaNode rhs = notFormulaNode.getRhs();
        
        if (rhs.getPrecedence() >= 0 && rhs.getPrecedence() < notFormulaNode.getPrecedence()) {
            this.sb.append("!(");
            notFormulaNode.getRhs().accept(this);
            this.sb.append(')');      
        }
        else {
            this.sb.append('!');
            notFormulaNode.getRhs().accept(this);
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.formula.ast.FormulaNodeVisitor#visit(org.clazzes.util.formula.ast.SwitchFormulaNode)
     */
    @Override
    public void visit(SwitchFormulaNode switchOpFormulaNode) {
        
        FormulaNode cond = switchOpFormulaNode.getCondition();
        FormulaNode lhs = switchOpFormulaNode.getLhs();
        FormulaNode rhs = switchOpFormulaNode.getRhs();
        
        if (FormulaHelper.needLeftBraces(switchOpFormulaNode.getPrecedence(),cond.getPrecedence())) {
            this.sb.append('(');
            cond.accept(this);
            this.sb.append(')');
        }
        else {
            cond.accept(this);
        }

        this.sb.append('?');

        if (FormulaHelper.needLeftBraces(switchOpFormulaNode.getPrecedence(),lhs.getPrecedence())) {
            this.sb.append('(');
            lhs.accept(this);
            this.sb.append(')');
        }
        else {
            lhs.accept(this);
        }
        
        this.sb.append(':');

        if (FormulaHelper.needRightBraces(switchOpFormulaNode.getPrecedence(),rhs.getPrecedence())) {
            this.sb.append('(');
            rhs.accept(this);
            this.sb.append(')');
        }
        else {
            rhs.accept(this);
        }
    }

    @Override
    public String toString() {
        return this.sb.toString();
    }
    
    /**
     * Reset this visitor an allow format another toplevel {@link FormulaNode}.
     */
    public void clear() {
        this.sb.setLength(0);
    }
}
