/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 30.10.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.formula;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Static helper function for our formula evaluator.
 */
public abstract class FormulaHelper {
    
    public static final List<String> BUILTIN_FUNCTIONS =
            Collections.unmodifiableList(
                    Arrays.asList(
                            "abs",
                            "signum",
                            "sin",
                            "cos",
                            "tan",
                            "asin",
                            "acos",
                            "atan",
                            "sqrt",
                            "log",
                            "log10",
                            "exp"));
    
    /**
     * Call a known function in {@link Math} by its name.
     * Known function names are
     * <ul>
     * <li><code>abs</code></li>
     * <li><code>signum</code></li>
     * <li><code>sin</code></li>
     * <li><code>cos</code></li>
     * <li><code>tan</code></li>
     * <li><code>asin</code></li>
     * <li><code>acos</code></li>
     * <li><code>atan</code></li>
     * <li><code>sqrt</code></li>
     * <li><code>log</code></li>
     * <li><code>log10</code></li>
     * <li><code>exp</code></li>
     * </ul>
     * 
     * @param f Then function name.
     * @param x The argument to pass to the function.
     * @return The function applied to <code>x</code>.
     */
    public static double callBuiltin(String f, double x) {
        
        switch(f) {
        case "abs":    return Math.abs(x);
        case "signum": return Math.signum(x);
        case "sqrt":   return Math.sqrt(x);
        case "sin":    return Math.sin(x);
        case "cos":    return Math.cos(x);
        case "tan":    return Math.tan(x);
        case "asin":   return Math.asin(x);
        case "acos":   return Math.acos(x);
        case "atan":   return Math.atan(x);
        case "log":    return Math.log(x);
        case "log10":  return Math.log10(x);
        case "exp":    return Math.exp(x);
        
        default:
            throw new IllegalArgumentException("Unknown builtin function ["+f+"].");
        }
    }
    
    /**
     * Convert a boolean to <code>0.0</code> or <code>1.0</code>.
     * @param b The input boolean.
     * @return <code>0.0</code> for <code>false</code> and <code>1.0</code> for <code>true</code>.
     */
    public static double booleanToDouble(boolean b) {
        return b ? 1.0 : 0.0;
    }

    /**
     * Return true for the precedence of the power operation or the
     * <code>?:</code> switch operation. 
     * @param precedence An operator precedence
     * @return Whether the given operation is right-associative.
     */
    public static boolean isRightAssociative(int precedence) {
        return precedence ==1 || precedence == 8;
    }
    
    /**
     * Return, whether we need braces around a left child expression inside a
     * syntax tree.
     * 
     * @param parentPrecedence The precedence of the parent node.
     * @param childPrecedence The precedence of the left child node.
     * @return Whether the child node must be embraced in string representations.
     */
    public static boolean needLeftBraces(int parentPrecedence, int childPrecedence) {
        
        if (parentPrecedence < 0 || childPrecedence < 0) {
            return false;
        }
        
        if (childPrecedence < parentPrecedence) {
            return true;
        }
        
        if (childPrecedence > parentPrecedence) {
            return false;
        }
        
        return isRightAssociative(parentPrecedence);
    }

    /**
     * Return, whether we need braces around a right child expression inside a
     * syntax tree.
     * 
     * @param parentPrecedence The precedence of the parent node.
     * @param childPrecedence The precedence of the right child node.
     * @return Whether the child node must be embraced in string representations.
     */
    public static boolean needRightBraces(int parentPrecedence, int childPrecedence) {
        
        if (parentPrecedence < 0 || childPrecedence < 0) {
            return false;
        }
        
        if (childPrecedence < parentPrecedence) {
            return true;
        }
        
        if (childPrecedence > parentPrecedence) {
            return false;
        }
        
        return !isRightAssociative(parentPrecedence);
    }
}
