/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: Dec 27, 2006
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.formula;

import java.text.NumberFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.clazzes.util.formula.ast.AndFormulaNode;
import org.clazzes.util.formula.ast.ConstFormulaNode;
import org.clazzes.util.formula.ast.DivideFormulaNode;
import org.clazzes.util.formula.ast.EqualsFormulaNode;
import org.clazzes.util.formula.ast.FormulaNode;
import org.clazzes.util.formula.ast.FunctionFormulaNode;
import org.clazzes.util.formula.ast.GeqFormulaNode;
import org.clazzes.util.formula.ast.GreaterFormulaNode;
import org.clazzes.util.formula.ast.LeqFormulaNode;
import org.clazzes.util.formula.ast.LessFormulaNode;
import org.clazzes.util.formula.ast.MinusFormulaNode;
import org.clazzes.util.formula.ast.MultiplyFormulaNode;
import org.clazzes.util.formula.ast.NegFormulaNode;
import org.clazzes.util.formula.ast.NeqFormulaNode;
import org.clazzes.util.formula.ast.NotFormulaNode;
import org.clazzes.util.formula.ast.OrFormulaNode;
import org.clazzes.util.formula.ast.PlusFormulaNode;
import org.clazzes.util.formula.ast.PowerFormulaNode;
import org.clazzes.util.formula.ast.SwitchFormulaNode;
import org.clazzes.util.formula.ast.SymbolFormulaNode;

/**
 * <p>A minimal formula evaluator , which evaluates formulae in double precision.</p>
 * 
 * This class evaluates mathematical formulae given as strings by a shift-reduce algorithm.
 * It under stands the following operations:
 * <table>
 *     <caption>Examples and Comments on known Operators</caption>
 * <tr><th>Operation</th><th>Examples(Locale.US)</th><th>Comment</th></tr>
 * <tr><td>&lt;literal&gt;</td><td>1,-3,-4.5,1.234e45</td><td>Numeric literals</td></tr>
 * <tr><td>&lt;symbol&gt;</td><td>a,b,$x1,a3,_f5,$a5.x4</td><td>Symbolic placeholder, all characters of JAVA identifiers
 *                                                       are allowed with the additional
 *                                                       feature, that dot characters are allowed.</td></tr>
 * <tr><td>&lt;expr&gt;</td><td>&lt;literal&gt;,&lt;symbol&gt;,-&lt;symbol&gt;,&lt;expr&gt;</td><td>A expr, which is either a literal or
 *                                                                a symbolic placeholder with an optional negative sign prefix.</td></tr>
 * <tr><td>&lt;expr&gt;+&lt;expr&gt;</td><td>a+3,-4.5e56+-23,-$1+45,PI+-E</td><td>Addition.</td></tr>
 * <tr><td>&lt;expr&gt;-&lt;expr&gt;</td><td>a-3,-4.5e56--23,-$1-45,PI--E</td><td>Subtraction.</td></tr>
 * <tr><td>&lt;expr&gt;*&lt;expr&gt;</td><td>a*3,-4.5e56*-23,-$1*45,PI*-E</td><td>Multiplication.</td></tr>
 * <tr><td>&lt;expr&gt;/&lt;expr&gt;</td><td>a/3,-4.5e56/-23,-$1/45,PI/-E</td><td>Division.</td></tr>
 * <tr><td>&lt;expr&gt;^&lt;expr&gt;</td><td>a^3,-4.5e56^-23,-$1^45,PI^-E</td><td>Exponentiation.</td></tr>
 * <tr><td>&lt;expr&gt;&lt;&lt;expr&gt;</td><td>a&lt;3.24,-4&lt;-3</td><td>Less than, returns 0 or 1.</td></tr>
 * <tr><td>&lt;expr&gt;&lt;=&lt;expr&gt;</td><td>a&lt;=3.24,-4&lt;=-3</td><td>Less than or equals, returns 0 or 1.</td></tr>
 * <tr><td>&lt;expr&gt;&gt;&lt;expr&gt;</td><td>a&gt;3.24,-3&gt;-4</td><td>Greater than, returns 0 or 1.</td></tr>
 * <tr><td>&lt;expr&gt;&gt;=&lt;expr&gt;</td><td>a&gt;=3.24,-3&gt;=-4</td><td>Greater than or equals, returns 0 or 1.</td></tr>
 * <tr><td>&lt;expr&gt;==&lt;expr&gt;</td><td>a==3.24,-4==-4</td><td>Test for bitwise equality, returns 0 or 1.</td></tr>
 * <tr><td>&lt;expr&gt;!=&lt;expr&gt;</td><td>a!=3.24,-3&ne;-4</td><td>Test for bitwise inequality, returns 0 or 1.</td></tr>
 * <tr><td>&lt;expr&gt;&amp;&amp;&lt;expr&gt;</td><td>a&amp;&amp;b,0&amp;&amp;1</td><td>Logical and, returns 1, if both sides are non-zero, 0 otherwise.</td></tr>
 * <tr><td>&lt;expr&gt;||&lt;expr&gt;</td><td>a||b,0||1</td><td>Logical or, returns 1, if either side is non-zero, 0 otherwise.</td></tr>
 * <tr><td>!&lt;expr&gt;</td><td>!a,!a&amp;&amp;b</td><td>Negation, non-zero input is converted to zero, zero to 1.</td></tr>
 * <tr><td>(&lt;expr&gt;)</td><td>(a+1)*(-3.4e56+b)</td><td>Grouping of expressions.</td></tr>
 * <tr><td>exp(&lt;expr&gt;)</td><td>exp(a*3)</td><td>Exponential function.</td></tr>
 * <tr><td>ln(&lt;expr&gt;)</td><td>ln(e)</td><td>Natural Logarithm.</td></tr>
 * <tr><td>log(&lt;expr&gt;)</td><td>log(100*a)</td><td>Logarithm to the base 10.</td></tr>
 * <tr><td>sin(&lt;expr&gt;)</td><td>sin(3*PI)</td><td>Sine function.</td></tr>
 * <tr><td>cos(&lt;expr&gt;)</td><td>cos(0.5*PI)</td><td>Cosine function.</td></tr>
 * <tr><td>tan(&lt;expr&gt;)</td><td>tan(0.25*PI)</td><td>Tangent function.</td></tr>
 * <tr><td>asin(&lt;expr&gt;)</td><td>asin(0.5)</td><td>Inverse Sine function.</td></tr>
 * <tr><td>acos(&lt;expr&gt;)</td><td>acos(3*alpha)</td><td>Inverse Cosine function.</td></tr>
 * <tr><td>atan(&lt;expr&gt;)</td><td>atan(0.5*PI)</td><td>Inverse Tangent function.</td></tr>
 * <tr><td>abs(&lt;expr&gt;)</td><td>abs(-0.5*PI)</td><td>Absolute value.</td></tr>
 * <tr><td>sgn(&lt;expr&gt;)</td><td>sgn(-1.1)</td><td>Sign of expression, returns -1,0 or 1</td></tr>
 * <tr><td>&lt;expr&gt;?&lt;expr&gt;:&lt;expr&gt;</td><td>a&lt;b ? a*3: b*5</td><td>Ternary operator, return first option, if first expression is non-zero.</td></tr>
 * </table>
 *
 * The operators obey the following order of precedence (higher numbers take precedence over lower numbers):
 * <table>
 *     <caption>Detailed precedence, mostly known from math</caption>
 * <tr><td>1</td><td>?:</td></tr>
 * <tr><td>2</td><td>||</td></tr>
 * <tr><td>3</td><td>&amp;&amp;</td></tr>
 * <tr><td>4</td><td>==,&ne;</td></tr>
 * <tr><td>5</td><td>&lt;,&lt;=,&gt;,&gt;=</td></tr>
 * <tr><td>6</td><td>+,-</td></tr>
 * <tr><td>7</td><td>*,/</td></tr>
 * <tr><td>8</td><td>^</td></tr>
 * <tr><td>9</td><td>!</td></tr>
 * </table>
 *
 * <p>Please note, that this engine does not know about any predefined constants like <code>PI</code>
 * or <code>E</code>. If you need such constants, please supply an appropriate implementation of
 * {@link SymbolValues} to {@link #evaluate(Locale, String, SymbolValues)} and
 * {@link #check(Locale, String)}.</p>
 * 
 * The parser is locale-aware and does not use grouping for number constants.
 * Thus, the following expressions are valid:
 * <pre>
 *  1 == FormulaEvaluator.evaluate(Locale.US,"2.345-1.345",null);
 *  1 == FormulaEvaluator.evaluate(Locale.GERMAN,"2,345-1,345",null);
 * </pre>
 */
public class FormulaEvaluator {

    private static final Map<String,Character> LEGACY_BUILTIN_FUNCTIONS;
    private static final Map<String,Character> BUILTIN_FUNCTIONS;
    private static final Map<Character,String> REV_BUILTIN_FUNCTIONS;
	
	private static final char NE = '\u2260';
	private static final char LE = '\u2264';
	private static final char GE = '\u2265';
    
	static {
		
	    LEGACY_BUILTIN_FUNCTIONS = new HashMap<String, Character>();
	    LEGACY_BUILTIN_FUNCTIONS.put("ln",Character.valueOf('l'));
        LEGACY_BUILTIN_FUNCTIONS.put("log",Character.valueOf('L'));
        LEGACY_BUILTIN_FUNCTIONS.put("sgn",Character.valueOf('A'));
	    
		BUILTIN_FUNCTIONS = new HashMap<String, Character>();
		BUILTIN_FUNCTIONS.put("sqrt",Character.valueOf('2'));
		BUILTIN_FUNCTIONS.put("exp",Character.valueOf('e'));
		BUILTIN_FUNCTIONS.put("log",Character.valueOf('l'));
		BUILTIN_FUNCTIONS.put("log10",Character.valueOf('L'));
        BUILTIN_FUNCTIONS.put("sin",Character.valueOf('s'));
		BUILTIN_FUNCTIONS.put("cos",Character.valueOf('c'));
		BUILTIN_FUNCTIONS.put("tan",Character.valueOf('t'));
		BUILTIN_FUNCTIONS.put("asin",Character.valueOf('S'));
		BUILTIN_FUNCTIONS.put("acos",Character.valueOf('C'));
		BUILTIN_FUNCTIONS.put("atan",Character.valueOf('T'));
        BUILTIN_FUNCTIONS.put("abs",Character.valueOf('a'));
        BUILTIN_FUNCTIONS.put("signum",Character.valueOf('A'));

        REV_BUILTIN_FUNCTIONS = new HashMap<Character,String>();
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('2'),"sqrt");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('e'),"exp");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('l'),"log");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('L'),"log10");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('s'),"sin");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('c'),"cos");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('t'),"tan");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('S'),"asin");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('C'),"acos");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('T'),"atan");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('a'),"abs");
        REV_BUILTIN_FUNCTIONS.put(Character.valueOf('A'),"signum");
	}
	
	private static final class Token {
		
		// operator '+', '-', '*', '/', '=', ' or '(' or ')' or '\0' (end-of-string marker).
		// ' ' means number
		public final char operator;
		// precedence:
		// -1 number
		// 0  terminator (')' or EOS)
        // 1  '?:' 
        // 2  '||' 
        // 3  '&&' 
        // 4  '==','!=" 
        // 5  '<','<=','>=','>' 
		// 6  '+','-' 
		// 7  '*','/' 
		// 8  '^' 
        // 9  '!' (logical not) 
		// 15 '(', builtin functions
		public final int precedence;
		// numeric value, if precedence == -1
		// for groups and builtin function, value is set to 1.0 or -1.0
		// in order to reflect a minus sign, which immediately precedes
		// the builtin function or group like in "-(a+b)" or "a*-sin(x)"
		public double value;
		
		public Token(char operator, int precedence) {
			super();
			this.operator = operator;
			this.precedence = precedence;
			this.value = 1.0;
		}
		
		public Token(char operator, double sign) {
			super();
			this.operator = operator;
			this.precedence = 0xf;
			this.value = sign;
		}
		
		public Token(double value) {
			this.operator=' ';
			this.precedence = -1;
			this.value = value;
		}
	
		public String toString() {
			if (this.operator == '\0') return "EOS";
			if (this.precedence < 0) return String.valueOf(this.value);
			return String.valueOf(this.operator);
		}
	}
	
	private final String expr;
	private final List<Token> stack;
	private final Map<Token,FormulaNode> nodeStack;
    private final NumberFormat numberFormat;
	private final SymbolValues symValues;
	private final int options;
	
	private FormulaEvaluator(Locale locale, String expr, SymbolValues symValues, int options, boolean nodesEnabled) {
		this.expr = expr;
		this.stack = new ArrayList<FormulaEvaluator.Token>(16);
		this.numberFormat = NumberFormat.getInstance(locale);
		this.numberFormat.setGroupingUsed(false);
		this.symValues = symValues;
		this.options = options;
		this.nodeStack = nodesEnabled ? new HashMap<Token,FormulaNode>(16) : null;
	}
	
	private final boolean isIdentifierPart(char c) {
	    
	    if (Character.isJavaIdentifierPart(c)) {
	        return true;
	    }
	    
	    if ((this.options & FormulaOptions.DOT_IDENTIFIERS) != 0 && c =='.') {
	        return true;
	    }
	    
	    if ((this.options & FormulaOptions.BRACKET_IDENTIFIERS) != 0 && (c =='[' || c ==']')) {
            return true;
        }
       
	    return false;
	}
	
	private boolean pushToken(ParsePosition pos) throws ParseException {
		
		pos.setErrorIndex(-1);
		
		while (pos.getIndex() < this.expr.length() &&
		        Character.isWhitespace(this.expr.charAt(pos.getIndex())))
			pos.setIndex(pos.getIndex()+1);
		
		if (pos.getIndex() >= this.expr.length()) {
			
			this.stack.add(new Token('\0',0));
			return false;
		}
		
		int oidx = pos.getIndex();
		
		char c = this.expr.charAt(oidx);
		Token token = null;
		
		double sign = 1.0;
		
		switch(c) {
		
		case '^':
			token = new Token(c,8);
			break;
		case '*':
		case '/':
			token = new Token(c,7);
			break;
		case '-':
			// this is a sign after an operator or at the start of an expression.
			if (this.stack.size() <= 0 || this.stack.get(this.stack.size()-1).precedence > 0) {
				
				++oidx;
				while (oidx < this.expr.length() && Character.isWhitespace(this.expr.charAt(oidx)))
					++oidx;
				
				if (oidx >= this.expr.length())
					throw new ParseException("Unexpected end of formula ["+this.expr+"] after minus sign.",oidx);
				
				sign = -1.0;
				c = this.expr.charAt(oidx);

				if (c=='-')
					throw new ParseException("Formula ["+this.expr+"] contains superfluous minus signs.",oidx);
				
				pos.setIndex(oidx);
				
				if (c=='(')
					token = new Token(c,sign);
				
				break;
			}
			//$FALL-THROUGH$
		case '+':
			token = new Token(c,6);
			break;
		case '<':
		case '>':
		    
		    if (oidx+1 < this.expr.length() && this.expr.charAt(oidx+1) == '=') {
		        
		        ++oidx;
		        token = new Token(c == '<' ? LE : GE,5);
		    }
		    else {
		        token = new Token(c,5);
		    }
		    break;
        
		case LE:
        case GE:
            token = new Token(c,5);
            break;
            
        case '=':
        case '!':
            if (oidx+1 < this.expr.length() && this.expr.charAt(oidx+1) == '=') {        
                ++oidx;
                token = new Token(c == '=' ? '=' : NE,4);
            }
            else if (c=='!') {
                token = new Token(c,9);
            }
            break;

        case NE:
            token = new Token(c,4);
            break;
            
        case '&':
            if (oidx+1 < this.expr.length() && this.expr.charAt(oidx+1) == '&') {        
                ++oidx;
                token = new Token('&',3);
            }
            break;

        case '|':
            if (oidx+1 < this.expr.length() && this.expr.charAt(oidx+1) == '|') {        
                ++oidx;
                token = new Token('|',2);
            }
            break;

        case '?':
        case ':':
                token = new Token(c,1);
            break;

        case ')':
			token = new Token(c,0);
			break;
		case '(':
			token = new Token(c,0xf);
			break;
		}

		if (token == null) {
			
			if (Character.isJavaIdentifierStart(c)) {
				
				int idx = oidx+1;
				
				while (idx < this.expr.length() && this.isIdentifierPart(this.expr.charAt(idx)))
					++idx;
				
				pos.setIndex(idx);
				
				String identifier = this.expr.substring(oidx,idx);

				if ((this.options & FormulaOptions.BRACKET_IDENTIFIERS) != 0) {
				    
				    int fio = identifier.indexOf('[');
				    int lio = identifier.lastIndexOf('[');
                    int fic = identifier.indexOf(']');
                    int lic = identifier.lastIndexOf(']');
                    
                    if (fio != lio) {
                        throw new ParseException("Identifier ["+identifier+"] contains two open brackets.",oidx+fio);  
                    }
                    if (fic != lic) {
                        throw new ParseException("Identifier ["+identifier+"] contains two closing brackets.",oidx+fic);  
                    }
                    if (fio != -1 && fio >= fic) {
                        throw new ParseException("Identifier ["+identifier+"] contains misaligned brackets.",oidx+fic);  
                    }
				}
				
				Character builtin = null;
				
				if ((this.options & FormulaOptions.LEGACY_BUILTINS) != 0) {
				    builtin = LEGACY_BUILTIN_FUNCTIONS.get(identifier);
				}
				
				if (builtin == null) {
				    builtin = BUILTIN_FUNCTIONS.get(identifier);
				}
				
				if (builtin == null) {
				
					Number v =
					    this.symValues == null ?
					            null : this.symValues.getSymbolValue(identifier);
					
					if (v == null)
						throw new ParseException("Unknown symbol ["+identifier+"] specified.",oidx);
				
					if (this.nodeStack != null) {
					}
					token = new Token(sign * v.doubleValue());
					
					if (this.nodeStack != null) {
                        FormulaNode node = new SymbolFormulaNode(identifier);
                        if (sign < 0.0) {
                            node = new NegFormulaNode(node);
                        }
					    this.nodeStack.put(token,node);
					}
				}
				else {
					
					while (idx < this.expr.length() && Character.isWhitespace(this.expr.charAt(idx)))
						++idx;

					if (this.expr.charAt(idx) != '(')
						throw new ParseException("Builtin function ["+identifier+"] not followed by an open brace.",idx);
					
					++idx;
					pos.setIndex(idx);
					
					token = new Token(builtin.charValue(),sign);
				}
			}
			else {
			    
				Number n = this.numberFormat.parse(this.expr, pos);
				
				if (pos.getIndex() > oidx) {
				    // work around nasty problems in NumerFormat with
	                // xxx.yyye+16
	                double v = n.doubleValue();
				    
	                if (pos.getIndex() < this.expr.length()) {
	                    
	                    char followup = this.expr.charAt(pos.getIndex());
	                    
	                    if (followup == 'e' || followup == 'E') {
	                        
	                        String mantissa = this.expr.substring(oidx,pos.getIndex());
	                        
	                        if (mantissa.indexOf('e') < 0 &&  mantissa.indexOf('E') < 0) {
	                            
	                            if (pos.getIndex()+1 < this.expr.length() && this.expr.charAt(pos.getIndex()+1) == '+') {
	                                
	                                int sidx = pos.getIndex()+2;
	                                pos.setIndex(sidx);
	                                this.numberFormat.setParseIntegerOnly(true);
	                                Number exp = this.numberFormat.parse(this.expr,pos);
	                                this.numberFormat.setParseIntegerOnly(false);
                                    
	                                if (pos.getIndex() > sidx) {
	                                    v *= Math.pow(10.0,exp.doubleValue());
	                                }
	                            }
	                        }
	                    }
	                }
				    
				    token = new Token(sign * v);
				    
				    if (this.nodeStack != null) {
                        this.nodeStack.put(token,new ConstFormulaNode(sign * n.doubleValue()));
                    }
				}
			}
		}
		else {
			pos.setIndex(oidx+1);
		}
		
		if (token == null)
			throw new ParseException("Unknown operator ["+c+"]",oidx);	

		this.stack.add(token);	
		
		return true;
	}
	
	private boolean reduce() {
		
		int l = this.stack.size() - 1;
		
		if (l<2) return false;
		
		Token last = this.stack.get(l);
		Token last1 = this.stack.get(l-1);
		Token last2 = this.stack.get(l-2);
		
		if (last2.precedence == 9 &&
		        last1.precedence < 0 &&
		        last.precedence < 9) {
		    
		    // unary logical not.
		    last1.value = FormulaHelper.booleanToDouble(last1.value == 0.0);
		    this.stack.remove(l-2);
		    return true;
		}
		
		if (l<3) return false;
        
		Token last3 = this.stack.get(l-3);
        
		// number ? number : number
		if (l>=5 &&
		        last2.precedence == 1 && last2.operator == ':' &&
		        // only reduce, if not a higher priority operation is in queue
		        (last.precedence < 1  || last.operator == ':') &&
		        last3.precedence < 0 &&
                last1.precedence < 0) {
		    
		    Token last4 = this.stack.get(l-4);
		    Token last5 = this.stack.get(l-5);
	        
		    // last5 ? last3 : last1  
		    if (last4.precedence == 1 && last4.operator=='?' &&
		            last5.precedence < 0) {
		        
		        last5.value = last5.value != 0.0 ? last3.value : last1.value;
		        this.stack.remove(l-1);
                this.stack.remove(l-2);
                this.stack.remove(l-3);
                this.stack.remove(l-4);
                return true;
		    }
		}

		if (last3.precedence == 0xf && last2.precedence < 0 && last1.operator == ')') {
		
			// take into account sign before brace...
			switch (last3.operator) {
			
			case '2':
				last2.value = last3.value * Math.sqrt(last2.value);
				break;
			case 'e':
				last2.value = last3.value * Math.exp(last2.value);
				break;
			case 'l':
				last2.value = last3.value * Math.log(last2.value);
				break;
            case 'L':
                last2.value = last3.value * Math.log10(last2.value);
                break;
			case 's':
				last2.value = last3.value * Math.sin(last2.value);
				break;
			case 'c':
				last2.value = last3.value * Math.cos(last2.value);
				break;
			case 't':
				last2.value = last3.value * Math.tan(last2.value);
				break;
			case 'S':
				last2.value = last3.value * Math.asin(last2.value);
				break;
			case 'C':
				last2.value = last3.value * Math.acos(last2.value);
				break;
			case 'T':
				last2.value = last3.value * Math.atan(last2.value);
				break;
            case 'a':
                last2.value = last3.value * Math.abs(last2.value);
                break;
            case 'A':
                last2.value = last3.value * Math.signum(last2.value);
                break;
			default:
			    // grouping by brackets
				last2.value *= last3.value;
			}
			this.stack.set(l-3,last2);
			this.stack.set(l-2,this.stack.remove(l));
			this.stack.remove(l-1);
			return true;
		}

		if (last3.precedence < 0 && last1.precedence < 0) {
			
			if (last2.precedence < last.precedence) return false;
			// exponentiation is right-associated.
			if (last.precedence == 8 && last2.precedence == 8) return false;
			
			switch (last2.operator) {
			case '+':
				last3.value += last1.value;
				break;
			case '-':
				last3.value -= last1.value;
				break;
				
			case '*':
				last3.value *= last1.value;
				break;
			case '/':
				last3.value /= last1.value;
				break;
				
			case '^':
				last3.value = Math.pow(last3.value,last1.value);
				break;
				
            case '<':
                last3.value = FormulaHelper.booleanToDouble(last3.value < last1.value);
                break;
            case '>':
                last3.value = FormulaHelper.booleanToDouble(last3.value > last1.value);
                break;
            case LE:
                last3.value = FormulaHelper.booleanToDouble(last3.value <= last1.value);
                break;
            case GE:
                last3.value = FormulaHelper.booleanToDouble(last3.value >= last1.value);
                break;
            case '=':
                last3.value = FormulaHelper.booleanToDouble(last3.value == last1.value);
                break;
            case NE:
                last3.value = FormulaHelper.booleanToDouble(last3.value != last1.value);
                break;
            case '&':
                last3.value = FormulaHelper.booleanToDouble(last3.value != 0.0 && last1.value != 0.0);
                break;
            case '|':
                last3.value = FormulaHelper.booleanToDouble(last3.value != 0.0 || last1.value != 0.0);
                break;

			default:
				return false;
			}
		
			this.stack.set(l-2,this.stack.remove(l));
			this.stack.remove(l-1);
			return true;
		}

		return false;
	}
	
	private double getValue() throws ParseException {
		if (this.stack.size() != 2 || this.stack.get(0).precedence != -1 || this.stack.get(1).operator != '\0')
			throw new ParseException("Cannot reduce formula ["+this.expr+"].",this.expr.length()-1);
		
		return this.stack.get(0).value;
	}
	
    private boolean reduceNode() {
        
        int l = this.stack.size() - 1;
        
        if (l<2) return false;
        
        Token last = this.stack.get(l);
        Token last1 = this.stack.get(l-1);
        Token last2 = this.stack.get(l-2);
        
        if (last2.precedence == 9 &&
                last1.precedence < 0 &&
                last.precedence < 9) {
            
            // unary logical not.
            this.nodeStack.put(last1,new NotFormulaNode(this.nodeStack.get(last1)));
            this.stack.remove(l-2);
            return true;
        }
        
        if (l<3) return false;
        
        Token last3 = this.stack.get(l-3);
        
        // number ? number : number
        if (l>=5 &&
                last2.precedence == 1 && last2.operator == ':' &&
                // only reduce, if not a higher priority operation is in queue
                (last.precedence < 1  || last.operator == ':') &&
                last3.precedence < 0 &&
                last1.precedence < 0) {
            
            Token last4 = this.stack.get(l-4);
            Token last5 = this.stack.get(l-5);
            
            // last5 ? last3 : last1  
            if (last4.precedence == 1 && last4.operator=='?' &&
                    last5.precedence < 0) {
                
                this.nodeStack.put(last5,new SwitchFormulaNode(this.nodeStack.get(last5),this.nodeStack.get(last3),this.nodeStack.get(last1)));

                this.stack.remove(l-1);
                this.stack.remove(l-2);
                this.stack.remove(l-3);
                this.stack.remove(l-4);
                return true;
            }
        }

        if (last3.precedence == 0xf && last2.precedence < 0 && last1.operator == ')') {
        
            // take into account sign before brace...
            String f = REV_BUILTIN_FUNCTIONS.get(last3.operator);
            
            if (f!=null) {
                
                FormulaNode node = new FunctionFormulaNode(f,this.nodeStack.get(last2));
                
                if (last3.value < 0.0) {
                    node = new NegFormulaNode(node);
                }
                
                this.nodeStack.put(last2,node);
            }
            else {
                // bracing with unary negation
                if (last3.value < 0.0) {
                    this.nodeStack.put(last2,new NegFormulaNode(this.nodeStack.get(last2)));
                }
            }

            this.stack.set(l-3,last2);
            this.stack.set(l-2,this.stack.remove(l));
            this.stack.remove(l-1);
            return true;
        }

        if (last3.precedence < 0 && last1.precedence < 0) {
            
            if (last2.precedence < last.precedence) return false;
            // exponentiation is right-associated.
            if (last.precedence == 8 && last2.precedence == 8) return false;
            
            switch (last2.operator) {
            case '+':
                this.nodeStack.put(last3,new PlusFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case '-':
                this.nodeStack.put(last3,new MinusFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
                
            case '*':
                this.nodeStack.put(last3,new MultiplyFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case '/':
                this.nodeStack.put(last3,new DivideFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
                
            case '^':
                this.nodeStack.put(last3,new PowerFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
                
            case '<':
                this.nodeStack.put(last3,new LessFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case '>':
                this.nodeStack.put(last3,new GreaterFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case LE:
                this.nodeStack.put(last3,new LeqFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case GE:
                this.nodeStack.put(last3,new GeqFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case '=':
                this.nodeStack.put(last3,new EqualsFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case NE:
                this.nodeStack.put(last3,new NeqFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case '&':
                this.nodeStack.put(last3,new AndFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;
            case '|':
                this.nodeStack.put(last3,new OrFormulaNode(this.nodeStack.get(last3),this.nodeStack.get(last1)));
                break;

            default:
                return false;
            }
        
            this.stack.set(l-2,this.stack.remove(l));
            this.stack.remove(l-1);
            return true;
        }

        return false;
    }
    
    private FormulaNode getNodeValue() throws ParseException {
        if (this.stack.size() != 2 || this.stack.get(0).precedence != -1 || this.stack.get(1).operator != '\0')
            throw new ParseException("Cannot reduce formula ["+this.expr+"].",this.expr.length()-1);
        
        return this.nodeStack.get(this.stack.get(0));
    }

	/**
	 * <p>Evaluate the given expression an return it's double precision value using
	 * {@link FormulaOptions#LEGACY_BUILTINS}|{@link FormulaOptions#DOT_IDENTIFIERS} as options.</p>
	 * 
	 * @param locale A locale used for parsing numeric literals.
	 * @param expr The expression as described in {@link FormulaEvaluator}.
	 * @param symValues A supplier of values for symbolic placeholders in the formula or
	 *                  <code>null</code>, if symbolic placeholder should be disabled.
	 * @return A double precision value of the given mathematical expression.
	 * @throws ParseException Upon parse errors or when the formula contains a placeholder,
	 *                        for which <code>symValues</code> return a <code>null</code> value.
	 */
	public static double evaluate(Locale locale, String expr, SymbolValues symValues) throws ParseException {
	    return evaluate(locale,expr,symValues,FormulaOptions.LEGACY_BUILTINS|FormulaOptions.DOT_IDENTIFIERS);
	}
	
	/**
	 * <p>Evaluate the given expression an return it's double precision value.</p>
	 * 
	 * @param locale A locale used for parsing numeric literals.
	 * @param expr The expression as described in {@link FormulaEvaluator}.
	 * @param symValues A supplier of values for symbolic placeholders in the formula or
	 *                  <code>null</code>, if symbolic placeholder should be disabled.
     * @param options Options a bit mask composed of options in {@link FormulaOptions}.
	 * @return A double precision value of the given mathematical expression.
	 * @throws ParseException Upon parse errors or when the formula contains a placeholder,
	 *                        for which <code>symValues</code> return a <code>null</code> value.
	 */
	public static double evaluate(Locale locale, String expr, SymbolValues symValues, int options) throws ParseException {
	    FormulaEvaluator e = new FormulaEvaluator(locale,expr,symValues,options,false);
		
		ParsePosition pos = new ParsePosition(0);
		
		while (e.pushToken(pos)) {
			
			while (e.reduce()) ;
		}
		
		while (e.reduce()) ;
		
		return e.getValue();
	}
	
	private static final class RecordingSymbolValues implements SymbolValues {

	    private final Set<String> symbols;
	    
	    public RecordingSymbolValues() {
	        this.symbols = new HashSet<String>();
	    }
	    
        public RecordingSymbolValues(Set<String> symbols) {
            this.symbols = symbols;
        }

        public Number getSymbolValue(String sym) {
            
            this.symbols.add(sym);
            return Double.NaN;
        }
	    
        public Set<String> getSymbols() {
            return this.symbols;
        }
	}
	
    /**
     * <p>Syntactically check the given expression an return the  set of
     * names of symbolic placeholders in the given expression using
     * {@link FormulaOptions#LEGACY_BUILTINS}|{@link FormulaOptions#DOT_IDENTIFIERS}
     * as options.</p>
     * 
     * <p>This function will e.g. return the set <code>["one","PI"]</code>
     * for the expression <code>sin(one/2*PI)</code>.
     * </p>
     * 
     * @param locale A locale used for parsing numeric literals.
     * @param expr The expression as described in {@link FormulaEvaluator}.
     * @return All placeholder names inthe given expression.
     * @throws ParseException Upon parse errors.
     */
    public static Set<String> check(Locale locale, String expr) throws ParseException {
        return check(locale,expr,FormulaOptions.LEGACY_BUILTINS|FormulaOptions.DOT_IDENTIFIERS);
    }
    /**
     * <p>Syntactically check the given expression an return the  set of
     * names of symbolic placeholders in the given expression.</p>
     * 
     * <p>This function will e.g. return the set <code>["one","PI"]</code>
     * for the expression <code>sin(one/2*PI)</code>.
     * </p>
     * 
     * @param locale A locale used for parsing numeric literals.
     * @param expr The expression as described in {@link FormulaEvaluator}.
     * @param options Options a bit mask composed of options in {@link FormulaOptions}.
     * @return All placeholder names inthe given expression.
     * @throws ParseException Upon parse errors.
     */
    public static Set<String> check(Locale locale, String expr, int options) throws ParseException {
        
        RecordingSymbolValues r = new RecordingSymbolValues();
        
        FormulaEvaluator e = new FormulaEvaluator(locale,expr,r,options,false);
        
        ParsePosition pos = new ParsePosition(0);
        
        while (e.pushToken(pos)) {
            
            while (e.reduce()) ;
        }
        
        while (e.reduce()) ;
        
        e.getValue();
        
        return r.getSymbols();
    }

    /**
     * <p>Parse the given expression an return the abstract syntax tree of the formula.
     * The provided optional set will be filled with the known symbolic placeholders
     * in the given expression using
     * {@link FormulaOptions#LEGACY_BUILTINS}|{@link FormulaOptions#DOT_IDENTIFIERS}
     * as options..</p>
     * 
     * <p>This function will e.g. return the set <code>["one","PI"]</code>
     * for the expression <code>sin(one/2*PI)</code>.
     * </p>
     * 
     * @param locale A locale used for parsing numeric literals.
     * @param expr The expression as described in {@link FormulaEvaluator}.
     * @param symbols A set filled with identifiers in the formula, if not <code>null</code>.
     * @return A formula node representing the AST.
     * @throws ParseException Upon parse errors.
     */
    public static FormulaNode parse(Locale locale, String expr, Set<String> symbols) throws ParseException {
        return parse(locale,expr,symbols,FormulaOptions.LEGACY_BUILTINS|FormulaOptions.DOT_IDENTIFIERS);
    }
    
    /**
     * <p>Parse the given expression an return the abstract syntax tree of the formula.
     * The provided optional set will be filled with the known symbolic placeholders
     * in the given expression.</p>
     * 
     * <p>This function will e.g. return the set <code>["one","PI"]</code>
     * for the expression <code>sin(one/2*PI)</code>.
     * </p>
     * 
     * @param locale A locale used for parsing numeric literals.
     * @param expr The expression as described in {@link FormulaEvaluator}.
     * @param symbols A set filled with identifiers in the formula, if not <code>null</code>.
     * @param options Options a bit mask composed of options in {@link FormulaOptions}.
     * @return A formula node representing the AST.
     * @throws ParseException Upon parse errors.
     */
    public static FormulaNode parse(Locale locale, String expr, Set<String> symbols, int options) throws ParseException {
        
        RecordingSymbolValues r = symbols == null ?
                new RecordingSymbolValues() : new RecordingSymbolValues(symbols);
        
        FormulaEvaluator e = new FormulaEvaluator(locale,expr,r,options,true);
        
        ParsePosition pos = new ParsePosition(0);
        
        while (e.pushToken(pos)) {
            
            while (e.reduceNode()) ;
        }
        
        while (e.reduceNode()) ;
        
        return e.getNodeValue();
    }
}
