/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.datetime;

import java.io.Serializable;
import java.text.ParseException;
import java.util.Calendar;
import java.util.TimeZone;

import javax.persistence.Embeddable;
import javax.persistence.Transient;

/**
 * A class to store UTC timestamps and timezone offset to database.
 * 
 * @author rbreuss
 * 
 */
@Embeddable
public class UtcTimestamp implements Serializable, Comparable<UtcTimestamp>, Cloneable {

    /**
     * To be changed at class-modification.
     */
    private static final long serialVersionUID = -4171006885461716579L;
    
    public static final long MILLISECONDS_PER_SECOND = 1000L;
    public static final long MILLISECONDS_PER_MINUTE = 60 * MILLISECONDS_PER_SECOND;
    public static final long MILLISECONDS_PER_HOUR   = 60 * MILLISECONDS_PER_MINUTE;
    public static final long MILLISECONDS_PER_DAY    = 24 * MILLISECONDS_PER_HOUR;
    
    private static final long REF_DAY = 2440588L;
    private static final long REF_TIMESTAMP = REF_DAY *  MILLISECONDS_PER_DAY;
    
    private static final int DAYS_IN_MONTH[] =
        new int [] { 29,31,28,31,30,31,30,31,31,30,31,30,31 };
    
    /**
     * This class is designed to handle UTC-Timestamps, so this field must not
     * be changed. It is used internally to calculate appropreate Calendar
     * Objects.
     */
    private static final TimeZone utcTimezone = TimeZone.getTimeZone("UTC");

    private static class DateFields {
        public final int year;
        public final int month;
        public final int day;
        
        DateFields(int year, int month, int day) {
            this.year = year;
            this.month = month;
            this.day = day;
        }
    };
    
    /**
     * The amount of milliseconds elapsed since 1970-01-01.
     */
    private Long utcMillis;

    /**
     * The timezone offset in minutes.
     */
    private Integer tzOffset;

    /**
     * A cache for decomposed year/month/day
     */
    private transient DateFields dateFields;
    
    @Transient
    private long getLocalTimestamp() {
        if(this.utcMillis==null) return 0L;
        
        long ts = this.utcMillis;
        
        if (this.tzOffset != null) ts += this.tzOffset * MILLISECONDS_PER_MINUTE;
        return ts + REF_TIMESTAMP;
    }
    
    private void initDateFields() {
        
        if (this.utcMillis == null) {
            this.dateFields = new DateFields(0,0,0);
            return;
        }
        
        long linear_day = this.getLocalTimestamp();
        
        linear_day /= MILLISECONDS_PER_DAY;
        
        if (linear_day < 2363522L /* 1759-01-01 */ ||
                linear_day > 5373484L /* 9999-12-31 */   )
        {
            this.dateFields = new DateFields(0,0,0);
            return;
        }

        long x;
        long j = linear_day - 1721119L;
        long y = (j*4 - 1)/146097;
        j = j*4 - 146097*y - 1;
        x = j/4;
        j = (x*4 + 3) / 1461;
        y = 100*y + j;
        x = (x*4) + 3 - 1461*j;
        x = (x + 4)/4;
        long m = (5*x - 3)/153;
        x = 5*x - 3 - 153*m;
        long d = (x + 5)/5;

        if ( m < 10 ) {
            m += 3;
        } else {
            m -= 9;
            ++y;
        }
         
        this.dateFields = new DateFields((int)y,(int)m,(int)d);
    }
    
    private static long makeUtcMillis (int year, int month, int day) {
        
        long c, ya;
        long y = year;
        long m = month;
        if ( m > 2 ) {
          m -= 3;
        } else {
          m += 9;
          y--;
        }
        c = y;
        c /= 100;
        ya = y - 100*c;
        long linear_day = 1721119 + day + (146097*c)/4 + (1461*ya)/4 + (153*m+2)/5;
        linear_day -= REF_DAY;
        return linear_day * MILLISECONDS_PER_DAY;
    }
    
    /**
     * @param y The year in the range from 1759 to 9999.
     * @return Whether the given year is a leap year in the gregorian calendar.
     */
    public static boolean isLeapYear (int y)
    {
      return (y % 4) == 0 && ((y % 400) == 0 || (y % 100) != 0); 
    }

    /**
     * @param y The year to consider.
     * @param m The month to consider.
     * @return The number of days in the given month.
     */
    public static int getDaysOfMonth (int y, int m)
    {
      if (m < 1 || m > 12) return 0;

      if (m == 2 && isLeapYear(y)) m = 0;

      return DAYS_IN_MONTH[m];
    }
    
    /**
     * Construct a new UtcTimestamp for the current time in the
     * utc timezone (tzOffset=0).
     * 
     * If you need the current time in your default timezone, consider using
     * {@link #now()}. 
     * 
     * @see #now()
     */
    public UtcTimestamp() {
        this(System.currentTimeMillis(), 0);
    }
    
    /**
     * Copy constructor.
     * 
     * @param a The timestamp to be copied.
     */
    public UtcTimestamp(UtcTimestamp a) {
        if (a == null)
        {
            this.tzOffset = null;
            this.utcMillis = null;
        }
        else
        {
            this.tzOffset = a.tzOffset;
            this.utcMillis = a.utcMillis;
        }
    }
    
    /** Construct a new UtcTimestamp for the passed Calendar object.
     * @param c
     */
    public UtcTimestamp(Calendar c) {
        super();
        this.setFromCalendar(c);
    }
    
    /**
     * Construct a new UtcTimestamp for the passed milliseconds with a tzOffset=0.
     * @param millis
     */
    public UtcTimestamp(long millis) {
        this(millis, 0);
    }

    /**
     * Construct a new UtcTimestamp for the passed milliseconds with a
     * tzOffset for the given timezone at the given instance in time.
     * 
     * @param timeZone The Timezone for which to determine the timezone offset.
     * @param millis The milliseconds since 1970-01-01 00:00:00 GMT.
     */
    public UtcTimestamp(TimeZone timeZone, long millis) {
        this(millis, timeZone.getOffset(millis)/60000);
    }

    /**
     * Construct a new UtcTimestamp for the current time with a
     * tzOffset for the given timezone at the current time.
     * 
     * @param timeZone The Timezone for which to determine the timezone offset.
     */
    public UtcTimestamp(TimeZone timeZone) {
        this(timeZone,System.currentTimeMillis());
    }

    /**
     * Construct a new UtcTimestamp for the passed milliseconds an timezone offset.
     * @param millis
     * @param tzOffset in minutes
     */
    public UtcTimestamp(long millis, int tzOffset) {
        this.utcMillis = Long.valueOf(millis);
        this.tzOffset = Integer.valueOf(tzOffset);
    }
    
    /**
     * Construct an UtcTimestamp from an ISO8601-formatted string.
     * 
     * @param s The string to parse.
     * @throws ParseException If the timestamp is not in the ISO8601 format.
     * @see #setString(String)
     */
    public UtcTimestamp(String s) throws ParseException {
        this.setString(s,null);
    }

    /**
     * Construct an UtcTimestamp from an ISO8601-formatted string.
     * 
     * @param s The string to parse.
     * @param timeZone A timezone to calculate the timezone offset for strings without
     *                 a timezone designator. 
     * @throws ParseException If the timestamp is not in the ISO8601 format.
     * @see #setString(String, TimeZone)
     */
    public UtcTimestamp(String s, TimeZone timeZone) throws ParseException {
        this.setString(s,timeZone);
    }

   /**
     * @return The current timestamp with the timezone offset of the default timezone.
     *         This method is in contrast to the default constructor,
     *         which generated the current timestamp with a time zone offset of zero.
     *         
     * @see #UtcTimestamp()
     */
    public static UtcTimestamp now() {
        
        return new UtcTimestamp(TimeZone.getDefault());
    }
    
    /**
     * @param tzOffset in minutes
     * @return The current timestamp with the given timezone offset.
     */
    public static UtcTimestamp now(int tzOffset) {
        
        return new UtcTimestamp(System.currentTimeMillis(),tzOffset);
    }
    
    /**
     * This method duplicates the functionality of {@link #UtcTimestamp(TimeZone)}
     * for convenience reasons.
     * 
     * @param timeZone The timezone to use for calculating the timezone offset.
     * @return The current timestamp with the timezone offset of the given timezone.
     */
    public static UtcTimestamp now(TimeZone timeZone) {
        
        return new UtcTimestamp(timeZone);
    }
    
    /**
     * Convert this UtcTimestamp to java.util.Calendar.
     * @return A calendar instance for the given instance in time and the
     *         field {@link Calendar#ZONE_OFFSET} set to the timezone offset
     *         of this UTC timestamp.
     */
    public Calendar toCalendar() {
        if (this.utcMillis == null || this.tzOffset == null)
        {
            return null;
        }
        else
        {
            Calendar c = Calendar.getInstance(UtcTimestamp.utcTimezone);
            c.setTimeInMillis(this.utcMillis.longValue() + (long) this.tzOffset.intValue() * 60000);
            c.set(Calendar.ZONE_OFFSET, this.tzOffset.intValue() * 60000);
            return c;
        }
    }
    
    /** 
     * Convert the passed Calendar object to UtcTimestamp.
     * @param c
     */
    @Transient
    public void setFromCalendar(Calendar c) {
        if (c != null) 
        {
            this.utcMillis = Long.valueOf(c.getTimeInMillis());
            this.tzOffset = Integer.valueOf((c.get(Calendar.ZONE_OFFSET) + c.get(Calendar.DST_OFFSET)) / 60000);
        }
        else
        {
            this.utcMillis = null;
            this.tzOffset = null;
        }
        this.dateFields = null;
    }
    
    /**
     * @return Returns the tzOffset.
     */
    public Integer getTzOffset() {
        return this.tzOffset;
    }

    /**
     * @param tzOffset
     *            The tzOffset to set.
     */
    public void setTzOffset(Integer tzOffset) {
        this.tzOffset = tzOffset;
        this.dateFields = null;
    }

    /**
     * @return Returns the utcMillis.
     */
    public Long getUtcMillis() {
        return this.utcMillis;
    }

    /**
     * @param utcMillis
     *            The utcMillis to set.
     */
    public void setUtcMillis(Long utcMillis) {
        this.utcMillis = utcMillis;
        this.dateFields = null;
    }
    
    /**
     * Set the day relative to the currently effective timezone.
     * 
     * @param year The year in the range from 1759 to 9999.
     * @param month The month in the range from 1 to 12.
     * @param day The month in the range from 1 to 31.
     */
    @Transient
    public void setDate(int year, int month, int day) {
        
        long ts = makeUtcMillis(year, month, day);
        if (this.tzOffset != null)
            ts -= this.tzOffset * MILLISECONDS_PER_MINUTE;
            
        this.utcMillis = ts;
        this.dateFields = null;
    }
    
    /**
     * Set the date, time and timezone offset the given instance in time in the given timezone
     * considering daylight savings time.
     * 
     * @param year The year in the range from 1759 to 9999.
     * @param month The month in the range from 1 to 12.
     * @param day The month in the range from 1 to 31.
     * @param millisOfDay The milliseconds since midnight of the given day.
     * @param timeZone The timezone to consider.
     */
    @Transient
    public void setDateTime(int year, int month, int day, int millisOfDay, TimeZone timeZone) {
        
        int noff = timeZone.getRawOffset();
        
        long ts = makeUtcMillis(year, month, day)-noff+millisOfDay;
        
        int off = timeZone.getOffset(ts);
        
        if (off != noff)
            ts += noff - off;
        
        this.tzOffset  = off / 60000;
        this.utcMillis = ts;
        this.dateFields = null;
    }
    
    /**
     * Set the date, time and timezone offset the given instance in time in the given timezone
     * considering daylight savings time.
     * 
     * @param year The year in the range from 1759 to 9999.
     * @param month The month in the range from 1 to 12.
     * @param day The month in the range from 1 to 31.
     * @param hour The hour of the day in the range from 0 to 23.
     * @param minute The minute of the hour in the range from 0 to 59.
     * @param second The second of the minute in the range from 0 to 59.
     * @param ms The millisecond of the second in the range from 0 to 999.
     * @param timeZone The timezone to consider.
     */
    @Transient
    public void setDateTime(int year, int month, int day, int hour, int minute, int second, int ms, TimeZone timeZone) {
        
        this.setDateTime(year, month, day, hour*3600000 + minute * 60000 + second * 1000 + ms, timeZone);
    }
    
    /**
     * Set the date, time and timezone offset the given instance in time in the given timezone
     * considering daylight savings time.
     * 
     * @param year The year in the range from 1759 to 9999.
     * @param month The month in the range from 1 to 12.
     * @param day The month in the range from 1 to 31.
     * @param hour The hour of the day in the range from 0 to 23.
     * @param minute The minute of the hour in the range from 0 to 59.
     * @param second The second of the minute in the range from 0 to 59.
     * @param timeZone The timezone to consider.
     */
    @Transient
    public void setDateTime(int year, int month, int day, int hour, int minute, int second, TimeZone timeZone) {
        
        this.setDateTime(year, month, day, hour*3600000 + minute * 60000 + second * 1000, timeZone);
    }
    
  /**
     * @param hour
     * @param minute
     * @param second
     */
    @Transient
    public void setTimeOfDay(int hour, int minute, int second) {
        setTimeOfDay(hour, minute, second, 0);
    }
    
    /**
     * @param hour
     * @param minute
     * @param second
     * @param millis
     */
    @Transient
    public void setTimeOfDay(int hour, int minute, int second, int millis) {
        long ts = this.getLocalTimestamp();
        
        ts = (ts / MILLISECONDS_PER_DAY) * MILLISECONDS_PER_DAY +
        hour * MILLISECONDS_PER_HOUR + minute * MILLISECONDS_PER_MINUTE + second * MILLISECONDS_PER_SECOND + (long) millis;
        
        if (this.tzOffset != null)
            ts -= this.tzOffset * MILLISECONDS_PER_MINUTE;
        
        this.utcMillis = ts - REF_TIMESTAMP;
        this.dateFields = null;
    }
    
    /**
     * @return The year in the range from 1759 to 9999.
     *         The returned month is relative to the local timezone, if a
     *         timezone offset is set. Zero is returned, if the thimestamp
     *         has been set to an illegal value.
     */
    @Transient
    public int getYear() {
        if (this.dateFields == null) initDateFields();
        return this.dateFields.year;
    }
    
   /**
    * @return The month in the range from 1 to 12.
    *         The returned month is relative to the local timezone, if a
    *         timezone offset is set.
    */
    @Transient
    public int getMonth() {
        if (this.dateFields == null) initDateFields();
        return this.dateFields.month;
    }
    
    /**
     * @return The day of month in the range from 1 to 31.
     *         The returned day is relative to the local timezone, if a
     *         timezone offset is set.
     */
    @Transient
    public int getDay() {
        if (this.dateFields == null) initDateFields();
        return this.dateFields.day;
    }
    
    /**
     * @return The day of the week represented by this timestamp in the
     *         timezone designated by tzOffset.
     *         
     * @see Calendar#SUNDAY
     * @see Calendar#MONDAY
     * @see Calendar#TUESDAY
     * @see Calendar#WEDNESDAY
     * @see Calendar#THURSDAY
     * @see Calendar#FRIDAY
     * @see Calendar#SATURDAY
     */
    @Transient
    public int getDayOfWeek() {
        return (int)((this.getLocalTimestamp() / MILLISECONDS_PER_DAY + 1)%7L) + 1;
    }
    
    /**
     * @return The number of day in the month corresponding to the
     *         timestamp in the local timezone.
     */
    @Transient
    public int getDaysOfMonth()
    {
        if (this.dateFields == null) initDateFields();
        return getDaysOfMonth(this.dateFields.year,this.dateFields.month);
    }

    /**
     * @return Whther the year corresponding to the
     *         timestamp in the local timezone is a leap year.
     */
    @Transient
    public boolean isLeapYear() {
        if (this.dateFields == null) initDateFields();
        return isLeapYear(this.dateFields.year);
    }
    
    /**
     * @return The hour of the day in the range from 0 to 23.
     *         The returned hour is relative to the local timezone, if a
     *         timezone offset is set.
     */
    @Transient
    public int getHour() {
        return (int)((this.getLocalTimestamp()/MILLISECONDS_PER_HOUR)%24L);
    }
    
    /**
     * @return The minutes in the range from 0 to 59.
     */
    @Transient
    public int getMinute() {
        return (int)((this.getLocalTimestamp()/MILLISECONDS_PER_MINUTE)%60L);
    }
    
    /**
     * @return The seconds in the range from 0 to 59.
     */
    @Transient
    public int getSecond() {
        return (int)((this.getLocalTimestamp()/MILLISECONDS_PER_SECOND)%60L);
    }
    
    /**
     * @return The milliseconds in the ragne from 0 to 999.
     */
    @Transient
    public int getMilliSecond() {
        return (int)(this.getLocalTimestamp()%MILLISECONDS_PER_SECOND);
    }
    
    /**
     * @param ms The number of milliseconds to add.
     */
    public void addMilliSeconds(long ms) {
        this.utcMillis += ms;
        this.dateFields = null;
    }
    
    /**
     * @param s The number of seconds to add.
     */
    public void addSeconds(long s) {
        this.addMilliSeconds(s * MILLISECONDS_PER_SECOND);
    }
    
    /**
     * @param m The number of minutes to add.
     */
    public void addMinutes(long m) {
        this.addMilliSeconds(m * MILLISECONDS_PER_MINUTE);
    }
    
    /**
     * @param h The number of hours to add.
     */
    public void addHours(long h) {
        this.addMilliSeconds(h * MILLISECONDS_PER_HOUR);
    }
    
    /**
     * @param d The number of days to add.
     */
    public void addDays(long d) {
        this.addMilliSeconds(d * MILLISECONDS_PER_DAY);
    }
    
    /**
     * Add <code>m</code> month to the given timestamp.
     * This method keeps the return timestamp on the last day in month, so
     * <code>UtcTimestamp("2007-02-28").addMonths(3) == UtcTimestamp("2007-05-31")</code> 
     *
     * @param m The number of month to add.
     */
    public void addMonths(int m) {
        if (this.dateFields == null) this.initDateFields();
        
        int mmm = this.dateFields.year * 12 + (this.dateFields.month-1) + m;
        
        int mm = mmm % 12 + 1;
        int yy = mmm / 12;
        
        int dd = this.dateFields.day;
        int dom = getDaysOfMonth(yy,mm);
        
        if (dd == getDaysOfMonth(this.dateFields.year, this.dateFields.month) ||
                dd > dom)
            dd = dom;
        
        long ts = makeUtcMillis(yy, mm, dd);
        ts += this.getLocalTimestamp() % MILLISECONDS_PER_DAY;
        
        if (this.tzOffset != null)
            ts -= this.tzOffset * MILLISECONDS_PER_MINUTE;
            
        this.utcMillis = ts;
        this.dateFields = null;
    }
    
    /**
     * Add <code>m</code> month to the given timestamp.
     * This method keeps the return timestamp on the last day in month, so
     * <code>UtcTimestamp("2007-02-28").addYear(-7) == UtcTimestamp("2000-02-29")</code>. 

     * @param y The number of years to add.
     */
    public void addYears(int y) {
        if (this.dateFields == null) this.initDateFields();
         
        int dd = this.dateFields.day;
        
        if (this.dateFields.month == 2 &&
                dd == getDaysOfMonth(this.dateFields.year, this.dateFields.month))
            dd = getDaysOfMonth(this.dateFields.year+y, this.dateFields.month);
            
        long ts = makeUtcMillis(this.dateFields.year+y, this.dateFields.month, dd);
        ts += this.getLocalTimestamp() % MILLISECONDS_PER_DAY;
        
        if (this.tzOffset != null)
            ts -= this.tzOffset * MILLISECONDS_PER_MINUTE;
            
        this.utcMillis = ts;
        this.dateFields = null;
    }
    
    /**
     * This method duplicates the funtionality of <code>Calendar.add()</code> in order
     * to make the transition from Calendar database objects to UtcTimestamp object
     * as easy as possible.
     * 
     * @param field The field to increment, one of {@link Calendar#YEAR},
     *           {@link Calendar#MONTH}, {@link Calendar#DAY_OF_YEAR}, {@link Calendar#DAY_OF_MONTH},
     *           {@link Calendar#HOUR}, {@link Calendar#HOUR_OF_DAY}, {@link Calendar#MINUTE},
     *           {@link Calendar#SECOND} or {@link Calendar#MILLISECOND}.
     * @param amount The amount to increment the given field.
     * 
     * @see Calendar#add(int, int)
     */
    public void add(int field, int amount) {
        switch(field) {
        case Calendar.YEAR:
            this.addYears(amount);
            break;
            
        case Calendar.MONTH:
            this.addMonths(amount);
            break;
            
        case Calendar.DAY_OF_MONTH:
        case Calendar.DAY_OF_YEAR:
            this.addDays(amount);
            break;
            
        case Calendar.HOUR:
        case Calendar.HOUR_OF_DAY:
            this.addHours(amount);
            break;

        case Calendar.MINUTE:
            this.addMinutes(amount);
            break;

        case Calendar.SECOND:
            this.addSeconds(amount);
            break;
            
        case Calendar.MILLISECOND:
            this.addMilliSeconds(amount);
            break;
        }
    }
    
    /**
     * Perform 'add' method to a clone of this.
     * @param field The field to increment.
     * @param amount The amount to increment the given field.
     * @return A new instance with increnemented timestamp.
     * @see #add(int, int)
     */
    public UtcTimestamp cloneAdd(int field, int amount)
    {
        UtcTimestamp test = null;
        test = (UtcTimestamp) this.clone();
        test.add(field, amount);
        return test;
    }
    public UtcTimestamp cloneAddYears(int amount)
    {
        return cloneAdd(Calendar.YEAR, amount);
    }
    public UtcTimestamp cloneAddMonths(int amount)
    {
        return cloneAdd(Calendar.MONTH, amount);
    }
    public UtcTimestamp cloneAddDays(int amount)
    {
        return cloneAdd(Calendar.DAY_OF_MONTH, amount);
    }
    public UtcTimestamp cloneAddHours(int amount)
    {
        return cloneAdd(Calendar.HOUR_OF_DAY, amount);
    }
    public UtcTimestamp cloneAddMinutes(int amount)
    {
        return cloneAdd(Calendar.MINUTE, amount);
    }
    public UtcTimestamp cloneAddSeconds(int amount)
    {
        return cloneAdd(Calendar.SECOND, amount);
    }
    public UtcTimestamp cloneAddMillis(int amount)
    {
        return cloneAdd(Calendar.MILLISECOND, amount);
    }
    
    /**
     * This method duplicates the funtionality of <code>Calendar.add()</code> considering
     * any change of the timezone offset of the given timezone. If the <code>timeZone</code> argument
     * is <code>null</code>, this method is equivalent to {@link #add(int, int)}.
     * 
     * @param timeZone The timezone to account for or <code>null</code>.
     * @param field The field to increment, one of {@link Calendar#YEAR},
     *           {@link Calendar#YEAR}, {@link Calendar#MONTH}, {@link Calendar#DAY_OF_MONTH},
     *           {@link Calendar#HOUR}, {@link Calendar#HOUR_OF_DAY}, {@link Calendar#MINUTE},
     *           {@link Calendar#SECOND} or {@link Calendar#MILLISECOND}.
     * @param amount The amount to increment the given field.
     * 
     * @see Calendar#add(int, int)
     */
    public void add(TimeZone timeZone, int field, int amount) {

        if (timeZone != null) {
        
            int old_off = timeZone.getOffset(this.utcMillis);
        
            this.add(field, amount);
   
            int new_off = timeZone.getOffset(this.utcMillis);
        
            if (old_off != new_off) {
                this.tzOffset += (new_off - old_off) / 60000;
                this.utcMillis  -= (new_off - old_off);
            }
        }
        else
            this.add(field, amount);     
    }
    
    /**
     * Return a deep copy of this timestamp, which is moved to the first day of the week. 
     * 
     * The timestamp object itself is not altered.
     * 
     * @param firstDayOfWeek The first day in the week as one of the constants
     *    {@link Calendar#SUNDAY}, ..., {@link Calendar#SATURDAY}.
     * @param timeZone The timezone used to correct the timestamp when
     *                 the timezone offset changes between this day and the first day in
     *                 the week. 
     */
    public UtcTimestamp cloneToFirstDayOfWeek(int firstDayOfWeek, TimeZone timeZone) {
        
        UtcTimestamp ret = (UtcTimestamp) this.clone();
        ret.moveToFirstDayOfWeek(firstDayOfWeek,timeZone);
        return ret;
    }
    
    /**
     * Move this timestamp to the first day of the week. 
     * 
     * @param firstDayOfWeek The first day in the week as one of the constants
     *    {@link Calendar#SUNDAY}, ..., {@link Calendar#SATURDAY}.
     * @param timeZone The timezone used to correct the timestamp when
     *                 the timezone offset changes between this day and the first day in
     *                 the week. 
     */
    public void moveToFirstDayOfWeek(int firstDayOfWeek, TimeZone timeZone) {
        
        int w = this.getDayOfWeek();
        
        this.add(timeZone,Calendar.DAY_OF_MONTH,-(7 + w - firstDayOfWeek) % 7);
    }
    
    /**
     * @param t1 The first timestamp.
     * @param t2 The second timestamp.
     * @return The number of day between the two timestamps.
     */
    public static int daysBetween(UtcTimestamp t1, UtcTimestamp t2) {
        
        return (int)(t2.getLocalTimestamp()/MILLISECONDS_PER_DAY-t1.getLocalTimestamp()/MILLISECONDS_PER_DAY);
    }
    
    /**
     * @param t1 The first timestamp.
     * @param t2 The second timestamp.
     * @param firstDayOfWeek The first day of the week as one of the constants
     *    {@link Calendar#SUNDAY}, ..., {@link Calendar#SATURDAY}..
     * @return The number of week between the two timestamps.
     */
    public static int weeksBetween(UtcTimestamp t1, UtcTimestamp t2, int firstDayOfWeek) {
        
       int days = daysBetween(t1, t2);
        
        int w1 = t1.getDayOfWeek();
        int w2 = t2.getDayOfWeek();
        
       // decrease start date to first day in week.
        // (difference increases as first day decreases...) 
        days += (7 + w1 - firstDayOfWeek) % 7;
        
        // decrease end date to first day in week.
        // (difference decreases as first day decreases...) 
        days -= (7 + w2 - firstDayOfWeek) % 7;
        
        assert (days % 7 == 0);
        
        return days / 7;
    }
    
    /**
     * @param t1 The first timestamp.
     * @param t2 The second timestamp.
     * @return The number of months between the two timestamps.
     */
    public static int monthsBetween(UtcTimestamp t1, UtcTimestamp t2) {
        
        return (t2.getMonth() - t1.getMonth()) + 12 * (t2.getYear() - t1.getYear());
    }
    
    /**
     * @param format One of {@link ISO8601Format#DATE_FORMAT},
     *               {@link ISO8601Format#DATETIME_FORMAT},
     *               {@link ISO8601Format#MILLISECOND_FORMAT},
     *               {@link ISO8601Format#MILLISECOND_FORMAT_NO_TZ},
     *               or {@link ISO8601Format#DATETIME_FORMAT_NO_TZ}.
     * @return The formatted timestamp cf. to ISO8601.
     */
    public String toString (int format)
    {
        if (this.utcMillis == null) return null;
        
        if (this.dateFields == null) initDateFields();
        
        if (format == ISO8601Format.DATE_FORMAT) {
            
            return String.format("%04d-%02d-%02d",
                    this.dateFields.year,
                    this.dateFields.month,
                    this.dateFields.day);

        } else if (format == ISO8601Format.DATETIME_FORMAT) {
            long ts = this.getLocalTimestamp();
            
            if (this.tzOffset != null) {
            
                if (this.tzOffset == 0)
                    return String.format("%04d-%02d-%02dT%02d:%02d:%02dZ",
                            this.dateFields.year,
                            this.dateFields.month,
                            this.dateFields.day,
                            (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                            (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                            (int)((ts/MILLISECONDS_PER_SECOND)%60L) );
                else if (this.tzOffset > 0)
                    return String.format("%04d-%02d-%02dT%02d:%02d:%02d+%02d:%02d",
                            this.dateFields.year,
                            this.dateFields.month,
                            this.dateFields.day,
                            (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                            (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                            (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                            this.tzOffset/60,
                            this.tzOffset%60);
                else
                    return String.format("%04d-%02d-%02dT%02d:%02d:%02d-%02d:%02d",
                            this.dateFields.year,
                            this.dateFields.month,
                            this.dateFields.day,
                            (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                            (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                            (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                            (-this.tzOffset)/60,
                            (-this.tzOffset)%60);
                
            } else {
                
                return String.format("%04d-%02d-%02dT%02d:%02d:%02d",
                        this.dateFields.year,
                        this.dateFields.month,
                        this.dateFields.day,
                        (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                        (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                        (int)((ts/MILLISECONDS_PER_SECOND)%60L) );
            }
            
        } else if (format == ISO8601Format.DATETIME_FORMAT_NO_TZ) {
            long ts = this.getLocalTimestamp();
            
            return String.format("%04d-%02d-%02dT%02d:%02d:%02d",
                    this.dateFields.year,
                    this.dateFields.month,
                    this.dateFields.day,
                    (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                    (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                    (int)((ts/MILLISECONDS_PER_SECOND)%60L) );
            
        } else if (format == ISO8601Format.MILLISECOND_FORMAT_NO_TZ) {
            long ts = this.getLocalTimestamp();
            
            return String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d",
                    this.dateFields.year,
                    this.dateFields.month,
                    this.dateFields.day,
                    (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                    (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                    (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                    (int)(ts%MILLISECONDS_PER_SECOND) );
        }
        /* ISO8601Format.MILLISECOND_FORMAT */
        else {
            
            long ts = this.getLocalTimestamp();
        
            if (this.tzOffset != null) {
            
                if (this.tzOffset == 0)
                    return String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03dZ",
                            this.dateFields.year,
                            this.dateFields.month,
                            this.dateFields.day,
                            (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                            (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                            (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                            (int)(ts%MILLISECONDS_PER_SECOND));
                else if (this.tzOffset > 0)
                    return String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d+%02d:%02d",
                            this.dateFields.year,
                            this.dateFields.month,
                            this.dateFields.day,
                            (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                            (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                            (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                            (int)(ts%MILLISECONDS_PER_SECOND),
                            this.tzOffset/60,
                            this.tzOffset%60);
                else
                    return String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d-%02d:%02d",
                            this.dateFields.year,
                            this.dateFields.month,
                            this.dateFields.day,
                            (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                            (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                            (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                            (int)(ts%MILLISECONDS_PER_SECOND),
                            (-this.tzOffset)/60,
                            (-this.tzOffset)%60);
                
            } else {
                
                return String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d",
                        this.dateFields.year,
                        this.dateFields.month,
                        this.dateFields.day,
                        (int)((ts/MILLISECONDS_PER_HOUR)%24L),
                        (int)((ts/MILLISECONDS_PER_MINUTE)%60L),
                        (int)((ts/MILLISECONDS_PER_SECOND)%60L),
                        (int)(ts%MILLISECONDS_PER_SECOND) );
            }
        }   
    }
    
    /**
     * Set an ISO8601 formatted timestamp to this object. If the string contains no timezone designator,
     * the returned timestamp has a <code>null</code> timezone offset. 
     * 
     * @param s A string in ISO8601 format.
     * 
     * @throws ParseException If the given string is not formatted like ISO8601.
     */
    @Transient
    public void setString(String s) throws ParseException
    {
        this.setString(s,null);
    }
    
    /**
     * Set an ISO8601 formatted timestamp to this object. If the string contains no timezone designator,
     * and the given timezone is not <code>null</code>,
     * the returned timestamp has a timezone offset corresponding to the given timezone.
     *  If the string contains no timezone designator and the given timezone is <code>null</code>,
     * the returned timestamp has a <code>null</code> timezone offset. 
     * 
     * @param s A string in ISO8601 format.
     * @param timeZone A timezone to calculate the timezone offset for strings without
     *                 a timezone designator. 
     *  
     * @throws ParseException If the given string is not formatted like ISO8601.
     */
    @Transient
    public void setString(String s, TimeZone timeZone) throws ParseException
        {
        int pos = 0;
        
        try {
            int y = Integer.valueOf(s.substring(pos,pos+4));
            pos+=4;
            if (s.charAt(pos) != '-') throw new ParseException(s,pos);
            ++pos;
            int m = Integer.valueOf(s.substring(pos,pos+2));
            pos+=2;
            if (s.charAt(pos) != '-') throw new ParseException(s,pos);
            ++pos;
            int d = Integer.valueOf(s.substring(pos,pos+2));
            pos += 2;
            
            if (pos < s.length()) {
                if (s.charAt(pos) != 'T') throw new ParseException(s,pos);
                ++pos;
                
                int hh = Integer.valueOf(s.substring(pos,pos+2));
                pos+=2;
                if (s.charAt(pos) != ':') throw new ParseException(s,pos);
                ++pos;
                int mm = Integer.valueOf(s.substring(pos,pos+2));
                pos+=2;
                if (s.charAt(pos) != ':') throw new ParseException(s,pos);
                ++pos;
                int ss = Integer.valueOf(s.substring(pos,pos+2));
                pos += 2;
               
                long ms = ((long)hh * MILLISECONDS_PER_HOUR) + ((long)mm * MILLISECONDS_PER_MINUTE) + ((long)ss * MILLISECONDS_PER_SECOND);
                
                if (pos < s.length() && s.charAt(pos) == '.') {
                    ++pos;
                    int l = 0;
                    while (pos+l < s.length() && Character.isDigit(s.charAt(pos+l))) ++l;
                
                    int sss= Integer.valueOf(s.substring(pos,pos+(l > 3 ? 3 : l)));
                    switch (l) {
                    case 1: sss *= 100; break;
                    case 2: sss *= 10; break;
                    }
                    
                    ms += sss;
                    pos+=l;
                 }
 
                Integer tz_off = null;
                
                if (pos < s.length()) {
                    if (s.charAt(pos) == 'Z') {
                        ++pos;
                        tz_off = 0;
                    }
                    else if (s.charAt(pos) == '+') {
                        ++pos;
                        int tz_hh = Integer.valueOf(s.substring(pos,pos+2));
                        pos+=2;
                        if (s.charAt(pos) != ':') throw new ParseException(s,pos);
                        ++pos;
                        int tz_mm = Integer.valueOf(s.substring(pos,pos+2));
                        pos+=2;
                        
                        tz_off = tz_hh*60 + tz_mm;
                    }
                    else if (s.charAt(pos) == '-') {
                        ++pos;
                        int tz_hh = Integer.valueOf(s.substring(pos,pos+2));
                        pos+=2;
                        if (s.charAt(pos) != ':') throw new ParseException(s,pos);
                        ++pos;
                        int tz_mm = Integer.valueOf(s.substring(pos,pos+2));
                        pos+=2;
                        
                        tz_off = tz_hh*-60 - tz_mm;
                    }
                    else 
                        throw new ParseException(s,pos);
                    
                    if (pos<s.length())
                        throw new ParseException(s,pos);
                }
                
                if (tz_off != null || timeZone == null) {
                    this.setTzOffset(tz_off);   
                    this.setDate(y,m,d);    
                    this.addMilliSeconds(ms);
                }
                else {
                    this.setDateTime(y,m,d,(int)ms,timeZone);
                }
            }
            else {

                if (timeZone == null) {
                    
                    this.setTzOffset(null);
                    this.setDate(y,m,d);
                }
                else {
                    this.setDateTime(y,m,d,0,timeZone);
                }
            }
            
        } catch(NumberFormatException e) {
            throw new ParseException(s,pos);
        }
        
    }
    
    /**
     * @return The result of <code>toString(ISO8601Format.MILLISECOND_FORMAT)</code>.
     * 
     * @see #toString(int)
     * @see ISO8601Format#DATE_FORMAT
     * @see ISO8601Format#DATETIME_FORMAT
     * @see ISO8601Format#MILLISECOND_FORMAT
     */
    public String toString() {
        
        return this.toString(ISO8601Format.MILLISECOND_FORMAT);
    }
    
    /* (non-Javadoc)
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    public int compareTo(UtcTimestamp o) {
        if (o == null) 
            return 1;
        if (this.utcMillis == null)
        {
            if (o.utcMillis == null)
                return 0;
            else
                return -1;
        }
        else
        {
            if (o.utcMillis == null)
                return 1;
            else
                return ((this.utcMillis < o.utcMillis) ? -1
                        : (this.utcMillis > o.utcMillis) ? 1
                        : 0);
        }
    }

    /**
     * Test if this timestamp is before the passed second timestamp.
     * @param other the second timestamp to test against.
     * @return true if this timestamp is before the second, false otherwise and if the second timestamp is null.
     */
    public boolean before(UtcTimestamp other)
    {
        if (other != null)
            return (this.getUtcMillis() < other.getUtcMillis());
        else
            return false;
    }
    
    /**
     * Test if this timestamp is after the passed second timestamp.
     * @param other the second timestamp to test against.
     * @return true if this timestamp is after the second and if the second timestamp is null, false otherwise.
     */
    public boolean after(UtcTimestamp other)
    {
        if (other != null)
            return (this.getUtcMillis() > other.getUtcMillis());
        else
            return true;
    }

    
    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object o)
    {
        if (o == null || !(o instanceof UtcTimestamp)) return false;
        
        UtcTimestamp ts = (UtcTimestamp)o;
        
        if (this.getUtcMillis() == null) {
            return ts.getUtcMillis() == null;
        }
        
        return this.getUtcMillis().equals(ts.getUtcMillis());
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    @Override
    protected Object clone() {
        return new UtcTimestamp(this);
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        if (this.getUtcMillis() == null) 
            return -1;
        
        return (int) (419430343L /* prime */ * this.getUtcMillis());
    }
}
