/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.datetime;

import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.Calendar;
import java.util.TimeZone;

/**
 * Formats Calendar objects cf. ISO8601.
 * 
 * Theoretically this class also parses strings to Calendars, but Calendars are a bad approach.
 * Do not use {@link #parseObject(String)} or {@link #parseObject(String, ParsePosition)}.
 * Rather use {@link UtcTimestamp} or {@link UtcTimestamp#setString(String)}.  
 * @author wglas
 */
public class ISO8601Format extends Format
{
    /**
     * To be changed upon class layout change.
     */
    private static final long serialVersionUID = 6403675010685159895L;

    /**
     * Format Calendars using only date fields.
     */
    public static int DATE_FORMAT = 1;
    
    /**
     * Format Calendars using date and time fields including the timezone designator.
     */
    public static int DATETIME_FORMAT = 2;
    
    /**
     * Format Calendars using date and time fields up to milliseconds including the timezone designator.
     */
    public static int MILLISECOND_FORMAT = 3;
    
    /**
     * Format Calendars using date and time fields skipping the timezone designator.
     */
    public static int DATETIME_FORMAT_NO_TZ = 4;
    
    /**
     * Format Calendars using date and time fields up to milliseconds skipping the timezone designator.
     */
    public static int MILLISECOND_FORMAT_NO_TZ = 5;
    
    private int format;
    private TimeZone timeZone;
    
    /**
     * Instantiate an ISO8601 formatter using the given format.
     * 
     * The created instance uses UTC as the timezone for {@link Calendar}
     * instances instantiated during parse operations.
     * 
     * @param format {@link ISO8601Format#DATE_FORMAT} or 
     *               {@link ISO8601Format#DATETIME_FORMAT} or
     *               {@link ISO8601Format#MILLISECOND_FORMAT} or
     *               {@link ISO8601Format#DATETIME_FORMAT_NO_TZ} or
     *               {@link ISO8601Format#MILLISECOND_FORMAT_NO_TZ}.
     */
    public ISO8601Format(int format)
    {
        this.format = format;
        this.timeZone = TimeZone.getTimeZone("UTC");
    }
    
    /**
     * Instantiate an ISO8601 formatter using the given format.
     * 
     * @param format {@link ISO8601Format#DATE_FORMAT} or 
     *               {@link ISO8601Format#DATETIME_FORMAT} or
     *               {@link ISO8601Format#MILLISECOND_FORMAT} or
     *               {@link ISO8601Format#DATETIME_FORMAT_NO_TZ} or
     *               {@link ISO8601Format#MILLISECOND_FORMAT_NO_TZ}.
     * @param timeZone The time zone used to instantiate {@link Calendar}
     *                 during parse operations. 
     */
    public ISO8601Format(int format, TimeZone timeZone)
    {
        this.format = format;
        this.timeZone = timeZone;
    }
    
    /**
     * <p>This method formats {@link Calendar} or {@link UtcTimestamp} instances according
     * to the format scpeified at construction time.</p>
     * 
     * <p>For {@link UtcTimestamp} instance it is simpler and faster to call
     * {@link UtcTimestamp#toString(int)} directly rather than using
     * an {@link ISO8601Format} instance.</p>
     */  
    @Override
    public StringBuffer format(Object o, StringBuffer buf,
            FieldPosition pos)
    {
        if (o instanceof UtcTimestamp)
            return buf.append(((UtcTimestamp)o).toString(this.format));
        
        Calendar c = (Calendar)o;
        
        if (this.format == DATE_FORMAT)
            buf.append(String.format("%04d-%02d-%02d",
                    c.get(Calendar.YEAR),
                    c.get(Calendar.MONTH)+1,
                    c.get(Calendar.DAY_OF_MONTH)
            ));
        else if (this.format == DATETIME_FORMAT)
        {
            int off = c.get(Calendar.ZONE_OFFSET) + c.get(Calendar.DST_OFFSET);
            
            if (off == 0)
                buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02dZ",
                        c.get(Calendar.YEAR),
                        c.get(Calendar.MONTH)+1,
                        c.get(Calendar.DAY_OF_MONTH),
                        c.get(Calendar.HOUR_OF_DAY),
                        c.get(Calendar.MINUTE),
                        c.get(Calendar.SECOND)
                ));
            else if (off > 0)
                buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d+%02d:%02d",
                        c.get(Calendar.YEAR),
                        c.get(Calendar.MONTH)+1,
                        c.get(Calendar.DAY_OF_MONTH),
                        c.get(Calendar.HOUR_OF_DAY),
                        c.get(Calendar.MINUTE),
                        c.get(Calendar.SECOND),
                        off / 3600000, (off / 60000) % 60
                ));
            else
                buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d-%02d:%02d",
                        c.get(Calendar.YEAR),
                        c.get(Calendar.MONTH)+1,
                        c.get(Calendar.DAY_OF_MONTH),
                        c.get(Calendar.HOUR_OF_DAY),
                        c.get(Calendar.MINUTE),
                        c.get(Calendar.SECOND),
                        off / -3600000, (off / -60000) % 60
                ));
               
        }
        else if (this.format == DATETIME_FORMAT_NO_TZ)
        {
            buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d",
                    c.get(Calendar.YEAR),
                    c.get(Calendar.MONTH)+1,
                    c.get(Calendar.DAY_OF_MONTH),
                    c.get(Calendar.HOUR_OF_DAY),
                    c.get(Calendar.MINUTE),
                    c.get(Calendar.SECOND)));
        }
        else if (this.format == MILLISECOND_FORMAT_NO_TZ)
        {
            buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d",
                    c.get(Calendar.YEAR),
                    c.get(Calendar.MONTH)+1,
                    c.get(Calendar.DAY_OF_MONTH),
                    c.get(Calendar.HOUR_OF_DAY),
                    c.get(Calendar.MINUTE),
                    c.get(Calendar.SECOND),
                    c.get(Calendar.MILLISECOND)));
        }
        else
        {
            int off = c.get(Calendar.ZONE_OFFSET) + c.get(Calendar.DST_OFFSET);
            
            if (off == 0)
                buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03dZ",
                        c.get(Calendar.YEAR),
                        c.get(Calendar.MONTH)+1,
                        c.get(Calendar.DAY_OF_MONTH),
                        c.get(Calendar.HOUR_OF_DAY),
                        c.get(Calendar.MINUTE),
                        c.get(Calendar.SECOND),
                        c.get(Calendar.MILLISECOND)
                ));
            else if (off > 0)
                buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d+%02d:%02d",
                        c.get(Calendar.YEAR),
                        c.get(Calendar.MONTH)+1,
                        c.get(Calendar.DAY_OF_MONTH),
                        c.get(Calendar.HOUR_OF_DAY),
                        c.get(Calendar.MINUTE),
                        c.get(Calendar.SECOND),
                        c.get(Calendar.MILLISECOND),
                        off / 3600000, (off / 60000) % 60
                ));
            else
                buf.append(String.format("%04d-%02d-%02dT%02d:%02d:%02d.%03d-%02d:%02d",
                        c.get(Calendar.YEAR),
                        c.get(Calendar.MONTH)+1,
                        c.get(Calendar.DAY_OF_MONTH),
                        c.get(Calendar.HOUR_OF_DAY),
                        c.get(Calendar.MINUTE),
                        c.get(Calendar.SECOND),
                        c.get(Calendar.MILLISECOND),
                        off / -3600000, (off / -60000) % 60
                ));
               
        }
        
        return buf;
    }

    static private int extractInt(String str, ParsePosition pos, int nDigit)
    {
       int ret = 0;
        
       int idx = pos.getIndex();
       int iDigit = 0;
       
       while (iDigit < nDigit && Character.isDigit(str.charAt(idx)))
        {
            ret = 10*ret + Character.digit(str.charAt(idx), 10);
            ++idx;
            ++iDigit;
        }
        
        if (iDigit < nDigit) { pos.setErrorIndex(idx); return -1; }
        
        pos.setIndex(idx);
        return ret;
    }
    

    /**
     * <p>This method parses an ISO8601 timestamp into a {@link Calendar} instance.
     * The returned calendar is instantiate using the timezone given at construction
     * time of this {@link ISO8601Format}.</p>
     * 
     * <p>Since a Calendar represent political timezones rather than fixed UTC
     * offsets, representing ISO8601 timestamp through a calendar is generally
     * discouraged. For an exact representation of ISO8601 timestamp
     * use {@link UtcTimestamp} or {@link UtcTimestamp#setString(String)}.</p>
     * 
     * @return A calendar <code>c</code> for which
     *   <code>c.get(Calendar.ZONE_OFFSET)+c.get(Calendar.DST_OFFSET) == tz_off</code>
     *   holds, if <code>tz_off</code> is the timezone offset given as part of the passed
     *   ISO8601 timestamp string. 
     */  
    @Deprecated
    @Override
    public Object parseObject(String str, ParsePosition pos)
    {
        int oidx = pos.getIndex();
        
        int year = extractInt(str,pos,4);
        
        if (year < 0) { pos.setIndex(oidx); return null; }
        if (str.charAt(pos.getIndex()) != '-') { pos.setErrorIndex(pos.getIndex()); pos.setIndex(oidx); return null; }
        pos.setIndex(pos.getIndex()+1);
        
        int month = extractInt(str,pos,2);
        if (month < 0) { pos.setIndex(oidx); return null; }
        if (str.charAt(pos.getIndex()) != '-') { pos.setErrorIndex(pos.getIndex()); pos.setIndex(oidx); return null; }
        pos.setIndex(pos.getIndex()+1);
        
        int day = extractInt(str,pos,2);
        if (day < 0) { pos.setIndex(oidx); return null; }
        Calendar c = Calendar.getInstance(this.timeZone);
        c.set(year, month-1, day);

        if (pos.getIndex() >= str.length() || str.charAt(pos.getIndex()) != 'T')
        {
            c.set(Calendar.HOUR_OF_DAY, 0);
            c.set(Calendar.MINUTE, 0);
            c.set(Calendar.SECOND, 0);
            c.set(Calendar.MILLISECOND, 0);
            return c;
        }
        
        pos.setIndex(pos.getIndex()+1);
        
        int hour = extractInt(str,pos,2);
        if (hour < 0) { pos.setIndex(oidx); return null; }
        if (str.charAt(pos.getIndex()) != ':') { pos.setErrorIndex(pos.getIndex()); pos.setIndex(oidx); return null; }
        pos.setIndex(pos.getIndex()+1);
      
        int minute = extractInt(str,pos,2);
        if (minute < 0) { pos.setIndex(oidx); return null; }
        if (str.charAt(pos.getIndex()) != ':') { pos.setErrorIndex(pos.getIndex()); pos.setIndex(oidx); return null; }
        pos.setIndex(pos.getIndex()+1);
      
        int second = extractInt(str,pos,2);
        if (second < 0) { pos.setIndex(oidx); return null; }
      
        c.set(Calendar.HOUR_OF_DAY, hour);
        c.set(Calendar.MINUTE, minute);
        c.set(Calendar.SECOND, second);
        
        if (pos.getIndex() >= str.length()) return c;

        int millis;
            
        if (str.charAt(pos.getIndex()) == '.')
        {
            pos.setIndex(pos.getIndex()+1);
            
            millis = extractInt(str,pos,3);
            if (millis < 0) { pos.setIndex(oidx); return null; }
        }
        else
            millis = 0;
        
        c.set(Calendar.MILLISECOND, millis);
        
        if (pos.getIndex() >= str.length()) return c;
        
        int tz_off;
        
        if (str.charAt(pos.getIndex()) == 'Z')
        {
            pos.setIndex(pos.getIndex()+1);
            tz_off = 0;
        }
        else
        {
            int sign = 0;
        
            switch (str.charAt(pos.getIndex()))
            {
            case '+':
                sign = 1;
                break;
            case '-':
                sign = -1;
                break;
            default:
                return c;
        }
       
            pos.setIndex(pos.getIndex()+1);

            int tz_hour = extractInt(str,pos,2);
            if (tz_hour < 0) { pos.setIndex(oidx); return null; }
            
            if (str.charAt(pos.getIndex()) == ':') {
                pos.setIndex(pos.getIndex() + 1);
            }
         
            int tz_minute = extractInt(str,pos,2);
            if (tz_hour < 0) { pos.setIndex(oidx); return null; }
         
            tz_off = sign * ( tz_hour * 3600000 + tz_minute * 60000 );
        }
        
        // we seek to assert the condition
        // c.get(ZONE_OFFSET)+c.get(DST_OFFSET) == tz_off
        c.set(Calendar.ZONE_OFFSET,tz_off-c.get(Calendar.DST_OFFSET));
        
        assert c.get(Calendar.ZONE_OFFSET)+c.get(Calendar.DST_OFFSET) == tz_off;
        
        return c;
    }

}
