/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 13.01.2008
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.datetime;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.persistence.Embeddable;
import javax.persistence.Transient;

/**
 * A class to store a simple Gregorian Date (i.e. year, month, day) as 8 digit integer (yyyymmdd) in a database.
 * The purpose is to protect the e.g. dates of birth, historical dates etc.
 * against any layer's pseudo intelligence regarding time zones and so on.
 *
 * @author lech
 *
 */
@Embeddable
public class GlobalDate implements Comparable<GlobalDate>, Serializable {

    private static final long serialVersionUID = -2458688803800920831L;

    protected int year;

    protected int month;

    protected int day;

    protected Integer yyyymmdd;

    public GlobalDate() {
        super();
        setDate(0,0,0);
    }

    /**
     * @param year 1-9999
     * @param month 1-12
     * @param day 1-31
     */
    public GlobalDate(int year, int month, int day) {
        super();
        this.setDate(year, month, day);
    }

    public GlobalDate(Date date) {
        super();
        this.setDate(date);
    }

    public GlobalDate(Calendar calendar) {
        super();
        this.setDate(calendar);
    }

    public GlobalDate(UtcTimestamp utcTimestamp) {
        super();
        this.setDate(utcTimestamp);
    }

    public GlobalDate(long utcMillis) {
        super();
        this.setDate(utcMillis);
    }

    public GlobalDate(int yyyymmdd) {
        super();
        this.setYyyymmdd(yyyymmdd);
    }

    protected void ymd2yyyymmdd() {
        this.yyyymmdd = this.year*10000 + this.month*100 + this.day;
    }

    protected void yyyymmdd2ymd() {
        this.year  = this.yyyymmdd / 10000;
        final int mmdd = this.yyyymmdd - this.year*10000;
        this.month = mmdd / 100;
        this.day   = mmdd - this.month*100;
    }

    /**
     * @param year 1-9999
     * @param month 1-12
     * @param day 1-31
     */
    @Transient
    public void setDate(int year, int month, int day) {
        this.year = year;
        this.month = month;
        this.day = day;
        ymd2yyyymmdd();
    }

    public Date toDate() {
        return toCalendar().getTime();
    }
    
    public Calendar toCalendar() {
        return new GregorianCalendar(this.year, this.month-1, this.day);
    }

    public UtcTimestamp toUtcTimestamp() {
        return new UtcTimestamp(toCalendar());
    }
    
    public long toMillis() {
        return toCalendar().getTimeInMillis();
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     * returns yyyy-mm-dd
     */
    public String toString() {
        StringBuffer sb=new StringBuffer();
        sb.append(Integer.valueOf(this.year));
        sb.append('-');
        sb.append((char)(this.month / 10 + '0'));
        sb.append((char)(this.month % 10 + '0'));
        sb.append('-');
        sb.append((char)(this.day / 10 + '0'));
        sb.append((char)(this.day % 10 + '0'));
        return sb.toString();
    }

    @Transient
    public void setDate(Date date) {
        Calendar calendar=Calendar.getInstance();
        calendar.setTime(date);
        setDate(calendar);
    }

    @Transient
    public void setDate(Calendar calendar) {
        setDate(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH)+1, calendar.get(Calendar.DAY_OF_MONTH));
    }

    @Transient
    public void setDate(UtcTimestamp utcTimestamp) {
        setDate(utcTimestamp.getYear(), utcTimestamp.getMonth(), utcTimestamp.getDay());
    }

    @Transient
    public void setDate(long utcMillis) {
        Calendar calendar=Calendar.getInstance();
        calendar.setTimeInMillis(utcMillis);
        setDate(calendar);
    }

    /**
     * @return the yyyymmdd
     */
    public Integer getYyyymmdd() {
        return this.yyyymmdd;
    }

    /**
     * @param yyyymmdd the yyyymmdd to set
     */
    public void setYyyymmdd(Integer yyyymmdd) {
        this.yyyymmdd = yyyymmdd;
        yyyymmdd2ymd();
    }

    /**
     * @return the year, 1-9999
     */
    @Transient
    public int getYear() {
        return this.year;
    }

    /**
     * @param year the year to set, 1-9999
     */
    @Transient
    public void setYear(int year) {
        this.year = year;
        ymd2yyyymmdd();
    }

    /**
     * @return the month, 1-12
     */
    @Transient
    public int getMonth() {
        return this.month;
    }

    /**
     * @param month the month to set, 1-12
     */
    @Transient
    public void setMonth(int month) {
        this.month = month;
        ymd2yyyymmdd();
    }

    /**
     * @return the day, 1-31
     */
    @Transient
    public int getDay() {
        return this.day;
    }

    /**
     * @param day the day to set, 1-31
     */
    @Transient
    public void setDay(int day) {
        this.day = day;
        ymd2yyyymmdd();
    }

    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + this.yyyymmdd;
        return result;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        final GlobalDate other = (GlobalDate) obj;
        if (this.yyyymmdd.equals(other.yyyymmdd.intValue()))
            return true;
        return false;
    }

    public int compareTo(GlobalDate o) {
        if (o==null)
            throw new NullPointerException("compareTo(GlobalDate other): other==null");
        return Integer.valueOf(this.yyyymmdd).compareTo(o.getYyyymmdd());
    }

    public boolean before(GlobalDate other) {
        if (other==null) 
            throw new NullPointerException("before(GlobalDate other): other==null");
        return this.yyyymmdd.intValue() < other.getYyyymmdd().intValue();
    }

    public boolean after(GlobalDate other) {
        if (other==null) 
            throw new NullPointerException("after(GlobalDate other): other==null");
        return this.yyyymmdd.intValue() > other.getYyyymmdd().intValue();
    }

}
