/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 20.01.2009
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.datetime;

import java.util.Calendar;
import java.util.Comparator;

/**
 * Static helper functions for calendar objects, which overcome the
 * most painful omissions in the Calendar API.
 * 
 * @author wglas
 */
public abstract class CalendarHelper {

    /**
     * A comparator instance which compares calendar by their date information
     * only. Time and time zone information is ignored.
     */
    public static Comparator<Calendar> DATE_ONLY_COMPARATOR =
        new Comparator<Calendar>() {

            /**
             * @param c1 The first date to compare by day.
             * @param c2 The second date to compare by day.
             * @return A value less than, equal to or greater than zero,
             *         if c1 is before c2, c1 is on the same day as c2 or c1 is after c2
             *         according to the {@link Calendar#YEAR} and {@link Calendar#DAY_OF_YEAR}
             *         fields.
             */
            public int compare(Calendar c1, Calendar c2) {
               
                int r = c1.get(Calendar.YEAR) - c2 .get(Calendar.YEAR);
                
                if (r != 0) return r;
                
                return c1.get(Calendar.DAY_OF_YEAR) - c2 .get(Calendar.DAY_OF_YEAR);
            }
        
    };
        
    /**
     * Calculate the number of days between the two given dates.
     * 
     * @param c1 The start date.
     * @param c2 The end date.
     * @return The number of days between <code>c1</code> and <code>c2</code>.
     */
    public static int daysBetween(Calendar c1, Calendar c2) {
        
        int yearDays = 0;
        
        int y1 = c1.get(Calendar.YEAR);
        int y2 = c2.get(Calendar.YEAR);
        
        if (y2 > y1) {
            
            Calendar c = (Calendar) c1.clone();
            
            while (c.get(Calendar.YEAR)<y2) {
                
                yearDays += c.getActualMaximum(Calendar.DAY_OF_YEAR);
                c.add(Calendar.YEAR,1);
            }
        }
        else if (y2 < y1) {
            
            Calendar c = (Calendar) c2.clone();
            
            while (c.get(Calendar.YEAR)<y1) {
                
                yearDays -= c.getActualMaximum(Calendar.DAY_OF_YEAR);
                c.add(Calendar.YEAR,1);
            }
        }
        
        return yearDays + c2.get(Calendar.DAY_OF_YEAR) - c1.get(Calendar.DAY_OF_YEAR);
    }
   
    /**
     * Calculate the number of weeks between the two given dates.
     * 
     * The number of weeks is actually is calculated by means of
     * the last day <code>d1</code> with <code>d1.get(Calendar.DAY_OF_WEEK) == firstDayInWeek</code>
     * equal to or before the dates <code>c1</code>
     * and the last day <code>d2</code> with <code>d2.get(Calendar.DAY_OF_WEEK) == firstDayInWeek</code>
     * equal to or before the dates <code>c1</code>.
     * 
     * The number of weeks is then given by
     * <code>daysBetween(d1,d2)/7</code>.
     * 
     * @param c1 The start date.
     * @param c2 The end date.
     * @param firstDayOfWeek The first day of the week as one of the constants
     *    {@link Calendar#SUNDAY}, ..., {@link Calendar#SATURDAY}..
     * @return The number of days between <code>c1</code> and <code>c2</code>.
     */
    public static int weeksBetween(Calendar c1, Calendar c2, int firstDayOfWeek) {
        
        int days = daysBetween(c1, c2);
        
        int w1 = c1.get(Calendar.DAY_OF_WEEK);
        int w2 = c2.get(Calendar.DAY_OF_WEEK);
        
       // decrease start date to first day in week.
        // (difference increases as first day decreases...) 
        days += (7 + w1 - firstDayOfWeek) % 7;
        
        // decrease end date to first day in week.
        // (difference decreases as first day decreases...) 
        days -= (7 + w2 - firstDayOfWeek) % 7;
        
        assert (days % 7 == 0);
        
        return days / 7;
    }
    
    /**
     * @param c The calendar to move to the first day of week.
     * @param firstDayOfWeek The first day in the week as one of the constants
     *    {@link Calendar#SUNDAY}, ..., {@link Calendar#SATURDAY}.
     */
    public static void moveToFirstDayOfWeek(Calendar c, int firstDayOfWeek) {
        
        int w = c.get(Calendar.DAY_OF_WEEK);
        
        c.add(Calendar.DAY_OF_YEAR,-(7 + w - firstDayOfWeek) % 7);
    }
    
    /**
     * This function move the given calendar to first day of the week, where
     * the first day of the week is determined by {@link Calendar#getFirstDayOfWeek()},
     * which reflects the locale-specific settings of the given calendar. 
     * 
     * @param c The calendar to move to the first day of week.
     */
    public static void moveToFirstDayOfWeek(Calendar c) {
        
        moveToFirstDayOfWeek(c,c.getFirstDayOfWeek());
    }
    
    /**
     * This function returns a copy of the given calendar, which lies on the first day of
     * the week. 
     *
     * @param c The input calendar, which is not modified.
     * @param firstDayOfWeek The first day in the week as one of the constants
     *    {@link Calendar#SUNDAY}, ..., {@link Calendar#SATURDAY}.
     * @return A modified deep copy <code>r</code> of <code>c</code> with
     *         <code>r.get(Calendar.DAY_OF_WEEK)==firstDayInWeek</code>.
     */
    public static Calendar cloneToFirstDayOfWeek(Calendar c, int firstDayOfWeek) {
        
        Calendar ret = (Calendar)c.clone();
        moveToFirstDayOfWeek(ret,firstDayOfWeek);
        return ret;
    }
    
    /**
     * This function returns a copy of the given calendar, which lies on the first day of
     * the week, where the first day of the week is determined by
     * {@link Calendar#getFirstDayOfWeek()},
     * which reflects the locale-specific settings of the given calendar. 
     * 
     * @param c The input calendar, which is not modified.
     *
     * @return A modified deep copy <code>r</code> of <code>c</code> with
     *         <code>r.get(Calendar.DAY_OF_WEEK)==firstDayInWeek</code>.
     */
    public static Calendar cloneToFirstDayOfWeek(Calendar c) {
        
        return cloneToFirstDayOfWeek(c,c.getFirstDayOfWeek());
    }
    
    /**
     * Calculate the number of months between the two given dates.
     * 
     * The number of months is actually is calculated by means of
     * the {@link Calendar#MONTH} and {@link Calendar#YEAR} fields.
     * 
     * @param c1 The start date.
     * @param c2 The end date.
     * @return The number of months between <code>c1</code> and <code>c2</code>.
     */
    public static int monthsBetween(Calendar c1, Calendar c2) {

        int yearDays = 0;
        
        int y1 = c1.get(Calendar.YEAR);
        int y2 = c2.get(Calendar.YEAR);
        
        if (y2 > y1) {
            
            Calendar c = (Calendar) c1.clone();
            
            while (c.get(Calendar.YEAR)<y2) {
                
                yearDays += c.getActualMaximum(Calendar.MONTH)+1;
                c.add(Calendar.YEAR,1);
            }
        }
        else if (y2 < y1) {
            
            Calendar c = (Calendar) c2.clone();
            
            while (c.get(Calendar.YEAR)<y1) {
                
                yearDays -= c.getActualMaximum(Calendar.MONTH)+1;
                c.add(Calendar.YEAR,1);
            }
        }
        
        return yearDays + c2.get(Calendar.MONTH) - c1.get(Calendar.MONTH);
    }
}
