package org.clazzes.svc.runner.sshd;

import java.io.IOException;
import java.io.OutputStream;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class RefCountedOutputStream extends OutputStream {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(RefCountedOutputStream.class);

    private static final class SharedData {
        private final OutputStream underlying;
        private final AtomicInteger refs = new AtomicInteger(1);

        public SharedData(OutputStream underlying) {
            this.underlying = underlying;
        }
    }

    private final SharedData data;
    private final AtomicBoolean closed = new AtomicBoolean(false);

    public RefCountedOutputStream(RefCountedOutputStream other) {
        this((OutputStream) other);
    }

    public RefCountedOutputStream(OutputStream other) {
        if (other instanceof RefCountedOutputStream refCounted) {
            this.data = refCounted.data;
            this.data.refs.incrementAndGet();
        } else {
            this.data = new SharedData(other);
        }
    }



    @Override
    public void flush() throws IOException {
        if (closed.get()) {
            throw new IOException("Stream closed");
        }

        this.data.underlying.flush();
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (closed.get()) {
            throw new IOException("Stream closed");
        }

        this.data.underlying.write(b, off, len);
    }

    @Override
    public void write(int b) throws IOException {
        this.write(new byte[] { (byte) b });
    }

    @Override
    public void close() throws IOException {
        if (closed.getAndSet(true)) {
            return;
        }

        var endRefs = this.data.refs.decrementAndGet();
        if (endRefs <= 0) {
            log.trace("Real close in ref counted input stream for {}", this.data.underlying);
            this.data.underlying.close();
        }
    }

    @Override
    public String toString() {
        return "RefCountedOutputStream [refCount="+this.data.refs.get()+", underlying="+this.data.underlying+"]";
    }
}
