package org.clazzes.svc.runner.sshd;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.lang.reflect.Array;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import org.clazzes.svc.api.cmd.Argument;
import org.clazzes.svc.api.cmd.CommandSet;
import org.clazzes.svc.api.cmd.Descriptor;
import org.clazzes.svc.api.cmd.Parameter;

// This resolves java native commands like svc:ls and others.
public class GogoCommandResolver implements CommandResolver {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(GogoCommandResolver.class);

    private Supplier<Map<String, CommandSet>> getCommandSets;
    public void setGetCommandSets(Supplier<Map<String, CommandSet>> getCommandSets) {
        this.getCommandSets = getCommandSets;
    }

    private final Function<String, Object> mkConverter(Class<?> clazz) {
        if (clazz == String.class) {
            return s -> s;
        }
        else if (clazz == int.class || clazz == Integer.class) {
            return Integer::valueOf;
        }
        else if (clazz == long.class || clazz == Long.class) {
            return Long::valueOf;
        }
        else if (clazz == double.class || clazz == Double.class) {
            return Double::valueOf;
        }
        else if (clazz == boolean.class || clazz == Boolean.class) {
            return Boolean::valueOf;
        }
        else if (clazz == URI.class) {
            return URI::create;
        }
        else if (clazz == UUID.class) {
            return UUID::fromString;
        }
        else if (clazz == Instant.class) {
            return Instant::parse;
        }
        else if (clazz == ZonedDateTime.class) {
            return ZonedDateTime::parse;
        }
        else if (clazz == OffsetDateTime.class) {
            return OffsetDateTime::parse;
        }
        else if (clazz == LocalDateTime.class) {
            return LocalDateTime::parse;
        }
        else if (clazz == LocalDate.class) {
            return LocalDate::parse;
        }
        else if (clazz == LocalTime.class) {
            return LocalTime::parse;
        }
        else if (clazz == ZoneId.class) {
            return ZoneId::of;
        }
        else if (clazz == Locale.class) {
            return Locale::forLanguageTag;
        }
        else {
            throw new IllegalArgumentException("Cannot coerse string to type ["+clazz+"]");
        }
    }

    private final CliCommand mkCommand(CommandSet commandSet, Method method) {
        var options = new Options();

        var helpText = Objects.requireNonNull(
            method.getAnnotation(Descriptor.class),
            "Command did not specify a help text."
        ).value();

        List<BiFunction<CommandLine, CommandEnvironment, Object>> parameterGetters = new ArrayList<>();

        int positionalIndex = 0;

        StringBuilder footer = new StringBuilder();

        for (var parameterType: method.getParameters()) {
            if (parameterType.getType() == PrintStream.class) {
                parameterGetters.add((cli, env) -> new PrintStream(new CloseProtectedOutputStream(env.stdout()), false, StandardCharsets.UTF_8));
                continue;
            }

            var paramHelpText = Objects.requireNonNull(
                parameterType.getAnnotation(Descriptor.class),
                "Parameter did not specify a help text."
            );


            var parameter = parameterType.getAnnotation(Parameter.class);
            if (parameter == null) {

                Argument arg = parameterType.getAnnotation(Argument.class);

                if (positionalIndex < 0) {
                    throw new RuntimeException("Positional argument after rest argument.");
                }

                if (footer.isEmpty()) {
                    footer.append("Positional arguments:\n");
                }

                if (parameterType.getType().isArray()) {
                    int index = positionalIndex;
                    positionalIndex = -1;

                    String name = arg == null ? "opt" : arg.value();

                    footer.append(String.format(Locale.ENGLISH," %-16s %s\n",name+"...",paramHelpText.value()));

                    var converter = mkConverter(parameterType.getType().getComponentType());

                    parameterGetters.add((cli, env) -> {
                        int remainingArgsSize = cli.getArgs().length - index;
                        var ret = Array.newInstance(
                            parameterType.getType().getComponentType(),
                            remainingArgsSize
                        );

                        for (int i = 0; i < remainingArgsSize; i++) {
                            Array.set(ret, i, converter.apply(cli.getArgs()[index + i]));
                        }

                        return ret;
                    });

                } else {
                    // Positional.
                    int index = positionalIndex++;

                    String name = arg == null ? "arg"+positionalIndex : arg.value();

                    footer.append(String.format(Locale.ENGLISH," %-16s %s\n",name,paramHelpText.value()));

                    var converter = mkConverter(parameterType.getType());

                    parameterGetters.add((cli, env) -> {
                        if (index >= cli.getArgs().length) {
                            throw new IllegalArgumentException("Missing argument ["+name+"].");
                        }
                        return converter.apply(cli.getArgs()[index]);
                    });
                }

                continue;
            }

            var converter = mkConverter(parameterType.getType());

            var longNames = Arrays.stream(parameter.names())
                .filter(name -> name.startsWith("--"))
                .map(name -> name.substring(2))
                .toList();

            var shortNames = Arrays.stream(parameter.names())
                .filter(name -> name.startsWith("-") && !name.startsWith("--"))
                .map(name -> name.substring(1))
                .toList();

            boolean isBoolean = !Parameter.UNSPECIFIED.equals(parameter.presentValue());

            var group = new OptionGroup();
            for (int i = 0; i < Math.max(longNames.size(), shortNames.size()); i++) {
                var builder = Option.builder()
                    .desc(paramHelpText.value())
                    .hasArg(!isBoolean);

                if (i < shortNames.size()) {
                    builder = builder.option(shortNames.get(i));
                }

                if (i < longNames.size()) {
                    builder = builder.longOpt(longNames.get(i));
                }

                group.addOption(builder.build());
            }

            options.addOptionGroup(group);

            parameterGetters.add((cli, env) -> {
                return converter.apply(
                    cli.hasOption(group)
                        ? isBoolean
                            ? parameter.presentValue()
                            : cli.getOptionValue(group)
                        : parameter.absentValue()
                );
            });
        }

        options.addOption("h", "help", false, "display this help and exit");

        var parser = new DefaultParser();
        var formatter = new HelpFormatter();

        return params -> {
            var writer = new PrintWriter(params.stdout(), false, StandardCharsets.UTF_8);
            try {
                Runnable printHelp = () -> {

                    String syntax = params.arguments().get(0);

                    formatter.printHelp(
                        writer,
                        formatter.getWidth(),
                        syntax,
                        helpText,
                        options,
                        formatter.getLeftPadding(),
                        formatter.getDescPadding(),
                        footer.toString(),
                        true
                    );
                };

                final CommandLine cli;
                try {
                    cli = parser.parse(options, params.arguments().subList(1, params.arguments().size()).toArray(String[]::new));
                } catch (ParseException e) {
                    writer.println(e.getMessage());
                    printHelp.run();
                    return 1;
                }
                if (cli.hasOption("h")) {
                    printHelp.run();
                    return 0;
                } else {
                    try {
                        method.invoke(commandSet, parameterGetters.stream().map(fn -> fn.apply(cli, params.env())).toArray());
                        return 0;
                    } catch (Exception e) {
                        e.printStackTrace(writer);
                        return 1;
                    }
                }
            } finally {
                writer.flush();
            }
        };
    }

    private final CliCommand mkCommand(CommandSet commandSet, String command) {
        return mkCommand(commandSet, getClassMethod(commandSet, command));
    }

    private final CliCommand searchCommandSet(CommandSet commandSet, String commandName, CommandEnvironment env) {
        if (commandSet.getCommands().contains(commandName)) {
            try {
                return mkCommand(commandSet, commandName);
            } catch (Exception e) {
                env.printException(e);
                return null;
            }
        }
        return null;

    }

    @Override
    public CliCommand resolveCommand(String commandName, CommandEnvironment env) {
        var commandSets = getCommandSets.get();

        var split = commandName.split(":", 2);
        if (split.length < 2) {
            for (var commandSet: commandSets.values()) {
                var command = searchCommandSet(commandSet, commandName, env);
                if (command != null) {
                    return command;
                }
            }

            return null;
        } else {
            var commandSet = commandSets.get(split[0]);
            if (commandSet == null) {
                return null;
            }

            return searchCommandSet(commandSet, split[1], env);
        }
    }

    private static final String helpText(CommandSet commandSet, String commandName) {
        try {
            var method = getClassMethod(commandSet, commandName);

            return Objects.requireNonNull(
                method.getAnnotation(Descriptor.class),
                "Command did not specify a help text."
            ).value();
        } catch (Exception e) {
            log.error("Error while trying to get help text", e);
            return "Error while trying to get help text";
        }
    }

    private static Method getClassMethod(CommandSet commandSet, String commandName) {
        return Arrays.stream(commandSet.getClass().getMethods())
            .filter(method2 -> method2.getName().equals(commandName))
            .findFirst()
            .orElseThrow(() -> new IllegalStateException("The CommandSet listed a command that it doesn't have a function for."));
    }

    @Override
    public List<CommandInfo> listCommands(String commandNamePrefix, CommandEnvironment env) {
        var commandSets = getCommandSets.get();

        var split = commandNamePrefix.split(":", 2);

        if (split.length < 2) {
            return Stream.concat(
                commandSets.entrySet()
                    .stream()
                    .filter(entry -> entry.getKey().startsWith(commandNamePrefix))
                    .flatMap(
                        entry -> entry
                            .getValue()
                            .getCommands()
                            .stream()
                            .map(name -> new CommandInfo(entry.getKey() + ":" + name, helpText(entry.getValue(), name)))
                    ),
                commandSets.values()
                    .stream()
                    .flatMap(
                        commandSet -> commandSet
                            .getCommands()
                            .stream()
                            .filter(name -> name.startsWith(commandNamePrefix))
                            .map(name -> new CommandInfo(name, helpText(commandSet, name)))
                    )
            ).toList();
        } else {
            var commandSet = commandSets.get(split[0]);
            if (commandSet == null) {
                return List.of();
            }

            return commandSet.getCommands()
                .stream()
                .filter(name -> name.startsWith(split[1]))
                .map(name -> new CommandInfo(split[0] + ":" + name, helpText(commandSet, name)))
                .toList();
        }
    }
}
