package org.clazzes.svc.runner.sshd;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.IntUnaryOperator;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.clazzes.parsercombinators.Parser;
import org.clazzes.parsercombinators.ParserUtils;
import org.clazzes.parsercombinators.string.StringParseInput;
import org.clazzes.parsercombinators.string.StringParser;

public class CommandParser {
    public record Span(int start, int end) {}

    public static enum SyncMode {
        // Declared with &
        ASYNC,
        // Declared with ;
        SYNC,
        ;

    }

    public static enum AndOrMode {
        // Declared with &&
        AND,
        // Declared with ||
        OR,
        ;
    }

    public sealed interface DoubleQuotedPart {
        public final record QuotedLiteral(String str, Span span) implements DoubleQuotedPart {}
        public final record CommandSubstitution(ParsedCommand command) implements DoubleQuotedPart {}
    }

    public sealed interface WordPart {
        public final record Literal(String str) implements WordPart {}
        public final record DoubleQuoted(
            List<DoubleQuotedPart> parts,
            Span openQuoteSpan,
            Span closeQuoteSpan
        ) implements WordPart {}
        public final record CommandSubstitution(ParsedCommand command) implements WordPart {}
    }

    public final record Word(List<WordPart> parts) {}

    public static enum DuplicateKind {
        INPUT,
        OUTPUT,
        BOTH,
        ;
    }

    public static enum FileRedirectionKind {
        INPUT,
        OUTPUT,
        OUTPUT_FORCE_CLOBBER,
        APPEND,
        ;
    }

    public sealed interface RedirectionKind {
        public final record File(Word word, FileRedirectionKind kind) implements RedirectionKind {}
        public final record Duplicate(Word duplicatedFd, DuplicateKind kind) implements RedirectionKind {}
        public final record HereDocument(String document) implements RedirectionKind {}
    }

    private sealed interface PrefixElement {}

    private sealed interface SuffixElement {}

    public final record Redirection(RedirectionKind kind, int fd) implements PrefixElement, SuffixElement {}
    private final record CmdWord(Word word) implements SuffixElement {}
    public final record AssignmentWord(String key, Word value) implements PrefixElement {}
    public sealed interface Command {
        public final record Simple(
            List<Redirection> redirections,
            List<AssignmentWord> assignments,
            Word mainWord,
            Span mainWordSpan,
            List<Word> otherWords
        ) implements Command {}
        public final record Compound() implements Command {}
        public final record FunctionDefinition() implements Command {}
    }
    public final record Pipeline(boolean hasBang, List<Command> commands) {}
    public final record AndOrElement(AndOrMode mode, Pipeline pipeline) {}
    public final record AndOr(Pipeline firstPipeline, List<AndOrElement> elements) {}
    public final record PosixListElement(SyncMode mode, AndOr andOr) {}
    public final record ParsedCommand(List<PosixListElement> list) {}

    private static final <T, R> List<R> mapList(List<T> list, Function<T, R> fn) {
        var ret = new ArrayList<R>(list.size());
        for (var i: list) {
            ret.add(fn.apply(i));
        }
        return ret;
    }

    private static final Span mapSpan(Span span, IntUnaryOperator fn) {
        return new Span(
            fn.applyAsInt(span.start()),
            fn.applyAsInt(span.end())
        );
    }

    private static DoubleQuotedPart mapDoubleQuotedPartSpan(DoubleQuotedPart dqPart, IntUnaryOperator fn) {
        if (dqPart instanceof DoubleQuotedPart.QuotedLiteral lit) {
            return new DoubleQuotedPart.QuotedLiteral(
                lit.str(),
                mapSpan(lit.span(), fn)
            );
        } else if (dqPart instanceof DoubleQuotedPart.CommandSubstitution sub) {
            return new DoubleQuotedPart.CommandSubstitution(
                mapSpans(sub.command(), fn)
            );

        } else {
            throw new RuntimeException("unsupported");
        }
    }

    private static WordPart mapWordPartSpans(WordPart part, IntUnaryOperator fn) {
        if (part instanceof WordPart.CommandSubstitution sub) {
            return new WordPart.CommandSubstitution(
                mapSpans(sub.command(), fn)
            );
        } else if (part instanceof WordPart.DoubleQuoted doubleQuoted) {
            return new WordPart.DoubleQuoted(
                mapList(
                    doubleQuoted.parts(),
                    dqPart -> mapDoubleQuotedPartSpan(dqPart, fn)
                ),
                mapSpan(doubleQuoted.openQuoteSpan(), fn),
                mapSpan(doubleQuoted.closeQuoteSpan(), fn)
            );
        } else if (part instanceof WordPart.Literal lit) {
            return lit;
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private static final Word mapWordSpans(Word word, IntUnaryOperator fn) {
        return new Word(
            mapList(
                word.parts(),
                part -> mapWordPartSpans(part, fn)
            )
        );
    }

    private static RedirectionKind mapRedirectionKindSpans(RedirectionKind kind, IntUnaryOperator fn) {
        if (kind instanceof RedirectionKind.Duplicate dup) {
            return new RedirectionKind.Duplicate(
                mapWordSpans(dup.duplicatedFd(), fn),
                dup.kind()
            );
        } else if (kind instanceof RedirectionKind.File file) {
            return new RedirectionKind.File(
                mapWordSpans(file.word(), fn),
                file.kind()
            );
        } else if (kind instanceof RedirectionKind.HereDocument hereDoc) {
            return hereDoc;
        } else {
            throw new RuntimeException("unsupported");
        }
    }

    private static final Command mapCommandSpans(Command command, IntUnaryOperator fn) {
        if (command instanceof Command.Simple simple) {
            return new Command.Simple(
                mapList(
                    simple.redirections(),
                    redirection -> new Redirection(
                        mapRedirectionKindSpans(redirection.kind(), fn),
                        redirection.fd()
                    )
                ),
                mapList(
                    simple.assignments(),
                    assignment -> new AssignmentWord(
                        assignment.key(),
                        mapWordSpans(assignment.value(), fn)
                    )
                ),
                mapWordSpans(simple.mainWord(), fn),
                mapSpan(simple.mainWordSpan(), fn),
                mapList(
                    simple.otherWords(),
                    otherWord -> mapWordSpans(otherWord, fn)
                )
            );
        } else {
            throw new RuntimeException("Unsupported kind of command");
        }
    }

    private static final Pipeline mapPipelineSpans(Pipeline input, IntUnaryOperator fn) {
        return new Pipeline(
            input.hasBang(),
            mapList(
                input.commands(),
                command -> mapCommandSpans(command, fn)
            )
        );
    }

    private static final ParsedCommand mapSpans(ParsedCommand input, IntUnaryOperator fn) {
        return new ParsedCommand(
            mapList(
                input.list(),
                listElement -> new PosixListElement(
                    listElement.mode(),
                    new AndOr(
                        mapPipelineSpans(
                            listElement.andOr().firstPipeline(),
                            fn
                        ),
                        mapList(
                            listElement.andOr().elements(),
                            andOrElement -> new AndOrElement(
                                andOrElement.mode(),
                                mapPipelineSpans(andOrElement.pipeline(), fn)
                            )
                        )
                    )
                )
            )
        );
    }

    private static final Parser<CharSequence, Integer> pos = input_ -> {
        var seq = StringParseInput.fromSequence(input_);
        return ParserUtils.ok(seq, seq.getBeginIndex());
    };

    private static final <T, R> Parser<CharSequence, R> withSpan(BiFunction<T, Span, R> fn, Parser<CharSequence, T> p) {
        return ParserUtils.and(
            (start, value, end) -> fn.apply(
                value,
                new Span(start, end)
            ),
            pos,
            p,
            pos
        );
    }

    // Posix documentation https://pubs.opengroup.org/onlinepubs/9699919799/utilities/V3_chap02.html
    private static final Parser<CharSequence, ParsedCommand> mkParser() {
        var commandParserRet = ParserUtils.<CharSequence, ParsedCommand>fixParser(commandParser -> {

        var optionalWhitespace = StringParser.regex("[ \t]*").map(m -> (Void) null)
            .named("optionalWhitespace");
        var mandatoryWhitespace = StringParser.regex("[ \t]+").map(m -> (Void) null)
            .named("mandatoryWhitespace");
        var optionalLinebreak = StringParser.regex("[ \t\r\n]*").map(m -> (Void) null)
            .named("optionalLinebreak");
        var mandatoryLinebreak = StringParser.regex("[ \t\r]*\n[ \t\r\n]*").map(m -> (Void) null)
            .named("mandatoryLinebreak");

        var and_if = StringParser.matchesString("&&");
        var or_if = StringParser.matchesString("||");


        var singleQuotedString = StringParser.regex("'([^']*)'")
            .map(match -> match.group(1))
            .named("singleQuotedString");

        var backslashEscape = StringParser.regex("\\\\(.)")
            .map(m -> m.group(1))
            .named("backslashEscape");

        var backTickedUnEscapePattern = Pattern.compile("\\\\(.)", Pattern.DOTALL);

        Function<Boolean, Parser<CharSequence, ParsedCommand>> backTicked = quoted -> ParserUtils.and(
            (a, commandStartPos, command, b) -> {
                var unEscaped = replaceAllWithPositions(
                    command,
                    backTickedUnEscapePattern,
                    m -> {
                        if ("$`\\".contains(m.group(1))) {
                            return m.group(1);
                        } else if ("\"".equals(m.group(1)) && quoted) {
                            return "\"";
                        } else {
                            return m.group();
                        }
                    }
                );
                System.out.println("unEscaped = " + unEscaped);
                // Context-sensitive recusion.
                // If this fails the entire parse fails.
                return mapSpans(
                    parseCommand(unEscaped.getKey()),
                    posToMap -> commandStartPos + unEscaped.getValue().applyAsInt(posToMap)
                );
            },
            StringParser.single('`'),
            pos,
            StringParser.regex("(?:(?:\\\\.)|[^`])*", Pattern.DOTALL)
                .map(m -> m.group()),
            StringParser.single('`')
        ).named("backTicked");

        var commandSubstitution = ParserUtils.delimited(
            StringParser.single('('),
            commandParser,
            StringParser.single(')')
        ).named("commandSubstitution");

        var doubleQuotedLiteral = ParserUtils.some(
            ParserUtils.or(
                backslashEscape,
                StringParser.regex("[^\\\\$`\"]+")
                    .map(m -> m.group())
            )
        )
            .map(list -> list.stream().collect(Collectors.joining("")))
            .named("doubleQuotedLiteral");

        var normalLiteral = ParserUtils.some(
            ParserUtils.or(
                backslashEscape,
                StringParser.regex("[^"+Pattern.quote("[{}|&;<>()\\ '\t\n\r\u00A0\\\"$`?*@!+\u201C\u201D\u2033\u2036\u2018\u2019")+"]+")
                    .map(m -> m.group())
            )
        )
            .map(list -> list.stream().collect(Collectors.joining("")))
            .named("normalLiteral");

        var doubleQuotedPart = ParserUtils.or(
            withSpan(
                DoubleQuotedPart.QuotedLiteral::new,
                doubleQuotedLiteral
            ),
            backTicked.apply(true).<DoubleQuotedPart>map(DoubleQuotedPart.CommandSubstitution::new),
            ParserUtils.discardingFirst(
                StringParser.single('$'),
                ParserUtils.or(
                    commandSubstitution.<DoubleQuotedPart>map(DoubleQuotedPart.CommandSubstitution::new),
                    pos.map(posValue -> new DoubleQuotedPart.QuotedLiteral("$", new Span(posValue - 1, posValue)))
                )
            )

        ).named("doubleQuotedPart");

        var doubleQuotedString = ParserUtils.and(
            (openSpan, parts, closeSpan) -> (WordPart) new WordPart.DoubleQuoted(parts, openSpan, closeSpan),
            withSpan(
                (a, b) -> b,
                StringParser.single('"')
            ),
            ParserUtils.many(doubleQuotedPart),
            withSpan(
                (a, b) -> b,
                StringParser.single('"')
            )
        ).named("doubleQuotedString");

        var wordPart = ParserUtils.or(
            singleQuotedString.<WordPart>map(WordPart.Literal::new),
            doubleQuotedString,
            backTicked.apply(false).<WordPart>map(WordPart.CommandSubstitution::new),
            ParserUtils.discardingFirst(
                StringParser.single('$'),
                ParserUtils.or(
                    commandSubstitution.<WordPart>map(WordPart.CommandSubstitution::new),
                    ParserUtils.pure(new WordPart.Literal("$"))
                )
            ),
            normalLiteral.<WordPart>map(WordPart.Literal::new)
        ).named("wordPart");

        var word = ParserUtils.some(wordPart)
            .map(Word::new)
            .named("word");

        var cmdWord = word.map(CmdWord::new);

        var ioRedirect = ParserUtils.and(
            (fd, kind) -> new Redirection(
                kind,
                fd.orElseGet(
                    () -> kind instanceof RedirectionKind.Duplicate dup
                        ? dup.kind() == DuplicateKind.OUTPUT
                            ? /* stdout */ 1
                            : /* stdin */ 0
                        : kind instanceof RedirectionKind.File file
                        ? file.kind() == FileRedirectionKind.INPUT
                            ? /* stdin */ 0
                            : /* stdout */ 1
                        : /* stdin */ 0
                )
            ),
            ParserUtils.optional(
                StringParser.regex("[0-9]{1,9}")
                    .map(m -> m.group())
                    .map(Integer::valueOf)
                .named("fd")
            ),
            ParserUtils.<CharSequence, RedirectionKind>or(
                ParserUtils.and(
                    (kind, wordValue) -> (RedirectionKind) new RedirectionKind.Duplicate(wordValue, kind),
                    ParserUtils.<CharSequence, DuplicateKind>or(
                        StringParser.matchesString("<>").map(v -> DuplicateKind.BOTH),
                        StringParser.matchesString(">&").map(v -> DuplicateKind.OUTPUT),
                        StringParser.matchesString("<&").map(v -> DuplicateKind.INPUT)
                    ),
                    ParserUtils.discardingFirst(
                        optionalWhitespace,
                        word
                    )
                ),
                ParserUtils.and(
                    (kind, wordValue) -> (RedirectionKind) new RedirectionKind.File(wordValue, kind),
                    ParserUtils.<CharSequence, FileRedirectionKind>or(
                        StringParser.matchesString(">|").map(v -> FileRedirectionKind.OUTPUT_FORCE_CLOBBER),
                        StringParser.matchesString(">>").map(v -> FileRedirectionKind.APPEND),
                        StringParser.matchesString(">").map(v -> FileRedirectionKind.OUTPUT),
                        StringParser.matchesString("<").map(v -> FileRedirectionKind.INPUT)
                    ),
                    ParserUtils.discardingFirst(
                        optionalWhitespace,
                        word
                    )
                )
            )
        ).named("redirection");

        var variableName = StringParser.regex("[\\p{Lu}\\p{Ll}_][0-9\\p{Lu}\\p{Ll}_]*")
            .map(m -> m.group())
            .named("variableName");

        var assignmentWord = ParserUtils.and(
            AssignmentWord::new,
            variableName,
            ParserUtils.discardingFirst(
                StringParser.single('='),
                word
            )
        ).named("assignmentWord");

        var cmdPrefix = ParserUtils.many(
            ParserUtils.discardingSecond(
                ParserUtils.or(
                    ioRedirect.map(v -> (PrefixElement) v),
                    assignmentWord.map(v -> (PrefixElement) v)
                ),
                mandatoryWhitespace
            )
        ).named("cmdPrefix");

        var cmdSuffix = ParserUtils.many(
            ParserUtils.discardingFirst(
                mandatoryWhitespace,
                ParserUtils.or(
                    ioRedirect.map(v -> (SuffixElement) v),
                    cmdWord.map(v -> (SuffixElement) v)
                )
            )
        ).named("cmdSuffix");


        var simpleCommand = ParserUtils.and(
            (prefixValue, startWord, wordValue, endWord, suffixValue) -> new Command.Simple(
                Stream.concat(
                    prefixValue
                        .stream()
                        .flatMap(
                            element -> element instanceof Redirection ret
                                ? Stream.of(ret)
                                : Stream.empty()
                        ),
                    suffixValue
                        .stream()
                        .flatMap(
                            element -> element instanceof Redirection ret
                                ? Stream.of(ret)
                                : Stream.empty()
                        )
                ).toList(),
                prefixValue
                    .stream()
                    .flatMap(
                        element -> element instanceof AssignmentWord ret
                            ? Stream.of(ret)
                            : Stream.empty()
                    )
                    .toList(),
                wordValue.word(),
                new Span(startWord, endWord),
                suffixValue
                    .stream()
                    .flatMap(
                        element -> element instanceof CmdWord ret
                            ? Stream.of(ret.word())
                            : Stream.empty()
                    )
                    .toList()
            ),
            cmdPrefix,
            pos,
            cmdWord,
            pos,
            cmdSuffix
        ).named("simpleCommand");

        var command = ParserUtils.or(
            simpleCommand.map(v -> (Command) v)
        ).named("command");

        var pipeline = ParserUtils.and(
            Pipeline::new,
            ParserUtils.optional(
                ParserUtils.discardingSecond(
                    StringParser.single('!'),
                    mandatoryWhitespace
                )
            )
                .map(o -> o.isPresent()),
            ParserUtils.someWithSeparator(
                command,
                ParserUtils.delimited(
                    optionalWhitespace,
                    StringParser.single('|'),
                    optionalLinebreak
                )
            )
        ).named("pipeline");

        var andOrElement = ParserUtils.and(
            AndOrElement::new,
            ParserUtils.delimited(
                optionalWhitespace,
                ParserUtils.or(
                    and_if.map(s -> AndOrMode.AND),
                    or_if.map(s -> AndOrMode.OR)
                ),
                optionalLinebreak
            ),
            pipeline
        ).named("andOrElement");

        var andOr = ParserUtils.and(
            AndOr::new,
            pipeline,
            ParserUtils.many(andOrElement)
        ).named("andOr");

        var separatorOp = ParserUtils.or(
            ParserUtils.delimited(
                optionalWhitespace,
                ParserUtils.or(
                    StringParser.single('&').map(c -> SyncMode.ASYNC),
                    StringParser.single(';').map(c -> SyncMode.SYNC)
                ),
                optionalWhitespace
            ),
            mandatoryLinebreak.map(c -> SyncMode.SYNC)
        ).named("separatorOp");

        var list = ParserUtils.<CharSequence, Stream<PosixListElement>>fixParser(
            restParser -> ParserUtils.optional(
                ParserUtils.and(
                    (element, restValue) -> Stream.concat(
                        Stream.of(
                            new PosixListElement(
                                restValue
                                    .map(Map.Entry::getKey)
                                    .orElse(SyncMode.SYNC),
                                element
                            )
                        ),
                        restValue
                            .stream()
                            .flatMap(Map.Entry::getValue)
                    ),
                    andOr,
                    ParserUtils.optional(
                        ParserUtils.and(
                            Map::entry,
                            separatorOp,
                            restParser
                        )
                    )
                )
            ).map(opt -> opt.orElseGet(Stream::empty))
        ).map(Stream::toList);

        return ParserUtils.delimited(
            optionalLinebreak,
            list,
            optionalLinebreak
        )
        .map(ParsedCommand::new)
        .named("commandParser");

        });

        return StringParser.withEof(commandParserRet);
    }

    private static final Parser<CharSequence, ParsedCommand> parser = mkParser();

    private static final Pattern lineContinuationsPattern = Pattern.compile("\\\\(\n|\\\\)");

    private static final Map.Entry<String, IntUnaryOperator> replaceAllWithPositions(String input, Pattern pattern, Function<MatchResult, String> replacer) {
        var substitutionsMap = new TreeMap<Integer, Integer>();
        var resultString = pattern.matcher(input)
            .replaceAll(m -> {
                var replacement = replacer.apply(m);

                var matchLength = m.end() - m.start();

                if (replacement.length() != matchLength) {
                    var previousPosShift =  Optional.ofNullable(substitutionsMap.lastEntry())
                        .map(Map.Entry::getValue)
                        .orElse(0);

                    var newPosShift = previousPosShift + (matchLength - replacement.length());

                    substitutionsMap.put(m.start() - previousPosShift, newPosShift);
                }

                return Matcher.quoteReplacement(replacement);
            });

        return Map.entry(
            resultString,
            pos -> pos + Optional.ofNullable(substitutionsMap.floorEntry(pos))
                .map(Map.Entry::getValue)
                .orElse(0)
        );
    }

    public static final ParsedCommand parseCommand(String commandString) {
        var strippedLineContinuations = replaceAllWithPositions(
            commandString,
            lineContinuationsPattern,
            m -> {
                if (m.group(1).charAt(0) == '\n') {
                    return "";
                } else {
                    return m.group();
                }
            }
        );

        return mapSpans(
            StringParser.parseFinal(parser, strippedLineContinuations.getKey()),
            strippedLineContinuations.getValue()
        );
    }
}
