package org.clazzes.svc.runner.sshd;

import java.util.Comparator;
import java.util.List;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import org.clazzes.parsercombinators.ParseException;
import org.clazzes.parsercombinators.string.StringParseInput;
import org.clazzes.svc.runner.sshd.CommandParser.AssignmentWord;
import org.clazzes.svc.runner.sshd.CommandParser.Command;
import org.clazzes.svc.runner.sshd.CommandParser.DoubleQuotedPart;
import org.clazzes.svc.runner.sshd.CommandParser.ParsedCommand;
import org.clazzes.svc.runner.sshd.CommandParser.Redirection;
import org.clazzes.svc.runner.sshd.CommandParser.RedirectionKind;
import org.clazzes.svc.runner.sshd.CommandParser.Span;
import org.clazzes.svc.runner.sshd.CommandParser.Word;
import org.clazzes.svc.runner.sshd.CommandParser.WordPart;
import org.clazzes.svc.runner.sshd.CommandParser.WordPart.DoubleQuoted;
import org.jline.reader.Highlighter;
import org.jline.reader.LineReader;
import org.jline.utils.AttributedString;
import org.jline.utils.AttributedStringBuilder;
import org.jline.utils.AttributedStyle;

public class CommandHighlighter implements Highlighter {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(CommandHighlighter.class);

    private final ShellExecutionEngine executionEngine;
    public CommandHighlighter(ShellExecutionEngine executionEngine) {
        this.executionEngine = executionEngine;
    }

    record HighlightSpan(Span span, AttributedStyle style) {}

    private AttributedStyle resolvedCommandStyle = AttributedStyle.DEFAULT.foreground(AttributedStyle.GREEN);
    private AttributedStyle doubleQuotedStyle = AttributedStyle.DEFAULT.foreground(AttributedStyle.YELLOW);
    private AttributedStyle unresolvedCommandStyle = AttributedStyle.DEFAULT.foreground(AttributedStyle.RED + AttributedStyle.BRIGHT).bold();
    private AttributedStyle parseErrorStyle = AttributedStyle.DEFAULT.foreground(AttributedStyle.RED + AttributedStyle.BRIGHT).bold();

    private Stream<HighlightSpan> extractDoubleQuotedPartHighlights(DoubleQuotedPart part) {
        if (part instanceof DoubleQuotedPart.QuotedLiteral lit) {
            return Stream.of(new HighlightSpan(lit.span(), doubleQuotedStyle));
        } else if (part instanceof DoubleQuotedPart.CommandSubstitution sub) {
            return extractHighlights(sub.command());
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private Stream<HighlightSpan> extractWordPartHighlights(WordPart part) {
        if (part instanceof WordPart.DoubleQuoted doubleQuoted) {
            return Stream.concat(
                Stream.of(
                    new HighlightSpan(doubleQuoted.openQuoteSpan(), doubleQuotedStyle),
                    new HighlightSpan(doubleQuoted.closeQuoteSpan(), doubleQuotedStyle)
                ),
                doubleQuoted.parts()
                    .stream()
                    .flatMap(this::extractDoubleQuotedPartHighlights)
            );
        } else if (part instanceof WordPart.Literal) {
            return Stream.empty();
        } else if (part instanceof WordPart.CommandSubstitution sub) {
            return extractHighlights(sub.command());
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private Stream<HighlightSpan> extractWordHighlights(Word word) {
        return word
            .parts()
            .stream()
            .flatMap(this::extractWordPartHighlights);
    }

    private Stream<HighlightSpan> extractRedirectionKindHighlights(RedirectionKind kind) {
        if (kind instanceof RedirectionKind.Duplicate dup) {
            return extractWordHighlights(dup.duplicatedFd());
        } else if (kind instanceof RedirectionKind.File file) {
            return extractWordHighlights(file.word());
        } else {
            return Stream.empty();
        }
    }

    private Stream<HighlightSpan> extractCommandHighlights(Command command) {
        if (command instanceof Command.Simple simple) {
            var mainWordHighlights = this.executionEngine.expandWordWithoutSideEffects(simple.mainWord())
                .map(expanded -> executionEngine.getResolver().resolveCommand(expanded, executionEngine.getEnv()) != null
                    ? resolvedCommandStyle
                    : unresolvedCommandStyle)
                .map(style -> Stream.of(new HighlightSpan(simple.mainWordSpan(), style)))
                .orElseGet(() -> extractWordHighlights(simple.mainWord()));

            return Stream.of(
                mainWordHighlights,
                simple
                    .assignments()
                    .stream()
                    .map(AssignmentWord::value)
                    .flatMap(this::extractWordHighlights),
                simple
                    .otherWords()
                    .stream()
                    .flatMap(this::extractWordHighlights),
                simple
                    .redirections()
                    .stream()
                    .map(Redirection::kind)
                    .flatMap(this::extractRedirectionKindHighlights)

            ).flatMap(Function.identity());
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private Stream<HighlightSpan> extractHighlights(ParsedCommand script) {
        return script
            .list()
            .stream()
            .map(listElement -> listElement.andOr())
            .flatMap(
                andOr -> Stream.concat(
                    Stream.of(andOr.firstPipeline()),
                    andOr
                        .elements()
                        .stream()
                        .map(andOrElement -> andOrElement.pipeline())
                )
            )
            .flatMap(pipeline -> pipeline.commands().stream())
            .flatMap(this::extractCommandHighlights)
            ;
    }

    private static final Comparator<Span> NON_OVERLAPPING = (a, b) -> {
        if (a.start() == b.start() && b.end() == a.end()) {
            return 0;
        }

        if (a.start() < b.start()) {
            if (a.end() > b.start()) {
                log.warn("a = {}", a);
                log.warn("b = {}", b);
                throw new RuntimeException("Overlapping highlight ranges.");
            }

            return -1;
        } else if (a.start() > b.start()) {
            if (b.end() > a.start()) {
                log.warn("a = {}", a);
                log.warn("b = {}", b);
                throw new RuntimeException("Overlapping highlight ranges.");
            }

            return 1;
        } else {
            log.warn("a = {}", a);
            log.warn("b = {}", b);
            throw new RuntimeException("Overlapping highlight ranges.");
        }
    };

    private AttributedString attribute(List<HighlightSpan> spans, String str) {
        var builder = new AttributedStringBuilder();
        int prevIndex = 0;

        for (var span: spans) {
            builder.append(new StringParseInput(str, prevIndex, span.span().start()));
            builder.append(new StringParseInput(str, span.span().start(), span.span().end()), span.style());
            prevIndex = span.span().end();
        }
        builder.append(new StringParseInput(str, prevIndex, str.length()));
        return builder.toAttributedString();
    }

    @Override
    public AttributedString highlight(LineReader reader, String buffer) {
        ParsedCommand command;
        try {
            command = CommandParser.parseCommand(buffer);
        } catch (ParseException e) {
            return new AttributedString(buffer, parseErrorStyle);
        }

        var highlights = extractHighlights(command)
            .sorted(Comparator.comparing(HighlightSpan::span, NON_OVERLAPPING))
            .toList();

        return attribute(highlights, buffer);


    }

    @Override
    public void setErrorPattern(Pattern errorPattern) {
        // IGNORE
    }

    @Override
    public void setErrorIndex(int errorIndex) {
        // IGNORE
    }
}
