/***********************************************************
 *
 * GOGO SSH support of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.sshd;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * An output stream replacing CR by CR-LF sequences.
 */
public class CRLFOutputStream extends FilterOutputStream {

    private int lastByte;
    private static final byte[] CRLF= new byte[] {'\r','\n'};

    public CRLFOutputStream(OutputStream out) {
        super(out);
        this.lastByte = -1;
    }

    /* (non-Javadoc)
     * @see java.io.FilterOutputStream#write(int)
     */
    @Override
    public void write(int b) throws IOException {

        if (b=='\n' && this.lastByte != '\r') {
            this.out.write(CRLF);
        }
        else {
            this.out.write(b);
        }

        this.lastByte = b;
    }



    public void write(byte b[], int off, int len) throws IOException {
        if ((off | len | (b.length - (len + off)) | (off + len)) < 0)
            throw new IndexOutOfBoundsException();

        int last = 0;

        for (int i = 0 ; i < len ; i++) {

            byte c = b[off + i];

            if (c=='\n' && this.lastByte != '\r') {
                if (i>last) {
                    this.out.write(b,off+last,i-last);
                }

                this.out.write(CRLF);
                last = i+1;
            }

            this.lastByte = c & 0xff;
        }

        if (len>last) {
            this.out.write(b,off+last,len-last);
        }
    }


}
