package org.clazzes.svc.runner.sshd;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import org.clazzes.parsercombinators.ParseException;
import org.clazzes.svc.runner.sshd.CommandParser.Command;
import org.clazzes.svc.runner.sshd.CommandParser.DoubleQuotedPart;
import org.clazzes.svc.runner.sshd.CommandParser.ParsedCommand;
import org.clazzes.svc.runner.sshd.CommandParser.Pipeline;
import org.clazzes.svc.runner.sshd.CommandParser.RedirectionKind;
import org.clazzes.svc.runner.sshd.CommandParser.Span;
import org.clazzes.svc.runner.sshd.CommandParser.Word;
import org.clazzes.svc.runner.sshd.CommandParser.WordPart;
import org.jline.reader.Candidate;
import org.jline.reader.Completer;
import org.jline.reader.CompletingParsedLine;
import org.jline.reader.LineReader;
import org.jline.reader.ParsedLine;
import org.jline.reader.Parser;
import org.jline.reader.SyntaxError;

public class CommandCompleter implements Completer, Parser {
    private static final <R, T> Optional<R> firstSome(List<T> list, Function<T, Optional<R>> fn) {
        for (var i: list) {
            var value = fn.apply(i);
            if (value.isPresent()) {
                return value;
            }
        }
        return Optional.empty();
    }

    record WordWithSpan(Word word, Span span) {}

    private static final Optional<WordWithSpan> extractDoubleQuotedPartMainWord(DoubleQuotedPart part, int cursor) {
        if (part instanceof DoubleQuotedPart.CommandSubstitution sub) {
            return extractMainWord(sub.command(), cursor);
        } else if (part instanceof DoubleQuotedPart.QuotedLiteral lit) {
            return Optional.empty();
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private static final Optional<WordWithSpan> extractWordPartMainWord(WordPart part, int cursor) {
        if (part instanceof WordPart.CommandSubstitution sub) {
            return extractMainWord(sub.command(), cursor);
        } else if (part instanceof WordPart.Literal lit) {
            return Optional.empty();
        } else if (part instanceof WordPart.DoubleQuoted doubleQuoted) {
            return firstSome(
                doubleQuoted.parts(),
                dqPart -> extractDoubleQuotedPartMainWord(dqPart, cursor)
            );
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private static final Optional<WordWithSpan> extractWordMainWord(Word word, int cursor) {
        return firstSome(
            word.parts(),
            wordPart -> extractWordPartMainWord(wordPart, cursor)
        );
    }

    private static final Optional<WordWithSpan> extractRedirectionKindMainWord(RedirectionKind kind, int cursor) {
        if (kind instanceof RedirectionKind.Duplicate dup) {
            return extractWordMainWord(dup.duplicatedFd(), cursor);
        } else if (kind instanceof RedirectionKind.File file) {
            return extractWordMainWord(file.word(), cursor);
        } else if (kind instanceof RedirectionKind.HereDocument) {
            return Optional.empty();
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private static final Optional<WordWithSpan> extractCommandMainWord(Command command, int cursor) {
        if (command instanceof Command.Simple simple) {
            return extractWordMainWord(simple.mainWord(), cursor)
                // Also match if the cursor is just after the main word, aka treat the span as inclusive on both sides.
                .or(() -> cursor >= simple.mainWordSpan().start() && cursor <= simple.mainWordSpan().end()
                    ? Optional.of(new WordWithSpan(simple.mainWord(), simple.mainWordSpan()))
                    : Optional.empty())

                .or(() -> firstSome(
                    simple.otherWords(),
                    otherWord -> extractWordMainWord(otherWord, cursor)
                ))
                .or(() -> firstSome(
                    simple.assignments(),
                    assignment -> extractWordMainWord(assignment.value(), cursor)
                ))
                .or(() -> firstSome(
                    simple.redirections(),
                    redirection -> extractRedirectionKindMainWord(redirection.kind(), cursor)
                ))
                ;
        } else {
            throw new RuntimeException("Unsupported");
        }
    }

    private static final Optional<WordWithSpan> extractPipelineMainWord(Pipeline pipeline, int cursor) {
        return firstSome(
            pipeline.commands(),
            command -> extractCommandMainWord(command, cursor)
        );
    }

    private static final Optional<WordWithSpan> extractMainWord(ParsedCommand command, int cursor) {
        return firstSome(
            command.list(),
            posixElement -> extractPipelineMainWord(posixElement.andOr().firstPipeline(), cursor)
                .or(() -> firstSome(
                    posixElement.andOr().elements(),
                    andOrElement -> extractPipelineMainWord(andOrElement.pipeline(), cursor)
                ))
        );
    }

    private final ShellExecutionEngine executionEngine;
    public CommandCompleter(ShellExecutionEngine executionEngine) {
        this.executionEngine = executionEngine;
    }

    @Override
    public void complete(LineReader reader, ParsedLine line, List<Candidate> candidates) {
        var commandString = line.line();
        var cursor = line.cursor();

        ParsedCommand command;
        try {
            command = CommandParser.parseCommand(commandString);
        } catch (ParseException e) {
            // Don't offer completion for invalid commands.
            return;
        }

        var mainWordOpt = extractMainWord(command, cursor);
        if (!mainWordOpt.isPresent()) {
            return;
        }

        var mainWordSpan = mainWordOpt.get().span();
        var mainWord = mainWordOpt.get().word();

        if (cursor != mainWordSpan.end()) {
            // TODO: Completions in the middle of a command.
            return;
        }

        var expanded = this.executionEngine.expandWordWithoutSideEffects(mainWord);

        if (!expanded.isPresent()) {
            return;
        }

        var prefix = expanded.get();

        var resolverCandidates = executionEngine.getResolver().listCommands(prefix, executionEngine.getEnv());
        for (var candidate: resolverCandidates) {
            candidates.add(new Candidate(
                candidate.name().substring(prefix.length()),
                candidate.name(),
                null,
                null,
                null,
                null,
                true,
                0
            ));
        }
    }

    @Override
    public ParsedLine parse(String line, int cursor, ParseContext context) throws SyntaxError {
        // TODO: Throw EOFError if we are able to accept a new line.

        var word = "";
        var words = List.of(line.substring(0, cursor), word, line.substring(cursor));
        return new CompletingParsedLine() {

            @Override
            public String word() {
                return word;
            }

            @Override
            public int wordCursor() {
                return 0;
            }

            @Override
            public int wordIndex() {
                return 1;
            }

            @Override
            public List<String> words() {
                return words;
            }

            @Override
            public String line() {
                return line;
            }

            @Override
            public int cursor() {
                return cursor;
            }

            @Override
            public CharSequence escape(CharSequence candidate, boolean complete) {
                return candidate;
            }

            @Override
            public int rawWordCursor() {
                return 0;
            }

            @Override
            public int rawWordLength() {
                return 0;
            }
        };
    }
}
