/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring.openmetrics;

import java.io.IOException;
import java.util.Map.Entry;
import java.util.SortedMap;

import org.clazzes.svc.runner.monitoring.IActiveMetrics;
import org.clazzes.svc.runner.monitoring.OpenMetricsType;
import org.clazzes.svc.runner.monitoring.Result;

public abstract class OpenMetricsFormatter {

    private static final String POS_INF = Double.toString(Double.POSITIVE_INFINITY);
    private static final String NEG_INF = Double.toString(Double.NEGATIVE_INFINITY);

    /**
     * Quote double quotes, line feeds and backslashes.
     * @param s The string to quote
     * @return The quoted string.
     */
    protected static final String quote(String s) {

        StringBuilder sb = null;
        int lasti = 0;

        for (int i=0; i<s.length();++i) {

            char c = s.charAt(i);
            if (c=='\n' || c =='\\' || c=='"') {
                if (sb == null) {
                    sb = new StringBuilder();
                }

                if (i>lasti) {
                    sb.append(s.substring(lasti,i));
                }

                lasti = i+1;

                if (c=='\n') {
                    sb.append("\n");
                }
                else {
                    sb.append('\\');
                    sb.append(c);
                }
            }
        }

        if (sb == null) {
            return s;
        }
        else {
            if (lasti < s.length()) {
                sb.append(s.substring(lasti));
            }

            return sb.toString();
        }
    }

    protected static final String formatNumber(Number n) {

        String s = n.toString();

        // identity comparison is OK here, because the
        // infinity constants are always the same as returned
        // by java-internal number formatters.
        if (n instanceof Double) {
            if (s == POS_INF) {
                return "+Inf";
            }
            if (s == NEG_INF) {
                return "-Inf";
            }
        }

        return s;
    }

    protected static void formatCounter(Appendable w, String id, IActiveMetrics<?> metrics) throws IOException {

        Result<?> r = metrics.getResult();

        double created = metrics.getCreatedMillis()*0.001;

        if (r == null) {
            w.append(id + "_created " + formatNumber(created)+"\n");
        }
        else {

            double timestamp = r.getEpochMillis()*0.001;

            w.append(id + "_total " +
                     formatNumber(r.getNumericResult())+" "+
                     formatNumber(timestamp)+"\n");

            w.append(id + "_created " +
                     formatNumber(created)+"\n");
        }
    }

    protected static void formatGauge(Appendable w, String id, IActiveMetrics<?> metrics) throws IOException {

        Result<?> r = metrics.getResult();

        if (r != null) {
            double timestamp = r.getEpochMillis()*0.001;

            w.append(id + " " +
                     formatNumber(r.getNumericResult())+" "+
                     formatNumber(timestamp)+"\n");
        }
    }

    protected static void formatSummary(Appendable w, String id, IActiveMetrics<?> metrics) throws IOException {

        SortedMap<Double,Number> sum = metrics.getSummary();

        double created = metrics.getCreatedMillis()*0.001;


        for (Entry<Double, Number> e: sum.entrySet()) {

            w.append(id + "{quantile=\""+formatNumber(e.getKey())+"\"} " +
                     formatNumber(e.getValue())+"\n");
        }

        w.append(id + "_created " + formatNumber(created)+"\n");
    }

    protected static void formatHistogram(Appendable w, String id, IActiveMetrics<?> metrics) throws IOException {

        SortedMap<Double,Integer> sum = metrics.getHistogram();

        double created = metrics.getCreatedMillis()*0.001;


        for (Entry<Double, Integer> e: sum.entrySet()) {

            w.append(id + "_bucket{le=\""+formatNumber(e.getKey())+"\"} " +
                     e.getValue()+"\n");
        }

        w.append(id + "_created " + formatNumber(created)+"\n");
    }

    public static void formatMetrics(Appendable w, IActiveMetrics<?> metrics) throws IOException {

        OpenMetricsType type = metrics.getOpenMetricsType();
        String id = quote(metrics.getTaggedMetrics().getId());

        w.append("# TYPE " + id + " " + type + "\n");

        switch (type) {

        case counter:
            formatCounter(w,id,metrics);
            break;
        case gauge:
            formatGauge(w,id,metrics);
            break;
        case histogram:
            formatHistogram(w,id,metrics);
            break;
        case summary:
            formatSummary(w,id,metrics);
            break;
        }
    }

}
