/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring;

import java.util.List;
import java.util.SortedMap;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import org.clazzes.svc.api.monitoring.AsyncMetrics;
import org.clazzes.svc.api.monitoring.Counter;
import org.clazzes.svc.api.monitoring.Gauge;
import org.clazzes.svc.api.monitoring.HealthCheck;
import org.clazzes.svc.api.monitoring.HealthInfo;
import org.clazzes.svc.api.monitoring.HealthStatus;
import org.clazzes.svc.api.monitoring.Histogram;
import org.clazzes.svc.api.monitoring.Summary;
import org.clazzes.svc.api.monitoring.TaggedMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricsHolder<T extends Comparable<T>> implements Runnable, IActiveMetrics<T> {

    private static final Logger log = LoggerFactory.getLogger(MetricsHolder.class);

    private final TaggedMetrics metrics;
    private final Callable<Result<T>> callable;
    private final Function<Long,Result<T>> initialFactory;
    private final Function<Throwable,Result<T>> exceptionFactory;
    private final OpenMetricsType openMetricsType;
    private final long createdMillis;
    private final AtomicInteger runCount;
    private Result<T> lastResult;
    private long timeoutNanos;
    private Future<?> future;
    private final ResultHistory<T> history;

    protected MetricsHolder(TaggedMetrics metrics,
           int keepResultsSeconds,
           Callable<Result<T>> callable,
           Function<Long,Result<T>> initialFactory,
           Function<Throwable,Result<T>> exceptionFactory,
           Function<Double,Result<T>> doubleFactory) {

        this.metrics = metrics;
        this.callable = callable;
        this.initialFactory = initialFactory;
        this.exceptionFactory = exceptionFactory;
        this.createdMillis = System.currentTimeMillis();
        this.runCount = new AtomicInteger();
        this.timeoutNanos = this.metrics.getTTLMillis() * 1000000L;

        this.history =
           (metrics instanceof Summary || metrics instanceof Histogram) ?
                new ResultHistory<>(keepResultsSeconds*1_000_000_000L, doubleFactory) :
                null;

        OpenMetricsType om;

        if (metrics instanceof Summary){
            om = OpenMetricsType.summary;
        }
        else if (metrics instanceof Histogram){
            om = OpenMetricsType.histogram;
        }
        else if (metrics instanceof Counter){
            om = OpenMetricsType.counter;
        }
        else {
            // healtch check and gauge are both gauges for openmetrics.
            om = OpenMetricsType.gauge;
        }
        this.openMetricsType = om;
    }


    public static MetricsHolder<HealthInfo> ofHealthCheck(HealthCheck hc,
           int keepResultsSeconds) {

        return new MetricsHolder<HealthInfo>(hc,keepResultsSeconds,
            () -> new HealthCheckResult(hc.call()),
            (interval) ->
                new HealthCheckResult(
                    new HealthInfo(HealthStatus.WARN,"First execution in ["+interval+"s]...")),
            (e) -> new HealthCheckResult(e),
            (d) -> new HealthCheckResult(d == null ? null : HealthInfo.ofValue(d))
        );
    }

    public static MetricsHolder<Long> ofCounter(Counter ctr,
           int keepResultsSeconds) {

        return new MetricsHolder<Long>(ctr,keepResultsSeconds,
            () -> new CounterResult(ctr.call()),
            null,
            (e) -> new CounterResult(e),
            (d) -> new CounterResult(d == null ? null : d.longValue())
        );
    }

    public static MetricsHolder<Double> ofGauge(Gauge gauge,
           int keepResultsSeconds) {

        return new MetricsHolder<Double>(gauge,keepResultsSeconds,
            () -> new GaugeResult(gauge.call()),
            null,
            (e) -> new GaugeResult(e),
            (d) -> new GaugeResult(d)
        );
    }

    @Override
    public TaggedMetrics getTaggedMetrics() {
        return this.metrics;
    }

    protected void putResult(Result<T> r) {

        synchronized(this) {
            this.lastResult = r;
            this.notifyAll();
        }

        if (this.history != null) {
            this.history.putResult(r);
        }
    }

    @Override
    public void run() {

        try {
            if (this.runCount.incrementAndGet() == 1) {

                Result<T> r;
                long startNanos = System.nanoTime();

                try {
                    r = this.callable.call();
                }
                catch(Exception e) {
                    r = this.exceptionFactory.apply(e);
                }

                r.calculateDuration(startNanos);
                this.putResult(r);
            }
            else {
                log.warn("Concurrent execution of metrics [{}] detected, waiting for primary exection.",
                    this.metrics.getClass().getName());

                this.wait(60000L);
            }

        } catch (InterruptedException e) {
            log.warn("Wait for concurrent execution of metrics [{}] has been interrupted.",
                    this.metrics.getClass().getName(),e);
        }
        finally {
            this.runCount.decrementAndGet();
        }
    }

    public boolean schedule(ScheduledExecutorService executorService) {

        if (this.metrics instanceof AsyncMetrics am) {

            long interval = am.getIntervalSeconds();

            if (interval > 0) {

                log.info("Scheduling metrics [{}] with interval [{}s].",
                            this.metrics.getId(),interval);

                if (this.initialFactory != null) {

                    // this is for starting with a WARN status for health checks.
                    synchronized(this) {
                        this.lastResult = this.initialFactory.apply(interval);
                    }
                }

                this.future = executorService.scheduleAtFixedRate(
                                this,
                                interval,interval,
                                TimeUnit.SECONDS);

                return true;
            }
            else {
                 log.info("Metrics [{}] with interval [{}s] with be executed synchronously.",
                            this.metrics.getId(),interval);

                return false;
            }
        }
        else {
            return false;
        }
    }

    public synchronized boolean isScheduled() {
        return this.future != null;
    }

    public synchronized void shutdown() {
        if (this.future != null) {
            this.future.cancel(true);
            this.future = null;
        }
    }

    @Override
    public Result<T> getResult() {

        boolean doRun;

        synchronized (this) {
            doRun =  this.future == null &&
                (this.lastResult == null ||
                 System.nanoTime() - this.lastResult.getNanoTime() >= this.timeoutNanos);
        }

        if(doRun) {
            this.run();
        }

        synchronized (this) {
            return this.lastResult;
        }
    }

    @Override
    public OpenMetricsType getOpenMetricsType() {
        return this.openMetricsType;
    }

    @Override
    public long getCreatedMillis() {
        return this.createdMillis;
    }

    @Override
    public SortedMap<Double,Number> getSummary() {

        if (this.metrics instanceof Summary summary) {
            return this.history.getSummary(summary.getQuantiles());
        }
        else {
            throw new UnsupportedOperationException("Cannot generate summary for non-summary metrics.");
        }
    }

    @Override
    public SortedMap<Double,Integer> getHistogram() {

        if (this.metrics instanceof Histogram histogram) {
            return this.history.getHistogram(histogram.getLEValues());
        }
        else {
            throw new UnsupportedOperationException("Cannot generate histogram for non-histogram metrics.");
        }
    }


    @Override
    public List<Result<T>> getHistory() {
        if (this.history == null) {
            return null;
        }
        else {
            return this.history.getHistory();
        }
    }


}
