/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring.health;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.text.NumberFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamWriter;

import org.clazzes.svc.api.monitoring.HealthCheck;
import org.clazzes.svc.api.monitoring.HealthInfo;
import org.clazzes.svc.api.monitoring.HealthStatus;
import org.clazzes.svc.runner.monitoring.IActiveMetrics;
import org.clazzes.svc.runner.monitoring.IActiveMonitoring;
import org.clazzes.svc.runner.monitoring.Result;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import jakarta.servlet.ServletException;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

public class SystemHealthServlet extends HttpServlet {

    private static final String NO_RESULT_YET = "No result yet...";

    private static final Logger log = LoggerFactory.getLogger(SystemHealthServlet.class);

    private final IActiveMonitoring monitoring;

    private static final String XHTML_NS_URI = "http://www.w3.org/1999/xhtml";

    // felix health checks set 'background-color:#FFFDF1;' on body, which
    // we refrain to use for contrast (anti-mustard look'n'feel)
    private static final String CSS = """
body { font-size:12px; font-family:arial,verdana,sans-serif; }
h1 { font-size:20px;}
table { font-size:12px; border:#ccc 1px solid; border-radius:3px; }
table th { padding:5px; text-align: left; background: #ededed; }
table td { padding:5px; border-top: 1px solid #ffffff; border-bottom:1px solid #e0e0e0; border-left: 1px solid #e0e0e0; }
.statusOK { background-color:#CCFFCC;}
.statusWARN { background-color:#FFE569;}
.statusCRITICAL { background-color:#F0975A;}
.statusHEALTH_CHECK_ERROR { background-color:#F16D4E;}
.statusnull { background-color:#CCCCCC;}
.helpText { color:grey; font-size:80%; }
            """;

    private static final String HELP_ID = """
health check ID to select - can also be specified multiple times like /system/health?id=system.memory&id=jdbc.MYDB
            """;

    private static final String HELP_TAG = """
health check tag to select - can also be specified multiple times like /system/health?tag=system&tag=jdbc
            """;

    // felix health check also know about txt|verbose.txt formats.
    private static final String HELP_FORMAT = """
Output format, html|json|yaml - The default format is deduced from the Accept HTTP-Header
            """;

    private static final DateTimeFormatter HTML_DTF =
        DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss",Locale.ENGLISH);

    private static final DateTimeFormatter JSON_DTF =
        DateTimeFormatter.ofPattern("uuuu-MM-dd'T'HH:mm:ss.SSSZ");

    private static final NumberFormat DURATION_FORMAT =
        NumberFormat.getInstance(Locale.ENGLISH);

    private static final XMLOutputFactory XOF = XMLOutputFactory.newDefaultFactory();

    static {
        DURATION_FORMAT.setMinimumFractionDigits(1);
        DURATION_FORMAT.setMaximumFractionDigits(1);
    }


    public SystemHealthServlet(IActiveMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    private static final void setNoCacheHeaders(HttpServletResponse resp) {
        resp.setHeader("Cache-Control","no-cache");
        resp.setHeader("Pragma","no-cache");
        resp.setHeader("Expires","0");
    }

    protected void formatHTML(HttpServletResponse resp,
            SortedMap<String,IActiveMetrics<HealthInfo>> selectedMetrics,
            Map<String,Result<HealthInfo>> results,
            HealthStatus overall) throws Exception {

        ServletOutputStream os = resp.getOutputStream();

        resp.setHeader("X-Frame-Options","SAMEORIGIN");
        resp.setHeader("Content-Language","en");
        resp.setContentType("text/html; charset=utf-8");

        os.write("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n".getBytes("UTF-8"));

        XMLStreamWriter xsw = XOF.createXMLStreamWriter(os);

        try {
            xsw.setDefaultNamespace(XHTML_NS_URI);

            xsw.writeStartElement("html");
            xsw.writeDefaultNamespace(XHTML_NS_URI);
            xsw.writeAttribute("lang","en");
            xsw.writeAttribute("xml:lang","en");
            xsw.writeStartElement("head");

            xsw.writeEmptyElement("meta");
            xsw.writeAttribute("http-equiv","Content-Type");
            xsw.writeAttribute("content","text/html; charset=utf-8");

            xsw.writeStartElement("style");
            xsw.writeCharacters(CSS);
            xsw.writeEndElement(); // </style>

            xsw.writeStartElement("title");
            xsw.writeCharacters("System Health");
            xsw.writeEndElement(); // </title>

            xsw.writeEndElement(); // </head>

            xsw.writeStartElement("body");

            xsw.writeStartElement("h1");
            xsw.writeCharacters("System Health");
            xsw.writeEndElement(); // </h1>

            // overall status
            xsw.writeStartElement("p");
            xsw.writeStartElement("span");
            xsw.writeAttribute("class","status"+overall);
            xsw.writeAttribute("style","padding:4px");
            xsw.writeStartElement("strong");
            xsw.writeCharacters("Overall Result: "+overall);
            xsw.writeEndElement(); // </strong>
            xsw.writeEndElement(); // </span>
            xsw.writeEndElement(); // </p>


            // status table
            xsw.writeStartElement("table");
            xsw.writeAttribute("id","healthCheckResults");
            xsw.writeAttribute("cellspacing","0");
            // header
            xsw.writeStartElement("thead");
            xsw.writeStartElement("tr");
            xsw.writeStartElement("th");
            xsw.writeCharacters("Health Check ");
            xsw.writeStartElement("span");
            xsw.writeAttribute("style","color:gray");
            xsw.writeCharacters("(tags)");
            xsw.writeEndElement(); // </span>
            xsw.writeEndElement(); // </th>
            xsw.writeStartElement("th");
            xsw.writeCharacters("Status");
            xsw.writeEndElement(); // </th>
            xsw.writeStartElement("th");
            xsw.writeCharacters("Log");
            xsw.writeEndElement(); // </th>
            xsw.writeStartElement("th");
            xsw.writeCharacters("Finished At");
            xsw.writeEndElement(); // </th>
            xsw.writeStartElement("th");
            xsw.writeCharacters("Time");
            xsw.writeEndElement(); // </th>
            xsw.writeEndElement(); // </tr>
            xsw.writeEndElement(); // </thead>

            // body
            xsw.writeStartElement("tbody");

            for (Entry<String, IActiveMetrics<HealthInfo>> e:selectedMetrics.entrySet()) {

                String id = e.getKey();
                Result<HealthInfo> r = results.get(id);
                HealthStatus status = r == null ? null : r.getResult().getStatus();

                if (status == null) {
                    status = HealthStatus.HEALTH_CHECK_ERROR;
                }

                xsw.writeStartElement("tr");
                xsw.writeAttribute("class","status"+status);

                xsw.writeStartElement("td");
                xsw.writeAttribute("title",e.getValue().getTaggedMetrics().getDescription());
                xsw.writeCharacters(id);
                xsw.writeEmptyElement("br");
                xsw.writeStartElement("span");
                xsw.writeAttribute("style","color:gray");

                List<String> tags = e.getValue().getTaggedMetrics().getTags();
                if (tags != null) {
                    xsw.writeCharacters(String.join(", ",tags));
                }
                xsw.writeEndElement(); // </span>
                xsw.writeEndElement(); // </td>

                xsw.writeStartElement("td");
                xsw.writeAttribute("style","font-weight:bold");
                xsw.writeCharacters(status.toString());
                xsw.writeEndElement(); // </td>

                xsw.writeStartElement("td");
                String expl = r == null ? NO_RESULT_YET : r.getResult().getExplanation();
                if (expl != null) {

                    String[] lines = expl.split("\n");

                    for (int i=0;i<lines.length;++i) {

                        if (i > 0) {
                            xsw.writeEmptyElement("br");
                        }
                        xsw.writeCharacters(lines[i]);
                    }
                }
                xsw.writeEndElement(); // </td>

                ZonedDateTime dt = r==null ?
                                        ZonedDateTime.now() :
                                        ZonedDateTime.ofInstant(
                                            Instant.ofEpochMilli(r.getEpochMillis()),
                                            ZoneId.systemDefault());

                xsw.writeStartElement("td");
                xsw.writeCharacters(HTML_DTF.format(dt));
                xsw.writeEndElement(); // </td>

                Long duration = r == null ? null : r.getNanosDuration();

                xsw.writeStartElement("td");

                if (duration != null) {

                    xsw.writeCharacters(DURATION_FORMAT.format(duration.longValue()*1.0e-6)+"ms");
                }
                xsw.writeEndElement(); // </td>

                xsw.writeEndElement(); // </tr>
            }

            xsw.writeEndElement(); // </tbody>
            xsw.writeEndElement(); // </table>

            // help
            /*
             * <div class="helpText">
<h3>Supported URL parameters</h3>
<b>tags</b>:Comma-separated list of health checks tags to select - can also be specified via path, e.g. /system/health/tag1,tag2.json. Exclusions can be done by prepending '-' to the tag name<br><b>names</b>:Comma-separated list of health check names to select. Exclusions can be done by prepending '-' to the health check name<br><b>format</b>:Output format, html|json|jsonp|txt|verbose.txt - an extension in the URL overrides this<br><b>httpStatus</b>:Specify HTTP result code, for example CRITICAL:503 (status 503 if result &gt;= CRITICAL) or CRITICAL:503,HEALTH_CHECK_ERROR:500,OK:418 for more specific HTTP status<br><b>combineTagsWithOr</b>:Combine tags with OR, active by default. Set to false to combine with AND<br><b>forceInstantExecution</b>:If true, forces instant execution by executing async health checks directly, circumventing the cache (2sec by default) of the HealthCheckExecutor<br><b>timeout</b>:(msec) a timeout status is returned for any health check still running after this period. Overrides the default HealthCheckExecutor timeout<br><b>hcDebug</b>:Include the DEBUG output of the Health Checks<br><b>callback</b>:name of the JSONP callback function to use, defaults to processHealthCheckResults<br>
</div>
             */
            xsw.writeStartElement("div");
            xsw.writeAttribute("class","helpText");

            xsw.writeStartElement("h3");
            xsw.writeCharacters("Supported URL parameters");
            xsw.writeEndElement(); // </h3>

            xsw.writeStartElement("b");
            xsw.writeCharacters("id");
            xsw.writeEndElement(); // </b>
            xsw.writeCharacters(" ");
            xsw.writeCharacters(HELP_ID);

            xsw.writeEmptyElement("br");

            xsw.writeStartElement("b");
            xsw.writeCharacters("tag");
            xsw.writeEndElement(); // </b>
            xsw.writeCharacters(" ");
            xsw.writeCharacters(HELP_TAG);

            xsw.writeEmptyElement("br");

            xsw.writeStartElement("b");
            xsw.writeCharacters("format");
            xsw.writeEndElement(); // </b>
            xsw.writeCharacters(" ");
            xsw.writeCharacters(HELP_FORMAT);

            xsw.writeEndElement(); // </div>

            xsw.writeEndElement(); // </body>
        }
        finally {
            xsw.close();
        }
    }

    /*
    {
  "overallResult": "OK",
  "results": [
    {
      "name": "Bundles Started",
      "status": "OK",
      "timeInMs": 0,
      "finishedAt": "2025-07-28T23:02:40.499",
      "tags": [
        "bundles",
        "docker",
        "monitoring"
      ],
      "messages": [
        {
          "status": "OK",
          "message": "All 1 bundles for pattern clazzes-util are started"
        }
      ]
    },...
  ]
     */

    protected void formatJackson(JsonGenerator w,
            SortedMap<String,IActiveMetrics<HealthInfo>> selectedMetrics,
            Map<String,Result<HealthInfo>> results,
            HealthStatus overall) throws Exception {

        w.writeStartObject();
        w.writeFieldName("overallResult");
        if (overall == null) {
            w.writeNull();
        }
        else {
            w.writeString(overall.toString());
        }

        w.writeFieldName("results");

        w.writeStartArray();

        for (Entry<String, IActiveMetrics<HealthInfo>> e:selectedMetrics.entrySet()) {

            String id = e.getKey();
            Result<HealthInfo> r = results.get(id);
            HealthStatus status =  r == null ? null : r.getResult().getStatus();

            if (status == null) {
                status = HealthStatus.HEALTH_CHECK_ERROR;
            }
            w.writeStartObject();

            w.writeStringField("id",id);
            w.writeStringField("status",status.toString());

            ZonedDateTime dt = r==null ?
                                    ZonedDateTime.now() :
                                    ZonedDateTime.ofInstant(
                                        Instant.ofEpochMilli(r.getEpochMillis()),
                                        ZoneId.systemDefault());

            w.writeStringField("finishedAt",JSON_DTF.format(dt));
            if (r != null && r.getNanosDuration() != null) {
                w.writeNumberField("timeInMs",r.getNanosDuration()*1.0e-6);
            }

            List<String> tags = e.getValue().getTaggedMetrics().getTags();
            if (tags != null) {
                w.writeFieldName("tags");
                w.writeStartArray();
                for (String tag:tags) {
                    w.writeString(tag);
                }
                w.writeEndArray();
            }

            w.writeFieldName("messages");
            w.writeStartArray();

            // FIXME maybe integrate history at this point, maybe as extra
            //       servlet parameter.
            // e.getValue().getHistory();

            // for the moment, we render the current result only.
            {
                w.writeStartObject();
                w.writeStringField("status",status.toString());
                String expl = r == null ? NO_RESULT_YET : r.getResult().getExplanation();
                if (expl != null) {
                    w.writeStringField("message",expl);
                }
                w.writeEndObject();

            }

            w.writeEndArray();

            w.writeEndObject();
        }

        w.writeEndArray();

        w.writeEndObject();
    }

    protected void formatJson(HttpServletResponse resp,
            SortedMap<String,IActiveMetrics<HealthInfo>> selectedMetrics,
            Map<String,Result<HealthInfo>> results,
            HealthStatus overall) throws Exception {

        resp.setContentType("application/json");

        JsonFactory jf = JsonFactory.builder().build();

        try (OutputStreamWriter osw = new OutputStreamWriter(resp.getOutputStream(),"utf-8")) {

            JsonGenerator gen = jf.createGenerator(osw);

            this.formatJackson(gen,selectedMetrics,results,overall);
            gen.flush();
        }
    }

    protected void formatYaml(HttpServletResponse resp,
            SortedMap<String,IActiveMetrics<HealthInfo>> selectedMetrics,
            Map<String,Result<HealthInfo>> results,
            HealthStatus overall) throws Exception {

        resp.setContentType("application/yaml");

        YAMLFactory yf = YAMLFactory.builder().build();

        try (OutputStreamWriter osw = new OutputStreamWriter(resp.getOutputStream(),"utf-8")) {

            JsonGenerator gen = yf.createGenerator(osw);

            this.formatJackson(gen,selectedMetrics,results,overall);
            gen.flush();
        }
    }

    protected void formatPlainText(HttpServletResponse resp,
            HealthStatus overall) throws Exception {

        resp.setContentType("text/plain;charset=utf-8");

        try (OutputStreamWriter osw = new OutputStreamWriter(resp.getOutputStream(),"utf-8")) {

            osw.write(String.valueOf(overall));
            osw.write('\n');
        }
    }

    protected static final HealthStatus aggregate(HealthStatus a, HealthStatus b) {

        if (a == null) {
            return b;
        }

        if (b == null || b == HealthStatus.HEALTH_CHECK_ERROR) {
            return HealthStatus.HEALTH_CHECK_ERROR;
        }

        return b.getValue() < a.getValue() ? b : a;
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

        String[] ids = req.getParameterValues("id");
        String[] tags = req.getParameterValues("tag");

        String format = req.getParameter("format");

        if (format == null) {
            String accept = req.getHeader("Accept");

            if (accept != null) {
                if ("application/json".equals(accept)) {
                    format = "json";
                }
                else if ("application/yaml".equals(accept)) {
                    format = "yaml";
                }
                else if ("text/plain".equals(accept) ||
                         accept.startsWith("text/plain,") ||
                         accept.startsWith("text/plain;")
                        ) {
                    format = "txt";
                }
                else if ("text/html".equals(accept) ||
                         accept.startsWith("text/html,") ||
                         accept.startsWith("text/html;") ||
                         "*/*".equals(accept) // curl
                        ) {
                    format = "html";
                }
                else {
                    log.warn("Unparseable Accept header [{}], assuming text/html",accept);
                }
            }
        }

        SortedMap<String,IActiveMetrics<HealthInfo>> selectedMetrics = new TreeMap<String,IActiveMetrics<HealthInfo>>();

        if (ids == null && tags == null) {
            selectedMetrics.putAll(this.monitoring.getHealthChecks());
        }
        else {
            if (ids != null) {
                for (String id:ids) {
                    IActiveMetrics<HealthInfo> hc = this.monitoring.getHealthCheck(id);
                    if (hc != null) {
                        selectedMetrics.put(id,hc);
                    }
                }
            }

            if (tags != null) {
                for (String tag : tags) {
                    for (IActiveMetrics<?> m : this.monitoring.getByTag(tag)) {
                        if (m.getTaggedMetrics() instanceof HealthCheck) {
                            selectedMetrics.put(m.getTaggedMetrics().getId(),
                                (IActiveMetrics<HealthInfo>) m);
                        }
                    }
                }
            }
        }

        Map<String,Result<HealthInfo>> results = new HashMap<String,Result<HealthInfo>>();

        HealthStatus overall = null;

        for (Entry<String, IActiveMetrics<HealthInfo>> e:selectedMetrics.entrySet()) {

            Result<HealthInfo> r = e.getValue().getResult();
            results.put(e.getKey(),r);
            overall = aggregate(overall,r == null ? HealthStatus.HEALTH_CHECK_ERROR : r.getResult().getStatus());
        }

        int responseCode;

        if (overall == null) {
            responseCode = 500;
        }
        else {
            switch(overall) {

            case OK:
            case WARN:
                responseCode = 200;
                break;
            case CRITICAL:
                responseCode = 503;
                break;
            default:
                responseCode = 500;
            }
        }

        resp.setStatus(responseCode);
        setNoCacheHeaders(resp);

        try {
            if ("yaml".equals(format)) {
                this.formatYaml(resp,selectedMetrics,results,overall);
            }
            else if ("json".equals(format)) {
                this.formatJson(resp,selectedMetrics,results,overall);
            }
            else if ("txt".equals(format)) {
                this.formatPlainText(resp,overall);
            }
            else {
                this.formatHTML(resp,selectedMetrics,results,overall);
            }
        } catch (Exception e) {
            log.error("Error formatting Health Checks",e);
            throw new ServletException("Cannot format health checks");
        }
    }

}
