/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring;

import java.util.ArrayList;
import java.util.List;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;

public class ResultHistory<T extends Comparable<T>> {

    private final long retentionNanons;
    private final Function<Double,Result<T>> doubleFactory;

    private SortedMap<Long,Result<T>> resultsByNanoTime;
    private SortedSet<Result<T>> resultsByValue;

    public ResultHistory(long retentionNanons, Function<Double,Result<T>> doubleFactory) {
        this.retentionNanons = retentionNanons;
        this.doubleFactory = doubleFactory;
        this.resultsByNanoTime = new TreeMap<Long,Result<T>>();
        this.resultsByValue = new TreeSet<Result<T>>();
    }

    public synchronized void putResult(Result<T> r) {

        long watermark = r.getNanoTime() - this.retentionNanons;

        SortedMap<Long,Result<T>> head = this.resultsByNanoTime.headMap(watermark);

        this.resultsByValue.removeAll(head.values());
        head.clear();

        this.resultsByNanoTime.put(r.getNanoTime(),r);

        if (r.isValid()) {
            this.resultsByValue.add(r);
        }
    }

    public synchronized List<Result<T>> getHistory() {

        return new ArrayList<Result<T>>(this.resultsByNanoTime.values());
    }

    public synchronized SortedMap<Double,Integer> getHistogram(double[] leValues) {

        SortedMap<Double,Integer> ret = new TreeMap<Double,Integer>();

        for (double le : leValues) {

            ret.put(le,
                this.resultsByValue.headSet(this.doubleFactory.apply(le)).size());
        }

        ret.put(Double.POSITIVE_INFINITY,this.resultsByValue.size());

        return ret;
    }

    public synchronized SortedMap<Double,Number> getSummary(double[] quantiles) {

        SortedMap<Double,Number> ret = new TreeMap<Double,Number>();

        if (quantiles == null || quantiles.length <= 0) {
            throw new IllegalArgumentException("Wee need to caclulate at least one quantile.");
        }

        int iq = 0;
        int ir = 0;
        double q = quantiles [iq];
        double nri = 1.0/(this.resultsByValue.size() - 1.0);

        for (Result<T> r : this.resultsByValue) {

            if (nri * ir > q) {

                ret.put(q,r.getNumericResult());
                ++iq;
                if (iq >= quantiles.length) {
                    break;
                }
                q = quantiles [iq];
            }

            ++ir;
        }

        return ret;
    }

}
