/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner.monitoring;

public abstract class Result<T extends Comparable<T>> implements Comparable<Result<T>> {

    private final long nanoTime;
    private final long epochMillis;
    private final T result;
    private final Throwable exception;
    private Long nanosDuration;

    public Result(T result) {
        this.nanoTime = System.nanoTime();
        this.epochMillis = System.currentTimeMillis();
        this.result = result;
        this.exception = null;
    }

    public Result(Throwable exception) {
        this.nanoTime = System.nanoTime();
        this.epochMillis = System.currentTimeMillis();
        this.result = null;
        this.exception = exception;
    }

    public Result(T result, Throwable exception) {
        this.nanoTime = System.nanoTime();
        this.epochMillis = System.currentTimeMillis();
        this.result = result;
        this.exception = exception;
    }

    public long getNanoTime() {
        return this.nanoTime;
    }

    public long getEpochMillis() {
        return this.epochMillis;
    }

    public T getResult() {
        return this.result;
    }

    public Throwable getException() {
        return this.exception;
    }

    abstract public boolean isValid();

    abstract public Number getNumericResult();

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (int) (this.nanoTime ^ (this.nanoTime >>> 32));
        result = prime * result + ((this.result == null) ? 0 : this.result.hashCode());
        return result;
    }

    public void calculateDuration(long startNanos) {
        this.nanosDuration = this.nanoTime-startNanos;
    }

    public Long getNanosDuration() {
        return nanosDuration;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        @SuppressWarnings("rawtypes")
        Result other = (Result) obj;
        if (this.nanoTime != other.nanoTime)
            return false;
        if (this.result == null) {
            if (other.result != null)
                return false;
        } else if (!this.result.equals(other.result))
            return false;
        return true;
    }

    @Override
    public int compareTo(Result<T> o) {

        int r;

        if (this.result == null) {
            r = o.getResult() == null ? 0 : -1;
        }
        else if (o.getResult() == null) {
            r = 1;
        }
        else {
            r = this.result.compareTo(o.getResult());
        }

        if (r != 0) {
            return r;
        }

        return Long.signum(this.nanoTime-o.getNanoTime());
    }

    @Override
    public String toString() {
        return "Result [nanoTime=" + this.nanoTime +
                ", epochMillis=" + this.epochMillis +
                ", result=" + this.result +
                ", exception=" + this.exception +
                ", nanosDuration=" + this.nanosDuration + "]";
    }

}
