package org.clazzes.svc.runner.monitoring;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ComponentSupport;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServicePriority;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.svc.api.monitoring.Counter;
import org.clazzes.svc.api.monitoring.Gauge;
import org.clazzes.svc.api.monitoring.HealthCheck;
import org.clazzes.svc.runner.monitoring.health.SystemHealthServlet;
import org.clazzes.svc.runner.monitoring.openmetrics.OpenMetricsServlet;
import org.clazzes.svc.runner.monitoring.system.CPUHealthCheck;
import org.clazzes.svc.runner.monitoring.system.ComponentsHealthCheck;
import org.clazzes.svc.runner.monitoring.system.DiskHealthCheck;
import org.clazzes.svc.runner.monitoring.system.MemoryHealthCheck;

import jakarta.servlet.Servlet;

@ServicePriority(6)
public class MonitoringComponent extends ComponentSupport implements Component {

    public static final String CONFIG_PID= "org.clazzes.svc.runner.monitoring";

    private final class ConfigListener extends ComponentSupport implements Consumer<ConfigWrapper>,AutoCloseable {

        private final ServiceRegistry serviceRegistry;
        private final ComponentManager componentManager;
        private ScheduledExecutorService executorService;
        private MonitoringEngine monitoringEngine;
        private List<String> activeSystemChecks;

        public ConfigListener(ServiceRegistry serviceRegistry, ComponentManager componentManager) {
            this.serviceRegistry = serviceRegistry;
            this.componentManager = componentManager;
        }

        @Override
        public void accept(ConfigWrapper config) {

            int monitoringThreads = config.getInt("monitoringThreads",4);
            int keepResultsSeconds = config.getInt("keepResultsSeconds",900);

            ConfigWrapper systemChecksTree = config.getSubTree("systemChecks");

            List<HealthCheck> systemChecks =
                    new ArrayList<HealthCheck>();

            systemChecks.add(new ComponentsHealthCheck(this.componentManager));

            if (systemChecksTree != null) {

                ConfigWrapper memoryConfig = systemChecksTree.getSubTree("memory");

                if (memoryConfig != null) {
                    systemChecks.add(
                        new MemoryHealthCheck(
                                memoryConfig.getDouble("warning",90.0),
                                memoryConfig.getDouble("critical",95.0)));
                }

                ConfigWrapper cpuConfig = systemChecksTree.getSubTree("cpu");

                if (cpuConfig != null) {
                    systemChecks.add(
                        new CPUHealthCheck(
                                cpuConfig.getDouble("warning",95.0),
                                cpuConfig.getDouble("critical",99.0)));
                }

                ConfigWrapper diskConfig = systemChecksTree.getSubTree("disk");

                if (diskConfig != null) {
                    systemChecks.add(
                        new DiskHealthCheck(
                                diskConfig.getDouble("warning",90.0),
                                diskConfig.getDouble("critical",95.0)));
                }
            }

            this.executorService = Executors.newScheduledThreadPool(monitoringThreads);
            MonitoringEngine me = new MonitoringEngine(this.executorService,keepResultsSeconds);

            this.addListener(this.serviceRegistry.listenAll(HealthCheck.class,
                (k,hc) -> {me.addHealthCheck(k, hc);},
                (k,hc) -> {me.removeHealthCheck(k);}
            ));

            this.addListener(this.serviceRegistry.listenAll(Counter.class,
                (k,hc) -> {me.addCounter(k, hc);},
                (k,hc) -> {me.removeCounter(k);}
            ));

            this.addListener(this.serviceRegistry.listenAll(Gauge.class,
                (k,hc) -> {me.addGauge(k, hc);},
                (k,hc) -> {me.removeGauge(k);}
            ));

            this.monitoringEngine = me;

            this.activeSystemChecks = new ArrayList<String>();

            for (HealthCheck hc : systemChecks) {
                this.monitoringEngine.addHealthCheck(hc.getId(),hc);
                this.activeSystemChecks.add(hc.getId());
            }

            this.addService(this.serviceRegistry,
                "/metrics",Servlet.class,
                new OpenMetricsServlet(monitoringEngine));

            this.addService(this.serviceRegistry,
                "/system/health",Servlet.class,
                new SystemHealthServlet(monitoringEngine));

            this.componentManager.commit();
        }

        @Override
        public void close() throws Exception {

            this.removeAllServices(this.serviceRegistry);
            this.closeAllListeners();

            if (this.executorService != null) {
                this.executorService.shutdownNow();
            }

            if (this.monitoringEngine != null) {

                if (this.activeSystemChecks != null) {
                    for (String key : this.activeSystemChecks) {
                        this.monitoringEngine.removeHealthCheck(key);
                    }
                    this.activeSystemChecks = null;
                }
                this.monitoringEngine = null;
            }
        }
    }

    @Override
    public void start(ServiceContext context) throws Exception {

        ServiceRegistry serviceRegistry = context.getService(ServiceRegistry.class).get();
        ComponentManager componentManager = context.getService(ComponentManager.class).get();
        ConfigurationEngine configEngine = context.getService(ConfigurationEngine.class).get();

        ConfigListener cfgListener = new ConfigListener(serviceRegistry,componentManager);

        this.addListener(configEngine.listen(CONFIG_PID,cfgListener));
        this.addListener(cfgListener);
    }

    @Override
    public void stop(ServiceContext context) throws Exception {
        this.closeAllListeners();
    }

}
