package org.clazzes.svc.runner.jetty;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServicePriority;
import org.clazzes.svc.api.ServiceRegistry;
import org.eclipse.jetty.ee10.servlet.ErrorHandler;
import org.eclipse.jetty.ee10.servlet.ServletContextHandler;
import org.eclipse.jetty.ee10.servlet.ServletHandler;
import org.eclipse.jetty.ee10.servlet.ServletHolder;
import org.eclipse.jetty.ee10.servlet.ServletMapping;
import org.eclipse.jetty.server.CustomRequestLog;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.Slf4jRequestLogWriter;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.server.handler.ResourceHandler;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.util.resource.ResourceFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.servlet.Servlet;

@ServicePriority(7)
public class JettyComponent implements Component {

    public static final String PID = "org.clazzes.svc.runner.jetty";

    private static final Logger log = LoggerFactory.getLogger(JettyComponent.class);

    private Server server;
    private AutoCloseable servletRegistration;
    private final Map<String,ServletHolder> servlets;

    public JettyComponent() {
        this.servlets = new ConcurrentHashMap<String,ServletHolder>();
    }

    @Override
    public void start(ServiceContext svcCtxt) throws Exception {

        ConfigurationEngine configurationEngine = svcCtxt.getService(ConfigurationEngine.class).get();

        ComponentManager componentManager = svcCtxt.getService(ComponentManager.class).get();

        configurationEngine.listen(PID,(config) -> {

            try {
                this.stop(svcCtxt);
            } catch (Exception e) {
                log.warn("Error stopping jetty component",e);
            }

            try {

                int port = config.getInt("port",8088);
                int nthreads = config.getInt("nthreads",8);

                QueuedThreadPool threadPool = new QueuedThreadPool(nthreads);
                threadPool.setName("svc-runner-jetty");

                Server jettyServer = new Server(threadPool);

                ErrorHandler errorHandler = new ErrorHandler();

                errorHandler.setShowMessageInTitle(
                    config.getBoolean("showMessageInTitle",false));

                errorHandler.setShowOrigin(
                    config.getBoolean("showOrigin",false));

                errorHandler.setShowCauses(
                    config.getBoolean("showCauses",false));

                errorHandler.setShowStacks(
                    config.getBoolean("showStacks",false));


                jettyServer.setErrorHandler(errorHandler);

                jettyServer.setRequestLog(new CustomRequestLog(new Slf4jRequestLogWriter(),CustomRequestLog.EXTENDED_NCSA_FORMAT));

                HttpConfiguration httpConfiguration = new HttpConfiguration();

                httpConfiguration.setSendServerVersion(
                    config.getBoolean("sendServerVersion",false));

                HttpConnectionFactory httpConnectionFactory = new HttpConnectionFactory(httpConfiguration);

                ServerConnector connector = new ServerConnector(jettyServer,httpConnectionFactory);
                connector.setPort(port);

                jettyServer.addConnector(connector);

                ServletContextHandler servletHandler = new ServletContextHandler();

                ConfigWrapper resourceTree = config.getSubTree("resources");

                if (resourceTree == null) {
                    jettyServer.setHandler(servletHandler);
                }
                else {

                    Handler.Sequence handlers = new Handler.Sequence();

                    for (String path : resourceTree.keySet()) {

                        Path fileSystemPath = resourceTree.getMandatoryParsed(Path::of,path);

                        if (Files.isDirectory(fileSystemPath)) {

                            log.info("Adding static resource [{}] as path [{}].",
                                    fileSystemPath,path);

                            ResourceHandler resourceHandler = new ResourceHandler();
                            resourceHandler.setBaseResource(ResourceFactory.of(resourceHandler).newResource(fileSystemPath));

                            ContextHandler contextHandler = new ContextHandler(resourceHandler,path);
                            handlers.addHandler(contextHandler);
                        }
                        else {
                            log.warn("Configured resource [{}] for path [{}] is not a directory, skipping it.",
                                    fileSystemPath,path);
                        }
                    }

                    handlers.addHandler(servletHandler);

                    jettyServer.setHandler(handlers);
                }

                AtomicBoolean started = new AtomicBoolean(false);

                jettyServer.addEventListener(new LifeCycle.Listener() {

                    @Override
                    public void lifeCycleFailure(LifeCycle event, Throwable cause) {
                        log.error("Start of jetty failed:",cause);
                        synchronized(JettyComponent.this) {
                            started.set(true);
                            JettyComponent.this.notifyAll();
                        }
                    }

                    @Override
                    public void lifeCycleStarted(LifeCycle event) {
                        log.info("Start of jetty finished successfully.");
                        synchronized(JettyComponent.this) {
                            started.set(true);
                            JettyComponent.this.notifyAll();
                        }
                    }

                });

                jettyServer.start();

                log.info("Waiting for full jetty startup...");

                synchronized (this) {
                    if (started.get() == false) {
                        this.wait(60000L);
                    }
                }

                log.info("Wait for full jetty startup finished with result [{}].",started.get());

                if (started.get() == false) {
                    log.warn("Jetty did not start up witin ohne minute, server might be most likely inoperable.");
                }

                ServiceRegistry registry = svcCtxt.getService(ServiceRegistry.class).get();

                // All the servletHandler coed is not thread-safe, so we need to add
                // our own synchronized-blocks.
                this.servletRegistration = registry.listenAll(Servlet.class,
                    (key,servlet) ->  { synchronized(this) {

                        log.info("Adding servlet [{}] for path [{}]",servlet.getServletInfo(),key);
                        ServletHolder holder = new ServletHolder(servlet);
                        servletHandler.addServlet(holder,key);
                        this.servlets.put(key,holder);
                    }},
                    (key,servlet) ->  { synchronized(this) {
                        ServletHolder holder = this.servlets.remove(key);
                        if (holder != null) {
                            try {
                                log.info("Stopping servlet holder for path [{}]",key);
                                holder.stop();

                                ServletHandler h = servletHandler.getServletHandler();

                                ServletMapping[] mappings = h.getServletMappings();

                                if (mappings.length > 0) {

                                    ServletMapping[] newm = new ServletMapping[mappings.length-1];

                                    int nnew = 0;

                                    for (ServletMapping m : mappings) {

                                        boolean matches = false;
                                        for (String p : m.getPathSpecs()) {
                                            matches |= Objects.equals(p,key);
                                        }

                                        if (!matches) {
                                            if (nnew < newm.length) {
                                                newm[nnew] = m;
                                            }
                                            ++nnew;
                                        }
                                    }

                                    if (nnew == newm.length) {
                                        if (log.isDebugEnabled()) {
                                            log.debug("New mappings after removal of [{}] are {}",
                                                      key,Arrays.toString(newm));
                                        }
                                        h.setServletMappings(newm);
                                    }
                                    else {
                                        log.warn("Cannot find a mapping for [{}] to remove.",key);
                                    }
                                }

                            } catch (Exception e) {
                                log.error("Error stopping servlet holder for path [{}]",key,e);
                            }
                        }
                    }}
                );

                this.server = jettyServer;
                componentManager.commit();

            } catch (Throwable e) {
                throw new RuntimeException("Error setting up Jetty Server: "+e.getMessage(),e);
            }
        });
    }

    @Override
    public void stop(ServiceContext svcCtxt) throws Exception {

        if (this.server != null) {
            try {
                this.server.stop();
            }
            catch(Throwable e) {
                log.error("Error stopping Jetty Server",e);
            }
            this.server = null;
        }

        if (this.servletRegistration != null) {
            try {
                this.servletRegistration.close();
            }
            catch(Throwable e) {
                log.error("Error cancelling listener for Servlet instances.",e);
            }
            this.servletRegistration = null;
            this.servlets.clear();
        }
    }

}