/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

 package org.clazzes.svc.runner.jdbc;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;

import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.monitoring.AsyncMetrics;
import org.clazzes.svc.api.monitoring.HealthCheck;
import org.clazzes.svc.api.monitoring.HealthInfo;
import org.clazzes.svc.api.monitoring.HealthStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zaxxer.hikari.HikariDataSource;

public class JdbcHealthCheck implements HealthCheck, AsyncMetrics {

    private final static Logger log = LoggerFactory.getLogger(JdbcHealthCheck.class);

    private final String name;
    private final String displayName;
    private final List<String> tags;
    private final HikariDataSource dataSource;
    private final String healthQuery;
    private final long intervalSeconds;

    public JdbcHealthCheck(String name,
                           String displayName,
                           List<String> tags,
                           HikariDataSource dataSource,
                           String healthQuery,
                           long intervalSeconds) {
        this.name = name;
        this.displayName = displayName;
        this.tags = tags;
        this.dataSource = dataSource;
        this.healthQuery = healthQuery;
        this.intervalSeconds = intervalSeconds;
    }

    private static List<String> getHealthTags(ConfigWrapper config) {

        List<String> healthTags = config.getStringList("healthTags");

        if (healthTags == null) {
            return null;
        }

        if (healthTags.contains("jdbc")) {
           return healthTags;
        }

        List<String> ret = new ArrayList<String>(healthTags.size()+1);

        ret.add("jdbc");
        ret.addAll(healthTags);
        return ret;
    }

    public static JdbcHealthCheck ofConfig(String name, HikariDataSource dataSource, ConfigWrapper config) {

        List<String> healthTags = getHealthTags(config);

        if (healthTags == null) {
            return null;
        }

        String healthQuery = config.getString("healthQuery");
        long healthInterval = config.getLong("healthInterval",10L);
        boolean healthShowUrl = config.getBoolean("healthShowUrl",false);

        String displayName;

        if (healthShowUrl) {

            displayName = name + "|" + dataSource.getJdbcUrl();
        }
        else {
            displayName = name;
        }

        return new JdbcHealthCheck(name,displayName,healthTags,dataSource,healthQuery,healthInterval);
    }

    @Override
    public String getDescription() {

        return "Check database "+this.name+" using query ["+this.healthQuery+"].";
    }

    @Override
    public String getId() {
        return "jdbc." + this.name;
    }

    @Override
    public List<String> getTags() {

        return this.tags;
    }


    @Override
    public long getIntervalSeconds() {

        return this.intervalSeconds;
    }

    @Override
    public HealthInfo call() throws Exception {

        try (Connection conn = this.dataSource.getConnection()) {

            if (this.healthQuery == null) {

                return new HealthInfo(HealthStatus.OK,"Datasource ["+this.displayName+"] is OK as per validationQuery.");
            }
            else {

                String queryResult;

                if (log.isDebugEnabled()) {
                    log.debug("Performing healthQuery [{}] for datasource [{}]...",
                        this.healthQuery,this.name);
                }

                try (Statement stmt = conn.createStatement();
                    ResultSet res = stmt.executeQuery(this.healthQuery)  )  {

                    if (res.next()) {

                        queryResult = res.getString(1);

                        if (res.next()) {
                            log.warn("More than one result performing healthQuery [{}] for datasource [{}], first result was [{}]",
                            this.healthQuery,this.name,queryResult);
                            return new HealthInfo(HealthStatus.HEALTH_CHECK_ERROR,"Datasource ["+this.displayName+"] returned more than one result for healthQuery, first result was ["+queryResult+"].");
                        }
                    }
                    else {
                        log.warn("No result performing healthQuery [{}] for datasource [{}]",
                        this.healthQuery,this.name);
                        return new HealthInfo(HealthStatus.HEALTH_CHECK_ERROR,"Datasource ["+this.displayName+"] returned no result for healthQuery.");
                    }
                }

                if (log.isDebugEnabled()) {
                    log.debug("healthQuery [{}] for datasource [{}] returned [{}].",
                        this.healthQuery,this.name,queryResult);
                }

                return new HealthInfo(HealthStatus.OK,"Check for datasource ["+this.displayName+"] returned ["+queryResult+"].");
            }
        }
    }

    @Override
    public String toString() {
        return "JdbcHealthCheck [name=" + this.name +
                    ", tags=" + this.tags +
                    ", healthQuery=" + this.healthQuery +
                    ", intervalSeconds=" + this.intervalSeconds +
                "]";
    }

}
