/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Map.Entry;
import java.util.ServiceLoader;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.sql.DataSource;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.CoreService;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServicePriority;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.svc.api.monitoring.HealthCheck;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.zaxxer.hikari.HikariDataSource;

@ServicePriority(10)
public class JdbcComponent implements Component {

    private static final Logger log = LoggerFactory.getLogger(JdbcComponent.class);

    private static final String PID = "org.clazzes.jdbc.provider";

    private static final long[] START_INTERVALS = new long[] { 5L, 10L, 15L, 30L, 60L };

    private SortedMap<String,DataSourceRecord> dataSources;
    private SortedMap<String,Future<?>> startJobFutures;
    private int nConfiguredDataSources;

    static {

        ServiceLoader<Driver> loader = ServiceLoader.load(JdbcComponent.class.getModule().getLayer(),Driver.class);

        for (Driver driver : loader) {

            log.info("Registering JDBC driver [{}].",driver.getClass().getName());
            try {
                DriverManager.registerDriver(driver);
            } catch (SQLException e) {
                log.error("Failed to register JDBC driver [{}]",driver.getClass().getName(),e);
            }
        }
    }

    private boolean activateDataSource(
                                ServiceRegistry registry,
                                ComponentManager componentManager,
                                String key,
                                HikariDataSource ds,
                                JdbcHealthCheck healthCheck,
                                ConfigWrapper dsConfig)
        throws InterruptedException {

        log.info("Activating datasource [{}] with URL [{}].",key,ds.getJdbcUrl());

        Connection connection = null;

        try {

            if (Thread.interrupted()){
                throw new InterruptedException();
            }

            // initialize the pool with the bundle calls loader imposed
            // as the context class loader.
            // This is the only way to enforce the creation of drier instance
            // from the supplied JDBC driver fragment bundles attached to the
            // jdbc-provider bundles.
            connection = ds.getConnection();

            if (Thread.interrupted()){
                throw new InterruptedException();
            }

            DatabaseMetaData md = connection.getMetaData();

            log.info("Connected datasource [{}] to [{}], version [{}] using driver [{}], version [{}].",
                            key,
                            md.getDatabaseProductName(),
                            md.getDatabaseProductVersion(),
                            md.getDriverName(),
                            md.getDriverVersion()
                            );

            boolean doCommit;

            synchronized(this) {
                this.dataSources.put(key,new DataSourceRecord(ds,healthCheck,dsConfig));
                this.startJobFutures.remove(key);
                registry.addService(key,DataSource.class,ds);

                if (healthCheck != null) {
                    registry.addService(healthCheck.getId(),HealthCheck.class,healthCheck);
                }

                log.info("Successfully activated JDBC datasource [{}] no. [{}/{}].",
                        key,this.dataSources.size(),this.nConfiguredDataSources);

                doCommit = this.dataSources.size() == this.nConfiguredDataSources;
            }

            if (doCommit) {
                componentManager.commit();
            }

            return true;
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch(Throwable e) {

            log.error("Activating datasource [{}] failed",key,e);
            return false;
        }
        finally {

            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    log.warn("Error returning probe connection to pool for datasource ["+key+"]",e);
                }
            }
        }
    }

    private void closeDataSource(ServiceRegistry registry, String key) {

        registry.removeService(key,DataSource.class);
        DataSourceRecord dr = this.dataSources.get(key);

        if (dr.getHealthCheck() != null) {
            registry.removeService(dr.getHealthCheck().getId(),HealthCheck.class);
        }

        log.info("Closing JDBC datasource [{}]...",key);

        try {
            dr.getDataSource().close();
            log.info("Successfully closed JDBC datasource [{}].",key);

        }
        catch(Throwable e) {
            log.error("Error closing JDBC datasource [{}].",key,e);
        }
}

    // to be called from within synchronized contexts.
    private void closeDataSources(ServiceRegistry registry) {

        if (this.startJobFutures != null) {
            for (Entry<String, Future<?>> e : this.startJobFutures.entrySet()) {

                log.info("Cancelling start job for datasource [{}]",e.getKey());
                e.getValue().cancel(true);
            }
        }

        if (this.dataSources != null) {

            for (String key : this.dataSources.keySet()) {

                this.closeDataSource(registry,key);
            }
        }
    }

    private class StartDataSourceRunnable implements Runnable {

        private final String key;
        private final HikariDataSource dataSource;
        private final JdbcHealthCheck healthCheck;
        private final ConfigWrapper dsConfig;
        private final ServiceRegistry registry;
        private final ComponentManager componentManager;
        private final ScheduledExecutorService scheduledExecutorService;

        private int ntry;

        public StartDataSourceRunnable(
                String key,
                HikariDataSource dataSource,
                JdbcHealthCheck healthCheck,
                ConfigWrapper dsConfig,
                ServiceRegistry registry,
                ComponentManager componentManager,
                ScheduledExecutorService scheduledExecutorService) {

            this.key = key;
            this.dataSource = dataSource;
            this.healthCheck = healthCheck;
            this.dsConfig = dsConfig;
            this.registry = registry;
            this.componentManager = componentManager;
            this.scheduledExecutorService = scheduledExecutorService;
            this.ntry = 0;
        }

        private long getNextExecutionDelay() {

            if (this.ntry >= START_INTERVALS.length) {
                return START_INTERVALS[START_INTERVALS.length-1];
            }
            else {
                return START_INTERVALS[this.ntry];
            }
        }

        @Override
        public void run() {

            try {

                ++this.ntry;
                log.error("Retry no. [{}] to connect to database [{}].",this.ntry,this.key);

                if (!JdbcComponent.this.activateDataSource(this.registry,this.componentManager,this.key,this.dataSource,this.healthCheck,this.dsConfig)) {

                    this.scheduledExecutorService.schedule(this,this.getNextExecutionDelay(),TimeUnit.SECONDS);
                }
            } catch (InterruptedException e) {
                log.error("Retry no. [{}] to connect to database [{}] has been interrupted.",
                        this.ntry,this.key,e);
            }

        }
    }

    @Override
    public void start(ServiceContext svcCtxt) throws Exception {

        ServiceRegistry registry = svcCtxt.getService(ServiceRegistry.class).get();
        ConfigurationEngine configurationEngine = svcCtxt.getService(ConfigurationEngine.class).get();
        CoreService cs = svcCtxt.getService(CoreService.class).get();
        ComponentManager componentManager = svcCtxt.getService(ComponentManager.class).get();

        ScheduledExecutorService scheduledExecutorService = cs.getScheduledExecutorService();

        configurationEngine.listen(
            PID,
            (config) -> {
                ConfigWrapper datasources = config.getSubTree("datasource");

                // FIXME maybe keep unchanged datasources like in jdbc-provider
                synchronized(this) {

                    this.closeDataSources(registry);

                    this.dataSources = new TreeMap<String,DataSourceRecord>();
                    this.startJobFutures = new TreeMap<String,Future<?>>();
                    this.nConfiguredDataSources = datasources.keySet().size();

                }
                for (String key : datasources.keySet()) {

                    ConfigWrapper dsConfig = datasources.getSubTree(key);

                    log.info("Creating JDBC datasource [{}]...",key);

                    try {
                        HikariDataSource ds = DataSourceHelper.createDataSource(dsConfig);

                        JdbcHealthCheck healthCheck = JdbcHealthCheck.ofConfig(key,ds,dsConfig);

                        log.info("Successfully created JDBC datasource [{}] with health check [{}].",
                                key,healthCheck);

                        if (!this.activateDataSource(registry,componentManager,key,ds,healthCheck,dsConfig)) {

                            long secs = START_INTERVALS[0];

                            log.info("Unable to connect JDBC datasource [{}] at first glance, retrying in [{}s].",
                                    key,secs);

                            ScheduledFuture<?> future = scheduledExecutorService.schedule(
                                new StartDataSourceRunnable(key,ds,healthCheck,dsConfig,registry,componentManager,scheduledExecutorService),
                                START_INTERVALS[0],TimeUnit.SECONDS);

                            synchronized (this) {
                                this.startJobFutures.put(key, future);
                            }
                        }

                    } catch (Throwable e) {
                        log.error("Error creating JDBC datasource [{}].",key,e);
                    }
                }
            }
        );
    }

    @Override
    public void stop(ServiceContext svcCtxt) throws Exception {

        ServiceRegistry registry = svcCtxt.getService(ServiceRegistry.class).get();

        synchronized(this) {
            this.closeDataSources(registry);
            this.dataSources = null;
            this.startJobFutures = null;
            this.nConfiguredDataSources = 0;
        }
    }

}
