/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.util.Map;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.clazzes.svc.api.ConfigurationHelper;
import org.clazzes.svc.api.CoreService;

public class VarsSupplier implements Supplier<String> {

    public static final Pattern VARIABLE_PATTERN = Pattern.compile("^(env|prop|conf|void|ref):([^:]*)(:-(.*))?$");

    private final String pid;
    private final String context;
    private final Map<String,Object> content;
    private final String key;
    private final String expr;
    private final ThreadLocal<Boolean> cycleCheck;

    public VarsSupplier(String pid, String context, Map<String,Object> content, String key, String expr) {
        this.pid = pid;
        this.context = context;
        this.content = content;
        this.expr = expr;
        this.key = key;
        this.cycleCheck = new ThreadLocal<Boolean>();
    }

    @Override
    public String get() {

        if (this.cycleCheck.get() != null) {
            throw new IllegalArgumentException("Recursive evaluation of ["+this.key+"] in expression ["+this.expr+"].");
        }

        try {

            this.cycleCheck.set(Boolean.TRUE);

            return VariableSubstition.substVars(this.expr,this.key,
            v -> {

                Matcher m = VARIABLE_PATTERN.matcher(v);

                if (m.matches()) {
                    String scheme = m.group(1);
                    String key = m.group(2);

                    String def = m.groupCount() == 4 ? m.group(4) : null;

                    if ("ref".equals(scheme)) {

                        if (def == null) {
                            return ConfigurationHelper.getMandatoryString(content,key);
                        }
                        else {
                            return ConfigurationHelper.getString(content,key,def);
                        }

                    }
                    else {
                        CoreService coreService = CoreServiceImpl.provider();

                        String ret = coreService.getSecret(this.pid,scheme,key);

                        return ret == null ? def : ret;
                    }
                }
                else {
                    throw new IllegalArgumentException("Invalid Variable ["+v+"] (must match ["+VARIABLE_PATTERN+"]).");
                }
            });
        }
        finally {
            cycleCheck.remove();
        }
    }

    public String getPid() {
        return this.pid;
    }

    public String getExpr() {
        return this.expr;
    }

    @Override
    public String toString() {
        return "VarsSupplier [pid=" + this.pid +
               ", context=" + this.context +
               ", key=" + this.key +
               ", expr=" + this.expr + "]";
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((this.pid == null) ? 0 : this.pid.hashCode());
        result = prime * result + ((this.context == null) ? 0 : this.context.hashCode());
        result = prime * result + ((this.key == null) ? 0 : this.key.hashCode());
        result = prime * result + ((this.expr == null) ? 0 : this.expr.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        VarsSupplier other = (VarsSupplier) obj;
        if (this.pid == null) {
            if (other.pid != null)
                return false;
        } else if (!this.pid.equals(other.pid))
            return false;
        if (this.context == null) {
            if (other.context != null)
                return false;
        } else if (!this.context.equals(other.context))
            return false;
        if (this.key == null) {
            if (other.key != null)
                return false;
        } else if (!this.key.equals(other.key))
            return false;
        if (this.expr == null) {
            if (other.expr != null)
                return false;
        } else if (!this.expr.equals(other.expr))
            return false;
        return true;
    }


}
