/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>This class loads the configuration from
 * <code>/etc/svc-runner/system.properties</code>
 */
public class VariableSubstition {

    // length of '${'
    private static final int DELIM_START_LEN = 2;
    // length of '}'
    private static final int DELIM_STOP_LEN = 1;
    private static final Pattern DELIM_START = Pattern.compile("(?<!\\\\)\\$\\{");
    private static final Pattern DELIM_STOP  = Pattern.compile("(?<!\\\\)\\}");
    private static final String DOLLAR_REPL = Matcher.quoteReplacement("$");

    /**
     * <p>
     * This method performs property variable substitution on the
     * specified value. If the specified value contains the syntax
     * <code>${prop-name}</code>, where <code>prop-name</code>
     * is resolved by the given accessor,
     * the corresponding property value is substituted for the variable
     * placeholder.</p>
     * <p>
     * Multiple variable placeholders may exist in the
     * specified value as well as nested variable placeholders, which
     * are substituted from inner most to outer most. Configuration
     * properties override system properties.
     * </p>
     * @param val The string on which to perform property substitution.
     * @param currentKey The key of the property being evaluated used to
     *        detect cycles.
     * @param cycleMap Map of variable references used to detect nested cycles.
     * @param accessor Function resolving placeholders by
     * @return The value of the specified string after system property substitution.
     * @throws IllegalArgumentException If there was a syntax error in the
     *         property placeholder syntax or a recursive variable reference.
    **/
    private static String substVars(String val, String currentKey,
        Map<String,String> cycleMap, Function<String,String> accessor)
        throws IllegalArgumentException
    {
        // Put the current key in the cycle map.
        cycleMap.put(currentKey, currentKey);

        // Assume we have a value that is something like:
        // "leading ${foo.${bar}} middle ${baz} trailing"

        // Find the first ending '}' variable delimiter, which
        // will correspond to the first deepest nested variable
        // placeholder.
        int stopDelim = -1;
        int startDelim = -1;

        Matcher stopMatcher = DELIM_STOP.matcher(val);
        Matcher startMatcher = DELIM_START.matcher(val);

        do {
            // If there is no stopping delimiter, then just return
            // the value since there is no variable declared.
            if (!stopMatcher.find()) {
                return val;
            }
            stopDelim = stopMatcher.start();


            // Try to find the matching start delimiter by
            // looping until we find a start delimiter that is
            // greater than the stop delimiter we have found.

            // If there is no starting delimiter, then just return
            // the value since there is no variable declared.
            if (!startMatcher.find(0)) {
                return val;
            }

            // first start delimiter
            startDelim = startMatcher.start();

            while (startMatcher.find() &&
                   startMatcher.start() < stopDelim) {

                // found a start delimiter even nearer to end delimiter
                startDelim = startMatcher.start();
            }
        }
        while ((startDelim > stopDelim) && (stopDelim >= 0));

        // At this point, we have found a variable placeholder so
        // we must perform a variable substitution on it.
        // Using the start and stop delimiter indices, extract
        // the first, deepest nested variable placeholder.
        String variable =
            val.substring(startDelim+DELIM_START_LEN, stopDelim);

        // Verify that this is not a recursive variable reference.
        if (cycleMap.get(variable) != null)
        {
            throw new IllegalArgumentException(
                "Recursive reference to variable [" + variable + "].");
        }

        // Get the value of the variable placeholder.
        String substValue = accessor.apply(variable);

        if (substValue == null) {
            throw new IllegalArgumentException(
                "Reference to unknown variable [" + variable + "].");
        }

        // Remove the found variable from the cycle map, since
        // it may appear more than once in the value and we don't
        // want such situations to appear as a recursive reference.
        cycleMap.remove(variable);

        // Append the leading characters, the substituted value of
        // the variable, and the trailing characters to get the new
        // value.
        val = val.substring(0, startDelim)
            + substValue
            + val.substring(stopDelim+DELIM_STOP_LEN,val.length());

        // Now perform substitution again, since there could still
        // be substitutions to make.
        val = substVars(val, currentKey, cycleMap, accessor);

        // Return the value.
        return val;
    }

    /**
     * <p>
     * This method performs property variable substitution on the
     * specified value. If the specified value contains the syntax
     * <code>${prop-name}</code>, where <code>prop-name</code>
     * is resolved by the given accessor,
     * the corresponding property value is substituted for the variable
     * placeholder.</p>
     * <p>
     * Multiple variable placeholders may exist in the
     * specified value as well as nested variable placeholders, which
     * are substituted from inner most to outer most. Configuration
     * properties override system properties.
     * </p>
     * @param val The string on which to perform property substitution.
     * @param currentKey The key of the property being evaluated used to
     *        detect cycles.
     * @param accessor Function resolving placeholders.
     * @return The value of the specified string after system property substitution.
     * @throws IllegalArgumentException If there was a syntax error in the
     *         property placeholder syntax or a recursive variable reference.
    **/
    public static String substVars(String val, String currentKey,
        Function<String,String> accessor)
        throws IllegalArgumentException
    {
        if (val == null) {
            throw new IllegalArgumentException("No value found for key ["+currentKey+"].");
        }

        return substVars(val,currentKey,new HashMap<String,String>(),accessor)
                .replaceAll("\\\\\\{","{")
                  .replaceAll("\\\\\\}","}")
                    .replaceAll("\\\\\\$",DOLLAR_REPL)
                      .replaceAll("\\\\\\\\","\\\\");
    }

    /**
     * <p>
     * This method performs property variable substitution on the
     * specified value. If the specified value contains the syntax
     * <code>${prop-name}</code>, where <code>prop-name</code>
     * refers to either a configuration property or a system property,
     * the corresponding property value is substituted for the variable
     * placeholder. Multiple variable placeholders may exist in the
     * specified value as well as nested variable placeholders, which
     * are substituted from inner most to outer most. Configuration
     * properties override system properties.
     * </p>
     * @param currentKey The key of the property being evaluated.
     * @param configProps Set of configuration properties.
     * @return The value of the specified string after system property substitution.
     * @throws IllegalArgumentException If there was a syntax error in the
     *         property placeholder syntax or a recursive variable reference.
     */
    public static String substConfigProps(String currentKey,
                                          Properties configProps)
        throws IllegalArgumentException {

        return substVars(configProps.getProperty(currentKey),
                currentKey,
                (k) -> {

                    // Try to get configuration properties first.
                    String ret = configProps.getProperty(k);

                    if (ret == null) {
                        // Ignore unknown property values.
                        return System.getProperty(k,"");
                    }
                    return ret;
                });
    }

    /**
     * <p>
     * This method performs property variable substitution on the
     * specified value. If the specified value contains the syntax
     * <code>${prop-name}</code>, where <code>prop-name</code>
     * refers to a system property,
     * the corresponding property value is substituted for the variable
     * placeholder.</p>
     * <p>
     * Multiple variable placeholders may exist in the
     * specified value as well as nested variable placeholders, which
     * are substituted from inner most to outer most. Configuration
     * properties override system properties.
     * </p>
     * @param val The string on which to perform property substitution.
     * @param currentKey The key of the property being evaluated used to
     *        detect cycles.
     * @return The value of the specified string after system property substitution.
     * @throws IllegalArgumentException If there was a syntax error in the
     *         property placeholder syntax or a recursive variable reference.
     */
    public static String substSystemProperties(String val, String currentKey)
        throws IllegalArgumentException {

        return substVars(val,currentKey,
                k -> System.getProperty(k,""));
    }
}
