/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.util.ArrayList;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;

import org.clazzes.svc.api.ThrowableInfo;
import org.slf4j.Logger;
import org.slf4j.helpers.FormattingTuple;
import org.slf4j.helpers.MessageFormatter;

/**
 * Base class for all throwable capturing listener contexts.
 */
public class ThrowableBucket {

    private final int maxSize;
    private Deque<ThrowableInfo> throwables;

    public ThrowableBucket() {
        this.maxSize = Config.getThrowableHistorySize();
    }

    /**
     * Record an exception in this context.
     * @param ti The throwable to record.
     */
    public synchronized void addThrowable(ThrowableInfo ti) {
        if (this.throwables == null) {
            this.throwables = new LinkedList<ThrowableInfo>();
        }
        else if (this.throwables.size() >= this.maxSize) {

            this.throwables.removeLast();
        }

        this.throwables.addFirst(ti);
    }

    /**
     * Record and log an exception to this context.
     * @param logger The slf4j logger to log the error to.
     * @param msg The message with slf4j placeholders. For recording exceptions,
     *            the number of placeholders must be one less than the number
     *            of arguments.
     * @param args The arguments to the logger, the last argument is
     *             the exception to be recorded.
     * @return The throwable information POJO for further processing.
     */
    public ThrowableInfo error(Logger logger, String msg, Object... args) {

        FormattingTuple fmt = MessageFormatter.arrayFormat(msg,args);

        logger.error(fmt.getMessage(),fmt.getThrowable());

        ThrowableInfo ti = new ThrowableInfo(fmt.getMessage(),fmt.getThrowable());

        this.addThrowable(ti);

        return ti;
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    /**
     * @return The copy of currently stored throwables.
     */
    public synchronized List<ThrowableInfo> getThrowables() {
        return this.throwables == null ? null : new ArrayList<ThrowableInfo>(this.throwables);
    }

}
