/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ServiceInfo;
import org.clazzes.svc.api.ServiceKey;
import org.clazzes.svc.api.ServiceRegistry;
import org.clazzes.svc.api.ThrowableInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceRegistryImpl extends HasContext implements ServiceRegistry {

    private static final Logger log = LoggerFactory.getLogger(ServiceRegistryImpl.class);

    private static final class ServiceHolder<T> extends ThrowableBucket {

        private final T service;

        public ServiceHolder(T service) {
            this.service = service;
        }

        public T getService() {
            return this.service;
        }
    }

    private static final class InstanceMap {

        private final Map<Class<?>,Map<String,?>> map;

        public InstanceMap() {
            this.map = new ConcurrentHashMap<Class<?>,Map<String,?>>();
        }

        public Set<Class<?>> keySet() {
            return this.map.keySet();
        }

        @SuppressWarnings("unchecked")
        public <T> Map<String,ServiceHolder<T>> get(Class<T> iface) {

            return (Map<String,ServiceHolder<T>>)this.map.get(iface);
        }

        @SuppressWarnings("unchecked")
        public <T> Map<String,ServiceHolder<T>> getOrCreate(Class<T> iface) {

            return (Map<String, ServiceHolder<T>>)
                this.map.computeIfAbsent(iface,
                                k -> new ConcurrentHashMap<String,Object>());
        }
    }

    private final InstanceMap registry;

    private static final class ClassListeners<T> {
        public final BiConsumer<String,T> added;
        public final BiConsumer<String,T> removed;

        public ClassListeners(BiConsumer<String,T> added, BiConsumer<String,T> removed) {
            this.added = added;
            this.removed = removed;
        }
    }

    private static final class InstanceListeners<T> {
        public final Consumer<T> added;
        public final Consumer<T> removed;

        public InstanceListeners(Consumer<T> added, Consumer<T> removed) {
            this.added = added;
            this.removed = removed;
        }
    }

    private final ConcurrentHashMap<Class<?>,List<ClassListeners<?>>> classListeners;
    private final ConcurrentHashMap<ServiceKey<?>,List<InstanceListeners<?>>> instanceListeners;

    public ServiceRegistryImpl() {
        this.registry = new InstanceMap();
        this.classListeners = new ConcurrentHashMap<Class<?>,List<ClassListeners<?>>>();
        this.instanceListeners = new ConcurrentHashMap<ServiceKey<?>,List<InstanceListeners<?>>>();
    }

    public void start(ServiceContext svcCtxt) {

        this.setContext(svcCtxt);
    }

    public void stop() {

        this.clearContext();
    }

    private <T> void callClassListener(
                        ServiceKey<T> ikey, ServiceHolder<T> service,
                        String label, BiConsumer<String,T> listener) {

        this.getContext().scheduleManipulator(() -> {
            try  {
                log.info("Calling '{}' class listener [{}] on {}",label,listener,ikey);
                listener.accept(ikey.getKey(),service.getService());
            }
            catch (Throwable e) {
                ThrowableInfo ti =
                    service.error(log,"Error calling '{}' class listener [{}] on {}",label,listener,ikey,e);

                this.getContext().getService(ComponentManager.class).ifPresent(
                    cm -> cm.recordServiceException(ti)
                );
            }
        });
    }

    private <T> void callInstanceListener(
                        ServiceKey<T> ikey,ServiceHolder<T> service,
                        String label, Consumer<T> listener) {

        this.getContext().scheduleManipulator(() -> {
            try  {
                log.info("Calling '{}' listener [{}] on {}",label,listener,ikey);
                listener.accept(service.getService());
            }
            catch (Throwable e) {
                ThrowableInfo ti =
                    service.error(log,"Error calling '{}' listener [{}] on {}",label,listener,ikey,e);

                this.getContext().getService(ComponentManager.class).ifPresent(
                    cm -> cm.recordServiceException(ti)
                );
            }
        });
    }

    private <T> List<InstanceListeners<?>> getInstanceListeners(ServiceKey<T> ikey) {

        return this.instanceListeners.computeIfAbsent(ikey,
                k -> new ArrayList<InstanceListeners<?>>());
    }

    private <T> List<ClassListeners<?>> getClassListeners(Class<T> iface) {
        return this.classListeners.computeIfAbsent(iface,
                k->new ArrayList<ClassListeners<?>>());
    }

    @Override
    public <T> void addService(String key, Class<T> iface, T service) {

        ServiceKey<T> ikey = new ServiceKey<T>(key,iface);

        Map<String,ServiceHolder<T>> iregistry = this.registry.getOrCreate(iface);

        if (iregistry.containsKey(key)) {
            throw new IllegalStateException("Duplicate "+ikey+" is already registered.");
        }

        List<InstanceListeners<?>> ils = this.getInstanceListeners(ikey);
        List<ClassListeners<?>> cls = this.getClassListeners(iface);

        synchronized (this) {

            ServiceHolder<T> h = new ServiceHolder<>(service);
            iregistry.put(key,h);

            log.info("Registered "+ikey);

            for (InstanceListeners<?> il : ils) {

                @SuppressWarnings("unchecked")
                InstanceListeners<T> til = (InstanceListeners<T>)il;

                this.callInstanceListener(ikey,h,"added",til.added);
            }

            for (ClassListeners<?> cl:cls) {

                @SuppressWarnings("unchecked")
                ClassListeners<T> tcl = (ClassListeners<T>)cl;

                this.callClassListener(ikey,h,"added",tcl.added);
            }
        }

    }

    @Override
    public <T> boolean removeService(String key, Class<T> iface) {

        ServiceKey<T> ikey = new ServiceKey<T>(key,iface);

        Map<String,ServiceHolder<T>> iregistry = this.registry.get(iface);

        if (iregistry == null) {
            log.warn("No service has been registered for interface ["+iface+"], will not call remove listeners.");
            return false;
        }

        List<InstanceListeners<?>> ils = this.getInstanceListeners(ikey);
        List<ClassListeners<?>> cls = this.getClassListeners(iface);

        synchronized(this) {

            ServiceHolder<T> service = iregistry.remove(key);

            if (service == null) {
                log.warn(ikey+" is not registered, will not call remove listeners.");
                return false;
            }

            log.info("Unregistered "+ikey);

            for (InstanceListeners<?> il : ils) {

                @SuppressWarnings("unchecked")
                InstanceListeners<T> til = (InstanceListeners<T>)il;

                this.callInstanceListener(ikey,service,"removed",til.removed);
            }

            for (ClassListeners<?> cl:cls) {

                @SuppressWarnings("unchecked")
                ClassListeners<T> tcl = (ClassListeners<T>)cl;

                this.callClassListener(ikey,service,"removed",tcl.removed);
            }
        }
        return true;
    }

    @Override
    public <T> Optional<T> getService(String key, Class<T> iface) {

        Map<String,ServiceHolder<T>> instances = this.registry.get(iface);

        if (instances != null) {

            ServiceHolder<T> instance = instances.get(key);

            if (instance != null) {
                return Optional.of(instance.getService());
            }
        }

        return Optional.empty();
    }

    protected <T> Optional<ServiceHolder<T>> getServiceHolder(String key, Class<T> iface) {

        Map<String,ServiceHolder<T>> instances = this.registry.get(iface);

        if (instances != null) {

            ServiceHolder<T> instance = instances.get(key);

            if (instance != null) {
                return Optional.of(instance);
            }
        }

        return Optional.empty();
    }

    @Override
    public <T> Map<String,T> getAll(Class<T> iface) {

        Map<String,ServiceHolder<T>> instances = this.registry.get(iface);

        if (instances == null) {
            return Collections.emptyMap();
        }
        else {
            Map<String,T> ret = new HashMap<String,T>(instances.size());
            instances.entrySet().forEach(e -> ret.put(e.getKey(),(T)e.getValue().getService()));
            return ret;
        }
    }

    protected <T> Map<String,ServiceHolder<T>> getAllHolders(Class<T> iface) {

        Map<String,ServiceHolder<T>> instances = this.registry.get(iface);

        if (instances == null) {
            return Collections.emptyMap();
        }
        else {
            return new HashMap<String,ServiceHolder<T>>(instances);
        }
    }

    @Override
    public <T> Optional<T> getAny(Class<T> iface) {

        Map<String,ServiceHolder<T>> instances = this.registry.get(iface);

        if (instances != null) {

            Iterator<ServiceHolder<T>> it = instances.values().iterator();
            if (it.hasNext()) {
                return Optional.of(it.next().getService());
            }
        }

        return Optional.empty();
    }

    @Override
    public <T> AutoCloseable listen(String key, Class<T> iface, Consumer<T> added, Consumer<T> removed) {

        ServiceKey<T> ikey = new ServiceKey<T>(key,iface);

        List<InstanceListeners<?>> ils = this.getInstanceListeners(ikey);

        InstanceListeners<T> listeners = new InstanceListeners<T>(added,removed);

        synchronized(this) {
            // call listeners for existing instances first.
            this.getServiceHolder(key,iface).ifPresent(
                service -> this.callInstanceListener(ikey,service,"added",added));
            ils.add(listeners);
        }

        return () -> {
            synchronized (this) {
                ils.remove(listeners);
                this.getServiceHolder(key,iface).ifPresent(
                    service -> this.callInstanceListener(ikey,service,"removed",removed));
            }
        };
    }

    @Override
    public <T> AutoCloseable listenAll(Class<T> iface, BiConsumer<String, T> added, BiConsumer<String, T> removed) {

        List<ClassListeners<?>> cls = this.getClassListeners(iface);

        ClassListeners<T> listeners = new ClassListeners<T>(added,removed);

        synchronized(this) {
            // call listeners for existing instances first.
            this.getAllHolders(iface).forEach(
                (key,service) -> {
                    ServiceKey<T> ikey = new ServiceKey<T>(key,iface);
                    this.callClassListener(ikey,service,"added",added);
                }
            );
            cls.add(listeners);
        }

        return () -> {
            synchronized(this) {
                cls.remove(listeners);
                this.getAllHolders(iface).forEach(
                    (key,service) -> {
                        ServiceKey<T> ikey = new ServiceKey<T>(key,iface);
                        this.callClassListener(ikey,service,"removed",removed);
                    }
                );
            }
        };
    }

    @Override
    public List<ServiceInfo> listServices() {

        return this.listServices(null);
    }

    protected final <T> void appendServiceInfos(List<ServiceInfo> services, Class<T> key) {

        Map<String,ServiceHolder<T>> instances = this.registry.get(key);

        for (Entry<String,ServiceHolder<T>> ee : instances.entrySet()) {

            String implementation;

            Object target = ee.getValue().getService();

            if (Proxy.isProxyClass(target.getClass())) {
                implementation = Proxy.getInvocationHandler(target).toString();
            }
            else {
                implementation = target.getClass().getName();
            }

            services.add(new ServiceInfo(key.getName(),ee.getKey(),
                            implementation,
                            ee.getValue().getThrowables()));
        }
    }

    @Override
    public List<ServiceInfo> listServices(String iface) {

        List<ServiceInfo> services = new ArrayList<ServiceInfo>(256);

        for (Class<?> key : this.registry.keySet()) {

            if (iface != null && !iface.equals(key.getName())) {
                continue;
            }

            this.appendServiceInfos(services,key);
        }

        Collections.sort(services);

        return services;
    }

}
