/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ConfigPidInfo;
import org.clazzes.svc.api.ConfigWrapper;
import org.clazzes.svc.api.ConfigurationEngine;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ThrowableInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationEngineImpl extends HasContext implements ConfigurationEngine {

    private static final String WATCH_CONFIG_FOLDER_PROPERTY = "svc.runner.watchConfigFolder";

    static final Logger log = LoggerFactory.getLogger(ConfigurationEngineImpl.class);

    private static final class ConfigSet extends ThrowableBucket {
        public final String location;
        public final ConfigWrapper config;

        public ConfigSet(String location, ConfigWrapper config) {
            this.location = location;
            this.config = config;
        }
    }

    private final Path admPath;
    private final Path admPathOverride;
    private final Map<String, ConfigSet> configSetsByPid;
    private final Map<String, List<Consumer<ConfigWrapper>>> listeners;

    private Future<?> scanFuture;

    public ConfigurationEngineImpl() {

        this.configSetsByPid = new ConcurrentHashMap<String, ConfigSet>();
        this.listeners = new ConcurrentHashMap<String, List<Consumer<ConfigWrapper>>>();

        Path etcDir = Config.getEtcDir();

        if (etcDir == null) {
            log.info("svc-runner etc path not configured, will not read properties from adm.d folder.");
            this.admPath = null;
            this.admPathOverride = null;
        } else {
            this.admPath = etcDir.resolve("adm.d");
            log.info("Will scan *.yaml from folder [{}].", this.admPath);

            this.admPathOverride = etcDir.resolve("adm.d.override");
            log.info("Will scan *.yaml from folder [{}].", this.admPathOverride);
        }
    }

    /**
     * Start the config engine in server mode.
     *
     * @param svcCtxt The service context we operate on.
     */
    void start(ServiceContext svcCtxt) {

        this.setContext(svcCtxt);
        try {
            this.loadAdmPathFiles();
        } catch (IOException e) {
            log.error("I/O error scanning [{}]", this.admPath, e);
        }

        boolean doWatch = Config.getBooleanProperty(WATCH_CONFIG_FOLDER_PROPERTY, false);

        if (doWatch) {
            this.scanAdmPath();
        }
    }

    public void stop() {

        if (this.scanFuture != null) {
            log.info("Intrerrupting scan of folder [{}].", this.admPath);

            this.scanFuture.cancel(true);
        }

        this.clearContext();
    }

    /**
     * Load a yaml configuration and start this engine for test purposes.
     *
     * @param is  The inpust stream to read.
     * @param res the resource name used for logging and error messages.
     * @throws IOException Upon read errors.
     */
    public void loadTestYaml(ServiceContext svcCtxt, InputStream is, String res) throws IOException {

        this.setContext(svcCtxt);
        log.info("Loading config from resource [{}] for unit test purposes.", res);
        this.loadYamlResource(is, res);
    }

    private void scanAdmPath() {

        this.scanFuture = CoreServiceImpl.INSTANCE.getExecutorService().submit(() -> {

            try (WatchService watcher = FileSystems.getDefault().newWatchService()) {

                log.info("Starting to watch folder [{}].",this.admPath);

                this.admPath.register(watcher,
                        StandardWatchEventKinds.ENTRY_CREATE,
                        StandardWatchEventKinds.ENTRY_MODIFY,
                        StandardWatchEventKinds.ENTRY_DELETE);

                if (Files.isDirectory((this.admPathOverride))) {

                    log.info("Starting to watch folder [{}].",this.admPathOverride);

                    this.admPathOverride.register(watcher,
                            StandardWatchEventKinds.ENTRY_CREATE,
                            StandardWatchEventKinds.ENTRY_MODIFY,
                            StandardWatchEventKinds.ENTRY_DELETE);
                }
                else {
                   log.warn("Folder [{}] does not exist, not watching it.",this.admPathOverride);

                }

                WatchKey key;
                while ((key = watcher.take()) != null) {

                    Path folder;

                    if (key.watchable() instanceof Path p) {
                        folder = p;
                    }
                    else {
                        log.warn("Watch key [{}] is not associated with a known folder.",key);
                        continue;
                    }

                    Set<Path> changedYamls = new HashSet<Path>();

                    for (WatchEvent<?> event : key.pollEvents()) {
                        WatchEvent.Kind<?> kind = event.kind();

                        if (kind == StandardWatchEventKinds.OVERFLOW) {
                            log.warn("Watch key for folder [{}] reported overflow.",folder);
                            continue;
                        }

                        @SuppressWarnings("unchecked")
                        WatchEvent<Path> ev = (WatchEvent<Path>) event;
                        Path filename = ev.context();

                        String fn = filename.getFileName().toString();
                        boolean isYaml = fn.endsWith(".yaml");

                        if (kind == StandardWatchEventKinds.ENTRY_DELETE) {

                            if (isYaml) {
                                log.warn(
                                    "YAML file [{}] in folder [{}] has been deleted, affected configurations will disappear in next program run.",
                                    filename,folder);
                            }
                            else {
                                if (log.isDebugEnabled()) {
                                    log.debug(
                                        "Non-YAML file [{}] in folder [{}] has been deleted.",
                                        filename,folder);
                                }
                            }
                        } else {

                            if (isYaml) {

                                Path yamlFile = folder.resolve(filename);
                                log.info("Received event [{}] for yaml file [{}].",
                                         kind,yamlFile);
                                changedYamls.add(yamlFile);
                            } else {
                                log.warn("Ignoring file [{}] in folder [{}], which is not *.yaml.",
                                        filename,folder);
                            }
                        }

                    }

                    for (Path yamlFile:changedYamls) {
                        this.loadAdmPathYamlFile(yamlFile);
                    }

                    if (!key.reset()) {
                        log.warn("Watch key for folder [{}] became invalid, leaving watcher loop.",folder);
                        break;
                    }
                }
            } catch (InterruptedException e) {
                log.info("Leaving watcher for folder [{}].",this.admPath);
                if (log.isDebugEnabled()) {
                    log.debug("Watcher for folder [{}] interrupted exception",
                              this.admPath,e);
                }
            } catch (Throwable e) {
                log.error("Error watching folder [{}]", this.admPath, e);
            }

        });
    }

    private void callListener(Consumer<ConfigWrapper> listener, String pid, ConfigSet configSet) {

        this.getContext().scheduleManipulator(() -> {
            try {
                log.info("Calling listener [{}] on PID {}", listener, pid);

                listener.accept(configSet.config);
            } catch (Throwable e) {

                ThrowableInfo ti =
                   configSet.error(log,"Error calling listener [{}] on PID {}",listener,pid,e);

                this.getContext().getService(ComponentManager.class).ifPresent(
                    cm -> cm.recordConfigException(ti)
                );
            }
        });
    }

    private void callListeners(String pid, ConfigSet configSet) {

        List<Consumer<ConfigWrapper>> pidListeners = this.listeners.get(pid);

        if (pidListeners == null) {
            return;
        }

        for (Consumer<ConfigWrapper> listener : pidListeners) {

            this.callListener(listener,pid,configSet);
        }
    }

    private void loadYamlResource(InputStream is, String res) throws IOException {

        Map<String, Map<String, ?>> configs = ConfigurationFiles.readYaml(is, res);

        for (Entry<String, Map<String, ?>> e : configs.entrySet()) {

            String pid = e.getKey();
            Map<String, ?> config = e.getValue();

            ConfigWrapper cw = new ConfigWrapper(pid, config);

            ConfigSet configSet = new ConfigSet(res,cw);

            ConfigSet old = this.configSetsByPid.put(pid,configSet);

            if (old != null) {

                if(old.config.getContent().equals(config)) {

                    log.info("Config PID [{}] did not change, not calling listeners.",pid);
                    continue;
                }
                else {
                    log.info("Config PID [{}] changed,calling listeners again.",pid);
                }
            }

            this.callListeners(pid,configSet);
        }
    }

    private void loadAdmPathYamlFile(Path p) {
        try (InputStream is = Files.newInputStream(p)) {

            this.loadYamlResource(is, p.toString());

        } catch (IOException e) {
            log.error("Error reading yaml file [" + p + "]", e);
        }
    }

    private void loadAdmDirectory(Path dir) throws IOException {

        if (Files.isDirectory(dir)) {
            log.info("Scanning config folder [{}] for *.yaml file...",dir);

            AtomicInteger nyaml = new AtomicInteger();

            Files.newDirectoryStream(dir, "*.yaml").forEach(p -> {
                this.loadAdmPathYamlFile(p);
                nyaml.incrementAndGet();
            });

             log.info("Loaded [{}] yaml files from config folder [{}].",
                      nyaml,dir);
        }
        else {
            log.warn("Skipping config folder [{}], which is not a directory.",dir);
        }
    }

    private void loadAdmPathFiles() throws IOException {

        if (this.admPath != null) {
            this.loadAdmDirectory(this.admPath);
        }

        if (this.admPathOverride != null) {
            this.loadAdmDirectory(this.admPathOverride);
        }
    }

    @Override
    public AutoCloseable listen(String pid, Consumer<ConfigWrapper> updated) {

        List<Consumer<ConfigWrapper>> pidListeners = this.listeners.computeIfAbsent(pid,
                k -> new CopyOnWriteArrayList<Consumer<ConfigWrapper>>());

        pidListeners.add(updated);

        ConfigSet configSet = this.configSetsByPid.get(pid);

        if (configSet != null) {
            callListener(updated,pid,configSet);
        }

        return () -> {
            pidListeners.remove(updated);
        };
    }

    @Override
    public List<ConfigPidInfo> listPids() {

        Set<String> pids = new HashSet<String>();

        pids.addAll(this.configSetsByPid.keySet());
        pids.addAll(this.listeners.keySet());

        List<String> sorted = new ArrayList<String>(pids);
        Collections.sort(sorted);

        List<ConfigPidInfo> ret = new ArrayList<ConfigPidInfo>();

        for (String pid : sorted) {

            ConfigSet configSet = this.configSetsByPid.get(pid);
            List<Consumer<ConfigWrapper>> pidListeners = this.listeners.get(pid);

            ret.add(new ConfigPidInfo(pid,
                    configSet == null ? null : configSet.config.getContent().size(),
                    pidListeners == null ? null : pidListeners.size(),
                    configSet == null ? null : configSet.location,
                    configSet == null ? null : configSet.getThrowables()));
        }

        return ret;
    }

    @Override
    public Optional<ConfigWrapper> getPid(String pid) {

        ConfigSet configSet = this.configSetsByPid.get(pid);

        if (configSet == null) {
            return Optional.empty();
        } else {
            return Optional.of(configSet.config);
        }
    }

}
