/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.lang.StackWalker.Option;
import java.lang.module.ModuleDescriptor;
import java.lang.module.ModuleReference;
import java.lang.module.ResolvedModule;
import java.net.URI;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.clazzes.svc.api.Component;
import org.clazzes.svc.api.ComponentInfo;
import org.clazzes.svc.api.ComponentLayerInfo;
import org.clazzes.svc.api.ComponentManager;
import org.clazzes.svc.api.ComponentState;
import org.clazzes.svc.api.ModuleInfo;
import org.clazzes.svc.api.ServiceContext;
import org.clazzes.svc.api.ThrowableInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ComponentManagerImpl extends HasContext implements ComponentManager {

    private static final Logger log = LoggerFactory.getLogger(ComponentManagerImpl.class);

    private final Map<String,ComponentLayer> layers;
    private final TreeMap<String,ComponentLayer> sortedLayers;
    private final Map<ModuleLayer,ComponentLayer> layersByModuleLayer;
    private final Map<String,ComponentHolder> componentsByModuleName;

    public ComponentManagerImpl() {
        this.layers = new HashMap<String,ComponentLayer>();
        this.sortedLayers = new TreeMap<String,ComponentLayer>();
        this.layersByModuleLayer = new ConcurrentHashMap<ModuleLayer,ComponentLayer>();
        this.componentsByModuleName = new ConcurrentHashMap<String,ComponentHolder>();
    }


    protected void addLayer(ComponentLayer layer) {

        synchronized(this) {

            this.layers.put(layer.getLabel(),layer);
            this.sortedLayers.put(layer.getKey(),layer);

            layer.getComponentsByModuleName().forEach(
                e -> {
                    ComponentHolder old =
                        this.componentsByModuleName.put(e.getKey(),e.getValue());

                    if (old != null) {
                        log.warn("Module name [{}] defines more than one component apart from [{}]",
                                 e.getKey(),old);
                    }
                }
            );
        }

        this.layersByModuleLayer.put(layer.getModuleLayer(),layer);
    }

    synchronized protected ComponentLayer fetchLayer(String lbl) {
        return this.layers.get(lbl);
    }

    protected ComponentLayer removeLayer(String lbl) {

        ComponentLayer ret;

        synchronized(this) {
            ret = this.layers.remove(lbl);

            if (ret != null) {

                this.sortedLayers.remove(ret.getKey());

                ret.getComponentsByModuleName().forEach(
                    e -> this.componentsByModuleName.remove(e.getKey())
                );
            }
        }

        if (ret != null) {
            this.layersByModuleLayer.remove(ret.getModuleLayer());
        }

        return ret;
    }

    protected List<Map.Entry<String,ComponentLayer>> fetchLayerList() {

        List<Map.Entry<String,ComponentLayer>> ret;

        synchronized (this) {
            ret =  new ArrayList<Map.Entry<String,ComponentLayer>>(this.layers.entrySet());
        }

        ret.sort((a,b) -> a.getValue().getKey().compareTo(b.getValue().getKey()));

        return ret;
    }

    protected List<Map.Entry<String,ComponentLayer>> fetchLayerList(ComponentFilter filter) {

        if (filter == null || filter.getLayer() == null) {
            return this.fetchLayerList();
        }
        else {

            ComponentLayer layer = this.fetchLayer(filter.getLayer());

            if (layer == null) {
                log.warn("Layer [{}] not found, returning empty component list.",layer);
                return Collections.emptyList();
            }
            else {
                return Collections.singletonList(new AbstractMap.SimpleEntry<String,ComponentLayer>(filter.getLayer(),layer));
            }
        }
    }

    protected ComponentLayer createLayer(LayerConfig layerConfig) {

        ModuleLayer parent;

        if (layerConfig.getParent() == null) {
            parent = ModuleLayer.boot();
        }
        else {
            ComponentLayer parentLayer =
                this.fetchLayer(layerConfig.getParent());

            if (parentLayer == null) {
                throw new IllegalStateException("Cannot find parent layer ["+layerConfig.getParent()+"] for ["+layerConfig.getKey()+"]");
            }

            parent = parentLayer.getModuleLayer();
        }

        log.info("Resolving layer [{}] for path [{}]",layerConfig.getKey(),layerConfig.getModulePath());
        ComponentLayer layer = ComponentLayer.of(layerConfig,parent,layerConfig.getModulePath());
        this.addLayer(layer);

        return layer;
    }

    public void startBootLayers(ServiceContext svcCtxt) {

        if (this.fetchLayer(LayerConfig.BOOT_LABEL) != null) {
            throw new IllegalStateException("ComponentManagerImpl.startBootLayers() called twice.");
        }

        ComponentLayer boot =
            ComponentLayer.of(ModuleLayer.boot(),
                              System.getProperty("jdk.module.path"));

        log.info("Resolving layer [{}] for path [{}] with opens {}",
                boot.getLabel(),boot.getPath(),boot.getLayerConfig().getOpens());

        this.addLayer(boot);

        for (Entry<String,LayerConfig> e : Config.getSortedLayerConfigs().entrySet()) {

            this.createLayer(e.getValue());
        }

        for (Entry<String,ComponentLayer> e : this.sortedLayers.entrySet()) {

            log.info("Starting layer [{}].",e.getKey());
            e.getValue().startAll(svcCtxt);
        }

        this.setContext(svcCtxt);
    }

    public void stopLayers(long stopTimeout) {

        ServiceContext svcCtxt = this.getContext();

        for (String key : this.sortedLayers.descendingKeySet()) {
            log.info("Deactivating layer [{}].",key);
            ComponentLayer layer = this.sortedLayers.get(key);

            layer.stopAll(svcCtxt,stopTimeout);
        }

        this.clearContext();
    }

    @Override
    public List<ComponentLayerInfo> listLayers() {

        List<Entry<String,ComponentLayer>> layerList = this.fetchLayerList();

        List<ComponentLayerInfo> ret = new ArrayList<ComponentLayerInfo>(layerList.size());

        for (Entry<String, ComponentLayer> e : layerList) {

            ret.add(new ComponentLayerInfo(e.getKey(),e.getValue().getPath(),e.getValue().getParentLabel()));
        }

        return ret;
    }

    protected static void listModulesOfLayer(List<ModuleInfo> ret, String layerLbl, ComponentLayer layer) {

        for (ResolvedModule m : layer.getModuleLayer().configuration().modules()) {

            ModuleReference ref = m.reference();

            ret.add(new ModuleInfo(layerLbl,
                            ref.descriptor().name(),
                            ref.descriptor().version(),
                            ref.location()));
        }
    }

    @Override
    public List<ModuleInfo> listModules() {

        List<ModuleInfo> ret = new ArrayList<ModuleInfo>(256);

        for (Entry<String, ComponentLayer> e : this.fetchLayerList()) {
            listModulesOfLayer(ret,e.getKey(),e.getValue());
        }

        Collections.sort(ret);

        return ret;
    }

    @Override
    public List<ModuleInfo> listModules(String layer) {

        List<ModuleInfo> ret = new ArrayList<ModuleInfo>(128);

        ComponentLayer realLayer = this.fetchLayer(layer);

        if (realLayer != null) {
            listModulesOfLayer(ret,layer,realLayer);
        }
        else {
            log.warn("Layer [{}] not found, returning empty module list.",layer);
        }

        Collections.sort(ret);

        return ret;
    }

    protected static void listComponentsOfLayer(List<ComponentInfo> ret, String layerLbl, ComponentLayer layer, ComponentFilter filter) {

        for (ComponentHolder h : layer.getComponents()) {

            Class<? extends Component> cls = h.getComponent().getClass();

            Module m = cls.getModule();

            ModuleDescriptor md = m.getDescriptor();

            if(log.isDebugEnabled()) {
                log.debug("filter,layerLbl,h,matches={},{},{},{}",
                 filter,layerLbl,h,filter == null ? false : filter.matches(layerLbl, h));
            }

            if (filter != null && !filter.matches(layerLbl, h)) {
                continue;
            }

            Optional<ResolvedModule> rm = m.getLayer().configuration().findModule(md.name());

            Optional<URI> location;

            if (rm.isPresent() &&
                Objects.equals(md,rm.get().reference().descriptor())) {
                location = rm.get().reference().location();
            }
            else {
                location = Optional.empty();
            }

            ModuleInfo mi = new ModuleInfo(layerLbl,md.name(),md.version(),location);

            ComponentInfo ci = new ComponentInfo(cls.getName(),h.getPriority(),h.getState(),mi,h.getThrowables());

            ret.add(ci);
        }
    }

    @Override
    public List<ComponentInfo> listComponents() {

        List<ComponentInfo> ret = new ArrayList<ComponentInfo>(256);

        for (Entry<String, ComponentLayer> e : this.fetchLayerList()) {
            listComponentsOfLayer(ret,e.getKey(),e.getValue(),null);
        }

        Collections.sort(ret);

        return ret;
    }

    @Override
    public List<ComponentInfo> listComponents(String filter) {

        if (filter == null) {
            return this.listComponents();
        }

        ComponentFilter cf = ComponentFilter.of(filter);

        List<ComponentInfo> ret = new ArrayList<ComponentInfo>(256);

        for (Entry<String, ComponentLayer> e : this.fetchLayerList(cf)) {
            listComponentsOfLayer(ret,e.getKey(),e.getValue(),cf);
        }

        Collections.sort(ret);

        return ret;
    }

    protected static void startComponentsOfLayer(ServiceContext svcCtxt,String layerLbl,ComponentLayer layer, ComponentFilter filter) {

        for (ComponentHolder h : layer.getComponents()) {

            if (filter != null && !filter.matches(layerLbl, h)) {
                continue;
            }

            ComponentLayer.startComponent(svcCtxt,h);
        }
    }

    @Override
    public void startComponent(String component) {

        ComponentFilter cf = ComponentFilter.of(component);

        for (Entry<String, ComponentLayer> e : this.fetchLayerList(cf)) {
            startComponentsOfLayer(this.getContext(),
                        e.getKey(),e.getValue(),cf);
        }
    }

    protected static void stopComponentsOfLayer(ServiceContext svcCtxt, String layerLbl, ComponentLayer layer, ComponentFilter filter) {

        for (ComponentHolder h : layer.getComponents()) {

            if (filter != null && !filter.matches(layerLbl, h)) {
                continue;
            }

            ComponentLayer.stopComponent(svcCtxt,h,10000L);
        }
    }

    @Override
    public void stopComponent(String component) {

        ComponentFilter cf = ComponentFilter.of(component);

        for (Entry<String, ComponentLayer> e : this.fetchLayerList(cf)) {
            stopComponentsOfLayer(
                        this.getContext(),
                        e.getKey(),e.getValue(),cf);
        }
    }

    @Override
    public void reloadLayer(String label, long timeout) {

        if (LayerConfig.BOOT_LABEL.equals(label)) {
            throw new IllegalArgumentException("The boot layer may not be reloaded.");
        }

        ServiceContext svcCtxt = this.getContext();

        ComponentLayer oldLayer = this.removeLayer(label);

        if (oldLayer == null) {
            throw new IllegalArgumentException("Layer ["+label+"] does not exist.");
        }

        oldLayer.stopAll(svcCtxt,timeout);

        try {
            svcCtxt.synchronize(timeout);
        } catch (InterruptedException e) {
            throw new IllegalStateException("Reload of layer ["+label+"] has been interrupted",e);
        }

        LayerConfig layerConfig = oldLayer.getLayerConfig();

        ComponentLayer newLayer = this.createLayer(layerConfig);
        newLayer.startAll(svcCtxt);
    }

    protected ComponentHolder resolveComponent(Class<?> cls) {

        Module module = cls.getModule();

        ComponentLayer cl = this.layersByModuleLayer.get(module.getLayer());

        if (cl == null) {
            return null;
        }

        return cl.getByModule(module);
    }

    protected static final String makeTrace(StackWalker sw) {

        return sw.walk(s -> s.map(f -> f.toStackTraceElement().toString()).collect(Collectors.joining("\n  ")));
    }

    protected Optional<ComponentHolder> resolveComponent() {

        StackWalker sw = StackWalker.getInstance(Option.RETAIN_CLASS_REFERENCE);

        Optional<ComponentHolder> ret = sw.walk(s ->
                       s.map(f ->
                             this.resolveComponent(f.getDeclaringClass()))
                               .filter(Objects::nonNull)
                                 .findFirst());

        if (ret.isEmpty()) {

            log.warn("Cannot map stack trace to component:\n  {}",makeTrace(sw));
        }
        else if (log.isDebugEnabled()) {
            log.debug("Mapped stack to [{}]:\n  {}",ret.get(),makeTrace(sw));
        }

        return ret;
    }

    @Override
    public void commit() {

        this.resolveComponent().ifPresent(
            ch -> {
                // allow double commits in order to allow config updates.
                ch.checkState(ComponentState.COMMITTED,
                        ComponentState.STARTING,
                        ComponentState.STARTED,
                        ComponentState.COMMITTED);

                log.info("Committed component [{}]",ch);
            }
        );
    }

    protected static final String makeTrace(Throwable throwable) {

        return Arrays.stream(throwable.getStackTrace()).map(Object::toString).collect(Collectors.joining("\n  "));
    }


    protected Optional<ComponentHolder> resolveComponent(Throwable throwable) {

        Optional<ComponentHolder> ret =
            Arrays.stream(throwable.getStackTrace()).map(f->
            this.componentsByModuleName.get(f.getModuleName()))
                               .filter(Objects::nonNull)
                                 .findFirst();
        if (ret.isEmpty()) {

            log.warn("Cannot map throwable stack trace to component:\n  {}",makeTrace(throwable));
        }
        else if (log.isDebugEnabled()) {
            log.debug("Mapped throwable stack to [{}]:\n  {}",ret.get(),makeTrace(throwable));
        }

        return ret;
    }

    @Override
    public void recordConfigException(ThrowableInfo throwable) {

        this.resolveComponent(throwable.getThrowable()).ifPresent(
            c -> {
                c.checkState(ComponentState.CONFIG_LISTENER_FAILED,
                              ComponentState.CONFIG_LISTENER_FAILED,
                              ComponentState.SERVICE_LISTENER_FAILED,
                              ComponentState.STARTED,
                              ComponentState.COMMITTED);
                c.addThrowable(throwable);
            }
        );
    }

    @Override
    public void recordServiceException(ThrowableInfo throwable) {

        this.resolveComponent(throwable.getThrowable()).ifPresent(
            c -> {
                c.checkState(ComponentState.SERVICE_LISTENER_FAILED,
                              ComponentState.CONFIG_LISTENER_FAILED,
                              ComponentState.SERVICE_LISTENER_FAILED,
                              ComponentState.STARTED,
                              ComponentState.COMMITTED);
                c.addThrowable(throwable);
            }
        );
    }

}
