/***********************************************************
 *
 * Service Runner framework runner using commons-daemon
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.runner;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>This class is used to start an svc-runner instance through
 * apache's <code>procrun.exe</code> service starter.
 * </p>
 * <p>The command line arguments for proc run shall be:</p>
 * <pre>
 * prunsrv //IS//svc-runner \
 *    --DisplayName="Clazzes.org svc-runner" --Install=prunsrv.exe --Jvm=auto \
 *    --StartMode=jvm --StopMode=jvm \
 *    --StartClass=org.clazzes.svc.runner.Bootstrap --StartParams=start \
 *    --StopClass=org.clazzes.svc.runner.Bootstrap --StopParams=stop
 * </pre>
 */
public class Bootstrap {

    private static final Logger log = LoggerFactory.getLogger(Bootstrap.class);

    static ServiceContextImpl svcCtxt;

    /**
     * @param args Command line arguments, must contain one single string
     *              "start" or "stop".
     * @throws Exception Upon errors setting up the framework.
     */
    public static void main(String[] args) throws Exception {

        // global setup.
        Setup.setup();

        String mode = null;

        if (args != null && args.length == 1) {

            mode = args[0];
        }

        if ("start".equals(mode)) {

            try {
                int keepaliveTimeout = Config.getIntProperty(Setup.KEEPALIVE_TIMEOUT_PROPERTY,10000);

                Config.registerSecurityProviders();

                svcCtxt = new ServiceContextImpl();

                svcCtxt.start();

                CoreServiceImpl cs = CoreServiceImpl.INSTANCE;

                // according to
                //   http://commons.apache.org/proper/commons-daemon/procrun.html
                // the start method should not exits until the stop method is called.
                while (!cs.waitForDestroy(keepaliveTimeout)) {
                    log.info("KEEPALIVE: Windows Service.");
                }
            }
            catch(Throwable e) {
                log.error("Unexpected error running svc-runner Windows Service",e);
            }

            log.info("Leaving svc-runner Windows Service.");
        }
        else if ("stop".equals(mode)) {

            try {
                int stopTimeout = Config.getIntProperty(Setup.STOP_TIMEOUT_PROPERTY,10000);

                svcCtxt.stop(stopTimeout);

                svcCtxt.synchronize(stopTimeout);
            }
            catch(Throwable e) {
                log.error("Unexpected error stopping svc-runner Windows Service",e);
            }
            finally {
                CoreServiceImpl.destroyInstance();
                Setup.stopLogback();
            }
        }
        else {
            throw new IllegalArgumentException("Unrecognized command line parameters sepcified (must be \"start\" or \"stop\").");
        }
    }
}
