/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * <p>An opens directive <code>srcmod/srcpkg=targetmod</code> like
 * </p>
 * <pre>
 * java.base/java.nio=org.apache.pdfbox.io
 * </pre>
 * <p>
 * which opens the package <code>srcpkg</code> of module <code>srcmod</code>
 * to <code>targetmod</code>.
 * </p>
 */
public class OpensDirective implements Serializable {

    private static final long serialVersionUID = -2324073488225704752L;

    private static final Pattern PACKAGE_REGEX =
        Pattern.compile("(?x)[a-z][a-z0-9_]*(\\.[a-z][a-z0-9_]*)*");

    private static final Pattern COMMA_SEPARATOR_REGEX =
        Pattern.compile("\\s*,\\s*");

    private final String sourceModule;
    private final String sourcePackage;
    private final String targetModule;

    protected OpensDirective(String sourceModule,String sourcePackage,String targetModule) {
        this.sourceModule = sourceModule;
        this.sourcePackage = sourcePackage;
        this.targetModule = targetModule;
    }

    public static OpensDirective of(String sourceModule,String sourcePackage,String targetModule) {

        if (!PACKAGE_REGEX.matcher(sourceModule).matches()) {
            throw new IllegalArgumentException("Source module ["+sourceModule+"] is not a valid module name.");
        }

        if (!PACKAGE_REGEX.matcher(sourcePackage).matches()) {
            throw new IllegalArgumentException("Source package ["+sourcePackage+"] is not a valid package name.");
        }

        if (!PACKAGE_REGEX.matcher(targetModule).matches()) {
            throw new IllegalArgumentException("Target module ["+targetModule+"] is not a valid module name.");
        }

        return new OpensDirective(sourceModule,sourcePackage,targetModule);
    }

    /**
     * Parse a directive <code>srcmod/srcpkg=targetmod</code>.
     * @param spec An open directive to parse.
     * @return The parsed open directive.
     */
    public static OpensDirective ofSpec(String spec) {

        int slash = spec.indexOf('/');
        int eqsign = spec.indexOf('=',slash+1);

        if (slash < 0 || eqsign < 0) {
            throw new IllegalArgumentException("Opens spec ["+spec+"] does not cntain a slash and an equals sign.");
        }

        return of(spec.substring(0,slash),spec.substring(slash+1,eqsign),spec.substring(eqsign+1));
    }

    /**
     * Parse a comma-separated list of opens directives like
     * <code>srcmod1/srcpkg1=targetmod1,srcmod2/srcpkg2=targetmod2,...</code>.
     * @param specList A comma-separated list of opens directives to parse.
     * @return The parsed list of opens directives.
     */
    public static List<OpensDirective> ofSpecList(String specList) {

        if (specList == null) {
            return List.of();
        }

        String stripped = specList.strip();

        if (stripped.isEmpty()) {
            return List.of();
        }

        String[] specs = COMMA_SEPARATOR_REGEX.split(stripped);

        List<OpensDirective> ret = new ArrayList<OpensDirective>(specs.length);

        for (String spec:specs) {

            // protect against double commas.
            if (!spec.isEmpty()) {
                ret.add(ofSpec(spec));
            }
        }

        return ret;
    }

    public String getSourceModule() {
        return this.sourceModule;
    }

    public String getSourcePackage() {
        return this.sourcePackage;
    }

    public String getTargetModule() {
        return this.targetModule;
    }

    @Override
    public String toString() {
        return this.sourceModule + "/" + this.sourcePackage + "=" + this.targetModule;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((this.sourceModule == null) ? 0 : this.sourceModule.hashCode());
        result = prime * result + ((this.sourcePackage == null) ? 0 : this.sourcePackage.hashCode());
        result = prime * result + ((this.targetModule == null) ? 0 : this.targetModule.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        OpensDirective other = (OpensDirective) obj;
        if (this.sourceModule == null) {
            if (other.sourceModule != null)
                return false;
        } else if (!this.sourceModule.equals(other.sourceModule))
            return false;
        if (this.sourcePackage == null) {
            if (other.sourcePackage != null)
                return false;
        } else if (!this.sourcePackage.equals(other.sourcePackage))
            return false;
        if (this.targetModule == null) {
            if (other.targetModule != null)
                return false;
        } else if (!this.targetModule.equals(other.targetModule))
            return false;
        return true;
    }




}
