/**
 * $Id$
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.clazzes.svc.runner;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Properties;

/**
 * Command line tool used to manage secrets.
 */
public class ManageSecrets {

    static public void usage() {

        System.err.println("usage:");
        System.err.println("  ManageSecrets -list <properties>");
        System.err.println("  ManageSecrets -add <properties> <pid> <key> <value>");
        System.err.println("  ManageSecrets -decrypt <properties> <pid> <key>");
        System.err.println("The 32 byte base64-encoded masterkey may be provided by the");
        System.err.println("SVCRUNNER_SECRETS_MK environment variable or by the");
        System.err.println("general option -mk <key>");
        System.exit(1);
    }

    protected static Properties readProps(String filename) throws Exception {

        Properties props = new Properties();

        try (FileInputStream fis = new FileInputStream(filename)) {

            System.err.println("Loading secrets from ["+filename+"].");
            props.load(fis);
        }
        catch(FileNotFoundException e) {
            System.err.println("File ["+filename+"] does not exist starting with empty properties.");
        }

        return props;
    }

    protected static void writeProps(Properties props, String filename) throws Exception {

        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

        String comment = "Stored by " + ManageSecrets.class.getName()+ " as " +df.format(new Date());

        try (FileOutputStream fos = new FileOutputStream(filename)) {

            System.err.println("Saving secrets to ["+filename+"].");
            props.store(fos,comment);
        }
    }

    /**
     * Main program for managing secrets files.
     *
     * @param args The program arguments
     */
    public static void main(String[] args) throws Exception {

        String secretsMasterKey = System.getenv("SVCRUNNER_SECRETS_MK");

        int iarg = 0;

        if (args.length < 1) {
            usage();
        }

        String command = args[iarg++];

        while (iarg+1 < args.length) {
            if ("-mk".equals(args[iarg])) {
                secretsMasterKey = args[iarg+1];
                iarg+=2;
            }
            else {
                break;
            }
        }

        if ("-list".equals(command)) {
            if (iarg+1 != args.length) {
                usage();
            }

            String filename = args[iarg];

            Properties props = readProps(filename);

            Enumeration<Object> kit = props.keys();

            while (kit.hasMoreElements()) {

                String key = String.valueOf(kit.nextElement());

                System.out.println(key+"="+props.getProperty(key));
            }
        }
        else if ("-add".equals(command)) {

            if (iarg+4 != args.length) {
                usage();
            }

            String filename = args[iarg];
            String pid = args[iarg+1];
            String key = args[iarg+2];
            String value = args[iarg+3];

            Properties props = readProps(filename);

            SecretsStore store = new SecretsStore(props,secretsMasterKey);

            System.err.println("Adding secret ["+key+"] for PID ["+pid+"].");
            store.encrypt(pid,key,value);

            writeProps(props,filename);
        }
        else if ("-decrypt".equals(command)) {

            if (iarg+3 != args.length) {
                usage();
            }

            String filename = args[iarg];
            String pid = args[iarg+1];
            String key = args[iarg+2];

            Properties props = readProps(filename);

            SecretsStore store = new SecretsStore(props,secretsMasterKey);

            System.err.println("Decrypting secret ["+key+"] for PID ["+pid+"].");

            String value = store.decrypt(pid,key);

            System.out.println(key+"="+value);
        }
        else {
            System.err.println("Invalid command ["+command+"].");
            usage();
        }

    }

}
