/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Main {

    private static final Logger log = LoggerFactory.getLogger(Main.class);

    /**
     * Start and run the svc-runner until it is stopped by
     * <code>Ctrl-C</code>.
     *
     * @param args Command line arguments.
     * @throws Exception Upon erros starting up the framework.
     */
    public static void main(String[] args) throws Exception {
        /*
        long debugTimeout = 10000;
        boolean debug = false;
        boolean synchronous = false;

        if (args != null) {
            for (int iarg=0;iarg<args.length;++iarg) {

                if ("--debug".equals(args[iarg])) {
                    debug = true;
                }
                else if ("--synchronous".equals(args[iarg])) {
                    synchronous = true;
                }
                else if ("--timeout".equals(args[iarg]) && iarg+1 < args.length) {

                    try {
                        ++iarg;
                        debugTimeout = Long.valueOf(args[iarg]) * 1000L;
                    }
                    catch (NumberFormatException e) {
                        log.warn("Argument ["+args[iarg]+"] for --timeout is non-numeric.");
                    }
                }
                else {
                    log.warn("Encountered unknown cmdline option ["+args[iarg]+"]");
                }
            }
        }
        */

        // global setup.
        Thread mainThread = Setup.setup();

        AtomicBoolean keepWorking = new AtomicBoolean(true);

        int stopTimeout = Config.getIntProperty(Setup.STOP_TIMEOUT_PROPERTY,10000);
        int keepaliveTimeout = Config.getIntProperty(Setup.KEEPALIVE_TIMEOUT_PROPERTY,10000);

        ServiceContextImpl svcCtxt = new ServiceContextImpl();

        try {

            Config.registerSecurityProviders();

            svcCtxt.start();

            Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                log.info("Shutdown hook caught exit signal...");
                synchronized (mainThread) {

                    keepWorking.set(false);
                    mainThread.notify();

                    try {
                        mainThread.wait(); // Give the main thread time to finish...
                        log.info("Shutdown hook detected finish of main thread.");
                        Runtime.getRuntime().halt(0);
                    } catch (InterruptedException e) {
                        log.error("Shutdown hook interrupted while waiting for main thread.",e);
                    }
                }
            }));

            synchronized(mainThread) {
                while (keepWorking.get()) {
                    if (log.isDebugEnabled()) {
                        log.debug("KEEPALIVE: main thread.");
                    }
                    mainThread.wait(keepaliveTimeout);
                }

            }

            log.info("Main thread will stop components upon exit signal.");
            svcCtxt.stop(stopTimeout);
            svcCtxt.synchronize(stopTimeout);

            log.info("About to leave main thread.");
        }
        catch(Throwable e) {
            log.error("Caught unexpected exception in svc-runner main",e);
        }
        finally {
            CoreServiceImpl.destroyInstance();
        }

        log.info("Informing shutdown hook about end of main thread.");

        synchronized(mainThread) {
            mainThread.notify();
        }

        log.info("Leaving main thread.");
        System.exit(0);
    }
}
