/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/


package org.clazzes.svc.runner;

import java.nio.file.Path;
import java.util.List;
import java.util.Map;

/**
 * A POJO for a configured layer.
 */
public class LayerConfig {

    public static final String BOOT_LABEL = "$boot";

    private final String label;
    private final Path modulePath;
    private final String parent;
    private final List<OpensDirective> opens;
    private String parentKey;
    private String key;

    protected LayerConfig(String label, Path modulePath, String parent, List<OpensDirective> opens) {
        this.label = label;
        this.modulePath = modulePath;
        this.parent = parent;
        this.opens = opens;
    }

    protected LayerConfig(List<OpensDirective> opens) {
        this.label = BOOT_LABEL;
        this.modulePath = null;
        this.parent = null;
        this.key = BOOT_LABEL;
        this.opens = opens;
    }

    public static LayerConfig ofUnresolved(String label, Path modulePath, String parent, List<OpensDirective> opens) {

        if (BOOT_LABEL.equals(label)) {
            throw new IllegalArgumentException("Cannot configure $boot layer.");
        }

        if (BOOT_LABEL.equals(parent)) {
            throw new IllegalArgumentException("Cannot use $boot layer as parent.");
        }

        return new LayerConfig(label,modulePath,parent,opens);
    }

    public static LayerConfig ofBoot() {
        return new LayerConfig(List.of());
    }

    public boolean isBoot() {
        return this.key == BOOT_LABEL;
    }

    public String getLabel() {
        return this.label;
    }

    public Path getModulePath() {
        return this.modulePath;
    }

    public String getParent() {
        return this.parent;
    }

    public List<OpensDirective> getOpens() {
        return this.opens;
    }

    protected void checkResolved() {

        if (this.key == null) {
            throw new IllegalArgumentException("Layer ["+this.label+"] is not yet resolved.");
        }
    }

    /**
     * The sortable key of the parent, which is created by {@link #resolve(Map)}.
     * @return The key <code>[parent.]label</code>.
     */
    public String getParentKey() {

        this.checkResolved();
        return this.parentKey;
    }

    /**
     * Get the key
     * <pre>
     * [parent.]parent.label
     * </pre>
     *
     * @return The full key of this layer, which is created by {@link #resolve(Map)}.
     */
    String getKey() {

        this.checkResolved();
        return this.key;
    }

    /**
     * Resolve the parent structure to a key
     * <pre>
     * [parent.]parent.label
     * </pre>
     * The key will be used to sort the layers by dependency depth.
     *
     * @param unresolved The map of unresolved layers by layer label.
     * @return the full path of this component.
     */
    public String resolve(Map<String,LayerConfig> unresolved) {

        if (this.key == null) {

            if (this.parent == null) {

                this.parentKey = null;
                this.key = this.label;
            }
            else {

                if (this.label.equals(this.parent)) {
                    throw new IllegalArgumentException("Layer ["+this.label+"] references itself as a parent.");
                }

                LayerConfig p = unresolved.get(this.parent);

                if (p==null) {
                    throw new IllegalArgumentException("Cannot resolve parent layer ["+this.parent+"] of layer ["+this.label+"].");
                }

                this.parentKey = p.resolve(unresolved);
                this.key =  this.parentKey + "." + this.label;
            }
        }

        return this.key;
    }

}
