/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.nio.file.Path;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.security.auth.DestroyFailedException;
import javax.security.auth.Destroyable;

import org.clazzes.svc.api.CoreService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>This class provides thread pools and secrets to applications.
 * </p>
 */
public class CoreServiceImpl implements CoreService, Destroyable {

    private static final Logger log = LoggerFactory.getLogger(CoreServiceImpl.class);

    public static final String EXECUTOR_POOL_SIZE_PROPERTY  = "svc.runner.executorPoolSize";
    public static final String SCHEDULED_EXECUTOR_POOL_SIZE_PROPERTY  = "svc.runner.scheduledExecutorPoolSize";
    public static final String STOP_EXECUTOR_TRIES_PROPERTY  = "svc.runner.stopExecutorTries";
    public static final String SECRETS_MASTER_KEY_PROPERTY = "svc.runner.secretsMasterKey";

    protected static final CoreServiceImpl INSTANCE = new CoreServiceImpl();

    public static final CoreService provider() {
        return INSTANCE;
    }

    private ExecutorService executorService;
    private ScheduledExecutorService scheduledExecutorService;
    private SecretsStore secretsStore;
    private boolean destroyed;
    private final int stopExecutorTries;

    protected void init() {

        int executorPoolSize = Config.getIntProperty(EXECUTOR_POOL_SIZE_PROPERTY,4);
        int scheduledExecutorPoolSize = Config.getIntProperty(SCHEDULED_EXECUTOR_POOL_SIZE_PROPERTY,4);

        this.executorService = Executors.newFixedThreadPool(executorPoolSize);
        this.scheduledExecutorService = Executors.newScheduledThreadPool(scheduledExecutorPoolSize);

        String secretsMasterKey = Config.getProperty(SECRETS_MASTER_KEY_PROPERTY,System.getenv("SVCRUNNER_SECRETS_MK"));

        if (secretsMasterKey == null) {
            this.secretsStore = null;
        }
        else {
            Properties props = new Properties();
            Config.loadSecretsProperties(props);
            this.secretsStore = new SecretsStore(props,secretsMasterKey);
            log.info("Successfully initialized secrets store with KVC [{}].",
                     this.secretsStore.getKvc());
        }
    }

    protected CoreServiceImpl() {

        this.stopExecutorTries = Config.getIntProperty(STOP_EXECUTOR_TRIES_PROPERTY,30);
        this.init();
    }


    @Override
    public Path getEtcDir() {

        return Config.getEtcDir();
    }


    @Override
    public ExecutorService getExecutorService() {
        return this.executorService;
    }

    @Override
    public ScheduledExecutorService getScheduledExecutorService() {
        return this.scheduledExecutorService;
    }

    @Override
    public String getSecret(String pid, String scheme, String key) {

        log.info("PID ["+pid+"] is requesting secret for key ["+scheme+":"+key+"]");

        try {
            if ("prop".equals(scheme)) {

                if (this.secretsStore == null) {
                    throw new UnsupportedOperationException("SVC secrets store is not configured, define SVCRUNNER_SECRETS_MK environment variable or svc.runner.secretsMasterKey config variable.");
                }

                return this.secretsStore.decrypt(pid,key);
            }
            else if ("conf".equals(scheme)) {

                return Config.getProperty(key);
            }
            else if ("env".equals(scheme)) {

                return System.getenv(key);
            }
            else if ("void".equals(scheme)) {

                // avoid confusing and eventually bundle-breaking exception if a value has
                // no explicit value configured and the defaultValue is "secret::void" to trigger secret:: support,
                // see https://gitlab.intra.iteg.at/osgi/core/osgi-runner-pkg/-/issues/6#note_8291
                return "";
            }
            else {
                throw new IllegalArgumentException("Unsupported key scheme ["+scheme+"], must be [prop], [env] or [void].");
            }
        } catch (Exception e) {
            throw new RuntimeException("Fetching secret key ["+key+"] for PID ["+pid+"]",e);
        }
    }

    private void shutdownService(ExecutorService service, String label) {

        log.info("Shutting down "+label+"...");

        List<Runnable> runnables = service.shutdownNow();

        if (runnables != null) {

            for (Runnable runnable : runnables) {
                log.warn("Runnable ["+runnable+"] has not been run by "+label+" before shutdown.");
            }
        }

        try {

            int ntries = 0;

            while (ntries < this.stopExecutorTries && !service.awaitTermination(1,TimeUnit.SECONDS)) {
                ++ntries;
                log.info("Waiting for "+label+" to terminate (try "+ntries+")...");
            }

            if (ntries < this.stopExecutorTries) {
                log.info(label+" has terminated after ["+ntries+"] tries.");
            }
            else {
                log.warn(label+" did not terminate after ["+ntries+"] tries.");
            }

        } catch (InterruptedException e) {
            log.warn("Waiting for "+label+" to terminate has been interrupted.");
        }

    }

    @Override
    public void destroy() throws DestroyFailedException {

        if (this.scheduledExecutorService != null) {
            this.shutdownService(this.scheduledExecutorService,"ScheduledExecutorService");
            this.scheduledExecutorService = null;
        }

        if (this.executorService != null) {
            shutdownService(this.executorService,"ExecutorService");
            this.executorService = null;
        }

        this.secretsStore = null;

        synchronized(this) {
            this.destroyed = true;
            this.notifyAll();
        }
    }

    @Override
    public synchronized boolean isDestroyed() {
        return this.destroyed;
    }

    public synchronized boolean waitForDestroy(long timeout) throws InterruptedException {
        if (!this.destroyed) {
            this.wait(timeout);
        }

        return this.destroyed;
    }

    public static void destroyInstance() {
        log.info("Destroying CoreService.");

        try {
            INSTANCE.destroy();
        }
        catch(Throwable e) {
            log.warn("Error destroying CoreService",e);
        }
    }

}
