/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.MappingIterator;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.fasterxml.jackson.dataformat.javaprop.JavaPropsMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import com.fasterxml.jackson.dataformat.yaml.YAMLMapper;

/**
 * Low-level handling of YAML and properties configuration files.
 */
public class ConfigurationFiles {

    private static final Logger log = LoggerFactory.getLogger(ConfigurationFiles.class);

    public static final Pattern SECRET_PATTERN = Pattern.compile("^secret::(env|prop|void):(.*)$");
    public static final Pattern PID_PATTERN = Pattern.compile("^[a-z][a-z0-9]*(\\.[a-z][a-z0-9]*)+$");

    public static final String SERVICE_PID_KEY = "service.pid";
    public static final String SERVICE_TYPE_KEY = "service.type";
    public static final String SERVICE_TYPE_SECRET = "secret";

    public static final String SERVICE_TYPE_VARS = "vars";

    protected static final class SecretSupplierJsonSerializer extends JsonSerializer<SecretSupplier> {

        @Override
        public void serialize(SecretSupplier value, JsonGenerator gen, SerializerProvider serializers)
                throws IOException {

            gen.writeStartObject();

            gen.writeFieldName(SERVICE_TYPE_KEY);
            gen.writeString(SERVICE_TYPE_SECRET);

            gen.writeFieldName("scheme");
            gen.writeString(value.getScheme());

            gen.writeFieldName("key");
            gen.writeString(value.getKey());

            gen.writeEndObject();
        }
    }

    protected static final class VarsSupplierJsonSerializer extends JsonSerializer<VarsSupplier> {

        @Override
        public void serialize(VarsSupplier value, JsonGenerator gen, SerializerProvider serializers)
                throws IOException {

            gen.writeStartObject();

            gen.writeFieldName(SERVICE_TYPE_KEY);
            gen.writeString(SERVICE_TYPE_VARS);

            gen.writeFieldName("expr");
            gen.writeString(value.getExpr());

            gen.writeEndObject();
        }
    }

    /**
     * Return the service PID of the given configuration set.
     * @param config The configuration read by one of the read methods of this rclkass.
     * @return The value of the key {@value #SERVICE_PID_KEY}.
     */
    public static String getServicePid(Map<String,?> config) {

        Object pid_o = config.get(SERVICE_PID_KEY);

        if (pid_o instanceof String pid) {
            return pid;
        }
        else {
            throw new IllegalArgumentException("Configuration map does not contain a "+SERVICE_PID_KEY+" property.");
        }
    }

    /**
     * Read a new-style YAML configuration file.
     * @param is An input stream to read from.
     * @param path A path used for logging
     * @return A map with config PIDs as keys and the parsed configurations as values.
     * @throws IOException Upon I/O errors.
     */
    public static Map<String,Map<String,?>> readYaml(InputStream is, String path) throws IOException {

        YAMLMapper yamlMapper = new YAMLMapper();

        JsonParser parser = yamlMapper.createParser(is);
        JavaType jt = TypeFactory.defaultInstance().constructParametricType(Map.class,String.class,Object.class);

        MappingIterator<Map<String,?>> it = yamlMapper.readValues(parser,jt);

        Map<String,Map<String,?>> ret = new HashMap<String,Map<String,?>>();

        int section = 0;

        while(it.hasNext()) {

            @SuppressWarnings("unchecked")
            Map<String,Object> values = (Map<String, Object>)it.next();
            ++section;

            Object pid_o = values.get(SERVICE_PID_KEY);

            if (pid_o instanceof String pid) {

                if (PID_PATTERN.matcher(pid).matches()) {

                    ConfigurationFiles.replaceNewSecrets(pid,null,values);

                    log.info("Parsed section [{}] with PID [{}] from file [{}]",
                                    section,pid,path);

                    Map<String,?> dup = ret.put(pid,values);

                    if (dup != null) {
                        log.warn("Replaced duplicate PID [{}] in file [{}]",pid,path);
                    }
                }
                else {
                    log.warn("Section [{}] in file [{}] contains invalid " + SERVICE_PID_KEY + " [{}], ignoring it.",
                    section,path,pid);
                }
            }
            else {
                log.warn("Section [{}] in file [{}] contains no " + SERVICE_PID_KEY + " property, ignoring it.",
                            section,path);
            }
        }

        return ret;
    }

    @SuppressWarnings("unchecked")
    protected static void replaceNewSecrets(String pid, String ctxt, Map<String,Object> cfg) {

        for (String k : cfg.keySet()) {

            Object v = cfg.get(k);

            if (v instanceof Map sub) {

                Object serviceType = sub.get(SERVICE_TYPE_KEY);

                if (serviceType == null) {


                    replaceNewSecrets(pid,ctxt == null?k:ctxt+"."+k,sub);
                }
                else if (SERVICE_TYPE_SECRET.equals(serviceType)) {

                    String scheme = null;
                    String key = null;

                    if (sub.size() == 3) {

                        Object scheme_o = sub.get("scheme");
                        Object key_o = sub.get("key");

                        if (scheme_o instanceof String s) {
                            scheme = s;
                        }

                        if (key_o instanceof String s) {
                            key = s;
                        }
                    }

                    if (scheme == null || key == null) {
                        throw new IllegalArgumentException("Map of [service.type=secret] does not have exactly 3 keys including [scheme] and [key].");
                    }

                    cfg.put(k,new SecretSupplier(pid,scheme,key));
                }
                else if (SERVICE_TYPE_VARS.equals(serviceType)) {

                    String expr = null;

                    if (sub.size() == 2) {

                        Object expr_o = sub.get("expr");

                        if (expr_o instanceof String s) {
                            expr = s;
                        }
                    }

                    if (expr == null) {
                        throw new IllegalArgumentException("Map of [service.type=vars] does not have exactly 2 keys including [expr].");
                    }

                    cfg.put(k,new VarsSupplier(pid,ctxt,cfg,"ref:"+k,expr));
                }
                else {
                    throw new IllegalArgumentException("Unknown [service.type="+serviceType+"].");
                }
            }
        }
    }

    /**
     * Read a YAML file from a given path.
     * @param p The path of the file to read.
     * @return A map with config PIDs as keys and the parsed configurations as values.
     *         Upon errors, <code>null</code> is returned.
     */
    public static Map<String,Map<String,?>> readYamlFile(Path p) {

        try (InputStream is = Files.newInputStream(p)) {

            return readYaml(is,p.toString());
        }
        catch (Exception e) {
            log.error("Error reading yaml file ["+p+"]",e);
            return null;
        }
    }

    /**
     * Write a set of configurations to a YAML file..
     * @param os The output stream to write to.
     * @param path The resource/file path used for logging.
     * @param configs A map with config PIDs as keys and the parsed configurations as values.
     * @throws IOException Upon I/O errors.
     */
    public static void writeYaml(OutputStream os, String path, Map<String,Map<String,?>> configs) throws IOException {

        SimpleModule sm = new SimpleModule();
        sm.addSerializer(SecretSupplier.class,new SecretSupplierJsonSerializer());
        sm.addSerializer(VarsSupplier.class,new VarsSupplierJsonSerializer());

        YAMLMapper mapper = new YAMLMapper();
        mapper.configure(YAMLGenerator.Feature.MINIMIZE_QUOTES,true);
        mapper.registerModule(sm);

        for (Entry<String, Map<String, ?>> e : configs.entrySet()) {
            mapper.writeValue(os,e.getValue());
        }
    }

    /**
     * Write a set of configurations to a YAML file..
     * @param p The file path to write to.
     * @param configs A map with config PIDs as keys and the parsed configurations as values.
     * @throws IOException Upon I/O errors.
     */
    public void writeYamlFile(Path p, Map<String,Map<String,?>> configs) throws IOException {
        try (OutputStream os = Files.newOutputStream(p)) {
            writeYaml(os,p.toString(),configs);
        }
    }

    /**
     * Read an old-style java properties configuration file.
     * @param is The input stream to read.
     * @param pid The PID to read.
     * @param path The path used for logging.
     * @return The parsed configuration with the service PID stored under
     *         the key {@value #SERVICE_PID_KEY}.
     * @throws IOException Upon I/O errors.
     */
    public static Map<String,?> readProperties(InputStream is, String pid, String path) throws IOException {

        log.info("Reading config PID [{}] from file [{}].",pid,path);

        JavaPropsMapper mapper = new JavaPropsMapper();

        JavaType jt = TypeFactory.defaultInstance().constructParametricType(Map.class,String.class,Object.class);

        Map<String,Object> values = mapper.readValue(is,jt);

        ConfigurationFiles.replaceLegacySecrets(pid,values);

        // move service.pid value to key "service.pid" w/o structure.
        Object service = values.get("service");

        String svcPid = null;

        if (service instanceof Map map) {
            Object svcPid_o = map.remove("pid");
            if (svcPid_o instanceof String s) {
                svcPid = s;
            }
            if (map.size() == 0) {
                values.remove("service");
            }
        }

        if (!pid.equals(svcPid)) {
            log.warn("File [{}] has wrong "+SERVICE_PID_KEY+" value, correcting it.",path);
        }
        else {
            log.info("File [{}] has correct "+SERVICE_PID_KEY+" value [{}].",path,svcPid);
        }

        values.put(SERVICE_PID_KEY,pid);

        return values;
    }

    @SuppressWarnings("unchecked")
    protected static void replaceLegacySecrets(String pid, Map<String,Object> cfg) {

        for (String k : cfg.keySet()) {

            Object v = cfg.get(k);

            if (v instanceof Map sub) {
                replaceLegacySecrets(pid,sub);
            }
            else if (v instanceof String s) {

                Matcher m = SECRET_PATTERN.matcher(s);

                if (m.matches()) {
                    String scheme = m.group(1);
                    String key = m.group(2);
                    cfg.put(k,new SecretSupplier(pid,scheme,key));
                }
            }
        }
    }

    /**
     * Read a CFG file <code>&lt;pid&gt;.cfg</code>from a given path.
     * @param p The path of the file to read.
     * @return A parsed configuration with the service PID stored under
     *         the key {@value #SERVICE_PID_KEY}.
     *         If the filename is not matching or upon I/O errors,
     *         <code>null</code> is returned.
     */
    public static Map<String,?> readCfgFile(Path p) {

        String fn = p.getFileName().toString();

        if (!fn.endsWith(".cfg")) {
            log.warn("Configuration file [{}] does not match [*.cfg].",p);
            return null;
        }

        String pid = fn.substring(0,fn.length()-4);

        if (ConfigurationFiles.PID_PATTERN.matcher(pid).matches()) {

            try (InputStream is = Files.newInputStream(p)) {

                return readProperties(is,pid,p.toString());
            }
            catch (Exception e) {
                log.error("Error reading cfg file ["+p+"]",e);
                return null;
            }
        }
        else {
            log.warn("Ignoring cfg file [{}] with invalid PID.",p);
            return null;
        }
    }

}
