/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/

package org.clazzes.svc.runner;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.Provider;
import java.security.Security;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.SortedMap;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>This class loads the configuration from
 * <code>/etc/svc-runner/system.properties</code>
 */
public class Config {

    public static final String ETC_PATH_PROPERTY = "svc.runner.etcPath";
    public static final String THROWABLE_HISTORY_SIZE_PROPERTY = "svc.runner.throwableHistorySize";

    public static final String SECURITY_PROVIDERS_PROPERTY = "svc.runner.securityProviders";

    // properties stating with this prefix define additional modulepaths
    public static final String MODULE_PATH_PROPERTY_PREFIX = "svc.runner.modulePath.";
    public static final String MODULE_PARENT_PROPERTY_PREFIX = "svc.runner.moduleParent.";
    public static final String MODULE_OPENS_PROPERTY_PREFIX = "svc.runner.moduleOpens.";

    private static final Pattern MODULE_LAYER_LABEL_RX = Pattern.compile("^[a-zA-Z_][a-zA-Z_-]*");

    private static final Path etcDir;
    private static final SortedMap<String,String> config;
    private static final Map<String,LayerConfig> sortedLayerConfigs;
    private static final Map<String,LayerConfig> layerConfigs;


    private static final String[] LOG_PROPERTIES = new String [] {
            "os.name",
            "os.version",
            "os.arch",
            "java.home",
            "java.class.path",
            "java.library.path",
            "java.endorsed.dirs",
            "java.ext.dirs",
            "java.io.tmpdir",
            "jdk.module.path",
            "file.encoding",
            "file.separator",
            "java.specification.name",
            "java.specification.version",
            "java.runtime.name",
            "java.runtime.version",
            "java.vm.name",
            "java.vm.version"
    };

    private static final Logger log = LoggerFactory.getLogger(Config.class);
    private static final Log aclLog = LogFactory.getLog(Config.class);
    private static final java.util.logging.Logger julLogger = java.util.logging.Logger.getLogger(Config.class.getName());

    static {

        log.info("Setting up SVC-Runner, system setup is:");

        for (String prop : LOG_PROPERTIES) {
            String v = System.getProperty(prop);
            if (v!=null) {
                log.info(" {} = {}",prop,v);
            }
        }

        log.info(" TimeZone.getDefault() = {}",TimeZone.getDefault().getID());
        log.info(" Locale.getDefault() = {}",Locale.getDefault());
        julLogger.info("java.util.logging successfully set up.");
        aclLog.info("org.apache.commons.logging successfully set up.");

        String etcPathProp = System.getProperty(ETC_PATH_PROPERTY);
        Path etcPath;

        if (etcPathProp == null) {

            etcPath = null;

            log.info("System property ["+ETC_PATH_PROPERTY+"] is not set, will not read configuration from disk.");
        }
        else {
            etcPath = Paths.get(etcPathProp);

            if (!Files.isDirectory(etcPath)) {
                throw new RuntimeException("Config etc path ["+etcPath+"] does not exist.");
            }

            log.info("Determined svc-runner etc path ["+etcPath+"].");
        }

        etcDir = etcPath;

        try {
            loadSystemProperties();
            config = loadConfigProperties();

            // unresolved layer configs.
            layerConfigs = new TreeMap<String,LayerConfig>();

            for (Entry<String, String> e : config.entrySet()) {

                String k = e.getKey();

                if (k.startsWith(MODULE_PATH_PROPERTY_PREFIX) &&
                    k.length() > MODULE_PATH_PROPERTY_PREFIX.length()) {

                    String lbl = k.substring(MODULE_PATH_PROPERTY_PREFIX.length());

                    if (MODULE_LAYER_LABEL_RX.matcher(lbl).matches()) {

                        String parent = config.get(MODULE_PARENT_PROPERTY_PREFIX+lbl);

                        String opensPfx = MODULE_OPENS_PROPERTY_PREFIX+lbl+".";

                        StringBuilder opensSpec = new StringBuilder();

                        for (Entry<String, String> te : config.tailMap(opensPfx).entrySet()) {

                            if (!te.getKey().startsWith(opensPfx)) {
                                break;
                            }

                            if (!opensSpec.isEmpty()) {
                                opensSpec.append(',');
                            }
                            opensSpec.append(te.getValue());
                        }

                        Path mp = Path.of(e.getValue());

                        List<OpensDirective> opens = OpensDirective.ofSpecList(opensSpec.toString());

                        log.info("Adding [{}] module path [{}] with parent [{}] an opens {}",
                                 lbl,mp,parent,opens);

                        layerConfigs.put(lbl,LayerConfig.ofUnresolved(lbl,mp,parent,opens));
                    }
                    else {
                        log.warn("Ignoring module path [{}] with invalid label [{}] (must match [{}]).",
                            e.getValue(),lbl,MODULE_LAYER_LABEL_RX);
                    }
                }
            }

            sortedLayerConfigs = new TreeMap<String,LayerConfig>();

            for (Entry<String, LayerConfig> e : layerConfigs.entrySet()) {

                LayerConfig layerConfig = e.getValue();

                String key = layerConfig.resolve(layerConfigs);

                log.info("Resolved layer [{}] to key [{}]",layerConfig.getLabel(),key);
                sortedLayerConfigs.put(key,layerConfig);
            }

        } catch (IOException e) {
            throw new RuntimeException("I/O error loading svc-runner config.",e);
        }
    }

    /**
     * @return The configured configuration directory path or <code>null</code>,
     *         if no such path has been configured through the system property
     *         {@value #ETC_PATH_PROPERTY}. The second case happens, when we are
     *         run as part of a test case.
     */
    public static Path getEtcDir() {
        return etcDir;
    }

    /**
     * @return The maximal number of throwables cached for a throwable info.
     */
    public static int getThrowableHistorySize() {
        return getIntProperty(THROWABLE_HISTORY_SIZE_PROPERTY,10);
    }

    /**
     * @return A map of additional module paths by layer label.
     */
    public static Map<String,LayerConfig> getLayerConfigs() {
        return layerConfigs;
    }

    /**
     * Return the configured layers in initialization order.
     * @return A map of additional module paths sorted by layer path.
     */
    public static Map<String,LayerConfig> getSortedLayerConfigs() {
        return sortedLayerConfigs;
    }

    /**
     * Return the value of a configuration property.
     * @param key The name of the property to fetch.
     * @return The value of the property or <code>null</code>, if the
     *         property has not been found.
     */
    public static String getProperty(String key) {
        return config.get(key);
    }

    /**
     * Return the value of a configuration property.
     * @param key The name of the property to fetch.
     * @param defaultValue The default value.
     * @return The value of the property or <code>defaultValue</code>,
     *         if the property has not been found.
     */
    public static String getProperty(String key, String defaultValue) {

        String ret = config.get(key);
        return ret == null ? defaultValue : ret;
    }

    /**
     * Return the value of an integer configuration priperty
     * @param key The name of the property to fetch.
     * @param defaultValue The default value.
     * @return The int value of the property or <code>defaultValue</code>,
     *         if the property has not been found.
     */
    public static int getIntProperty(String key, int defaultValue) {
        String ret = config.get(key);
        return ret == null ? defaultValue : Integer.parseInt(ret);
    }

    /**
     * Return the value of a boolean configuration priperty
     * @param key The name of the property to fetch.
     * @param defaultValue The default value.
     * @return The int value of the property or <code>defaultValue</code>,
     *         if the property has not been found.
     */
    public static boolean getBooleanProperty(String key, boolean defaultValue) {
        String ret = config.get(key);
        return ret == null ? defaultValue : Boolean.parseBoolean(ret);
    }

    /**
     * SVC initialization step 1.
     *
     * Load system properties from <code>${svc.runner.etcPath}/system.properties</code>,
     * if such a file exists.
     *
     * @throws IOException Upon errors.
     */
    private static void loadSystemProperties() throws IOException {

        if (etcDir == null) {
            return;
        }

        Properties props = new Properties();

        Path confFile = etcDir.resolve("system.properties");

        if (Files.exists(confFile)) {

            log.info("Loading system properties from ["+confFile+"].");

            try (InputStream is = Files.newInputStream(confFile)) {
                props.load(new InputStreamReader(is,"UTF-8"));
            }

            for (Map.Entry<Object,Object> entry: props.entrySet()) {

                String k = entry.getKey().toString();
                String v = VariableSubstition.substSystemProperties(entry.getValue().toString(),k);

                System.setProperty(k,v);
            }
        }
    }

    private static final void loadProperties(Properties props, Path confDir) throws IOException {

        List<Path> configFiles = new ArrayList<Path>();

        Files.
            newDirectoryStream(confDir,"*.properties").
            forEach(p -> configFiles.add(p));

        Collections.sort(configFiles);

        for (Path configFile : configFiles) {

            log.info("Loading SVC properties from ["+configFile+"].");

            try (InputStream is = Files.newInputStream(configFile)) {
                props.load(new InputStreamReader(is,"UTF-8"));
            }
        }
    }

    /**
     * SVC-Runner initialization step 2.
     *
     * Load SVC configuration properties from
     * <code>${svc.runner.etcPath}/conf.d/*.properties</code>,
     * if such a file exists.
     *
     * @throws IOException Upon errors.
     */
    private static SortedMap<String,String> loadConfigProperties() throws IOException {

        Properties props = new Properties();

        if (etcDir != null) {
            Path confDir = etcDir.resolve("conf.d");

            if (Files.isDirectory(confDir)) {
                loadProperties(props,confDir);
            }
        }

        SortedMap<String,String> ret = new TreeMap<String, String>();

        for (Enumeration<?> e = System.getProperties().propertyNames();e.hasMoreElements();) {

            String key = e.nextElement().toString();

            if (key.startsWith("svc.")) {

                ret.put(key,System.getProperty(key));
            }
        }

        for (Object key: props.keySet()) {

            String k = key.toString();

            String v = VariableSubstition.substConfigProps(k,props);

            ret.put(k,v);
        }

        return ret;
    }

    /**
     * Load secrets properties in
     * <code>${svc.runner.etcPath}/secrets.d/*.properties</code>
     * @param props The properties object to fill.
     * @throws UncheckedIOException Upon read errors.
     */
    public static void loadSecretsProperties(Properties props) {

        if (etcDir != null) {

            Path confDir = etcDir.resolve("secrets.d");

            if (Files.isDirectory(confDir)) {

                try {
                    loadProperties(props,confDir);
                } catch (IOException e) {
                    throw new UncheckedIOException("Unable to load secrets from ["+confDir+"]",e);
                }
            }
        }
    }

    /**
     * Split a comma-separated string into parts and trim each part by
     * removing leading and trailing whitespace.
     *
     * @param s A string possibly containing comma separators.
     * @return An array of the trimmed, comma-separated parts or <code>null</code>,
     *         if the input was <code>null</code>.
     */
    public static String[] splitCommaSeparatedString(String s) {

        if (s == null) {
            return null;
        }
        else {
            return s.trim().split("\\s*,\\s*");
        }
    }

    /**
     * SVC initialization step 3.
     *
     * Load all security providers specified in the OSGI framework
     * configuration property <code>svc.runner.securityProviders</code>,
     * which bears a comma-separated list of class names.
     *
     * @throws Exception Upon class loading or access control problems.
     */
    public static void registerSecurityProviders() throws Exception {

        String secProvProp = getProperty(SECURITY_PROVIDERS_PROPERTY);

        if (secProvProp == null) {
            return;
        }

        ClassLoader cl = Config.class.getClassLoader();

        String[] classNames = splitCommaSeparatedString(secProvProp);

        for (String className:classNames) {

            if (className.isEmpty()) {
                continue;
            }

            log.info("Registering security provider ["+className+"].");

            @SuppressWarnings("unchecked")
            Class <? extends Provider> providerClass = (Class<? extends Provider>)cl.loadClass(className);

            Security.addProvider(providerClass.getDeclaredConstructor().newInstance());
        }
    }

}
