/***********************************************************
*
* Service Runner of the clazzes.org project
* https://www.clazzes.org
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
***********************************************************/


package org.clazzes.svc.runner;

import java.lang.module.ModuleDescriptor.Version;
import java.util.Optional;
import java.util.regex.Pattern;

public class ComponentFilter {

    private final String layer;
    private final Pattern namePattern;
    private final Version moduleVersion;


    /**
     * Poor man's glob support.
     * @param glob a pattern, which might contain <code>*</code> wildcards.
     * @return A pattern reprenting the given glob.
     */
    protected static final Pattern pmGlob(String glob) {


        String[] parts = glob.split("\\*");

        int i=0;

        StringBuilder sb = new StringBuilder(Pattern.quote(parts[i++]));

        for (;i<parts.length;++i) {

            sb.append(".*");
            sb.append(Pattern.quote(parts[i]));
        }

        return Pattern.compile(sb.toString());
    }

    public ComponentFilter(String layer, Pattern namePattern, Version moduleVersion) {
        this.layer = layer;
        this.namePattern = namePattern;
        this.moduleVersion = moduleVersion;
    }

    public static ComponentFilter of(String filter) {

        int spos = filter.indexOf('/');

        String layer;
        String nameAndVersion;

        if (spos < 0) {
            layer = null;
            nameAndVersion = filter;
        }
        else {
            layer = filter.substring(0, spos);
            nameAndVersion = filter.substring(spos+1);
        }

        int atpos = nameAndVersion.lastIndexOf('@');

        Pattern namePattern;
        Version moduleVersion;

        if (atpos < 0) {
            namePattern = pmGlob(nameAndVersion);
            moduleVersion = null;
        }
        else {
            namePattern = pmGlob(nameAndVersion.substring(0,atpos));
            moduleVersion = Version.parse(nameAndVersion.substring(atpos+1));
        }

        return new ComponentFilter(layer,namePattern,moduleVersion);
    }

    public String getLayer() {
        return this.layer;
    }

    public Pattern getNamePattern() {
        return this.namePattern;
    }

    public Version getModuleVersion() {
        return this.moduleVersion;
    }

    public boolean matches(String layer, ComponentHolder h) {

        return this.matches(layer,h.getClassName(),h.getModuleVersion());
    }

    public boolean matches(String layer, String moduleName, Optional<Version> v) {

        if (this.layer != null && !this.layer.equals(layer)) {
            return false;
        }

        if (!this.namePattern.matcher(moduleName).matches()) {
            return false;
        }

        if (this.moduleVersion != null &&
            (v.isEmpty() ||
            !this.moduleVersion.equals(v.get()))) {
            return false;
        }

        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();

        if (this.layer != null) {
            sb.append(this.layer);
            sb.append("/");
        }

        sb.append(this.namePattern);

        if (this.moduleVersion != null) {
            sb.append("@");
            sb.append(this.moduleVersion);
        }

        return sb.toString();
    }
}
