/***********************************************************
 *
 * SVC module API of the clazzes.org project
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.svc.module.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.clazzes.svc.module.api.PluginDescription;
import org.clazzes.svc.module.api.PluginService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>This service is registereed as a svc-api ServiceRegistry service
 * under the key <code>svc-module-api</code> and should be used like:
 * </p>
 * <pre>
 *   ServiceProxyFactory.getServiceProxy(registry,PluginService.class)
 * </pre>
 */
public class PluginServiceImpl implements PluginService {

    private static final Logger log = LoggerFactory.getLogger(PluginServiceImpl.class);

    private final Map<String,List<PluginDescription>> pluginsByProject;

    private static final Comparator<PluginDescription> PC =
        new Comparator<PluginDescription>() {

        @Override
        public int compare(PluginDescription o1, PluginDescription o2) {

            int r;

            // reverse order by priority.
            if (o1.getPriority() == null) {
                r = o2.getPriority() == null ? 0 : 1;
            }
            else {
                if (o2.getPriority() == null) {
                    r = -1;
                }
                else {
                    r = o2.getPriority().compareTo(o1.getPriority());
                }
            }

            if (r != 0) {
                return r;
            }

            // alphabetic order by name.
            return o1.getName().compareTo(o2.getName());
        }
    };

    public PluginServiceImpl() {
        this.pluginsByProject = new ConcurrentHashMap<String,List<PluginDescription>>();
    }

    public void pluginBound(String key, PluginDescription plugin) {

        String project = plugin.getProject();

        List<PluginDescription> plugins =
            this.pluginsByProject.computeIfAbsent(
                project,
                (k) -> new ArrayList<PluginDescription>()
            );

        log.info("Bound [{}] for key [{}]",plugin,key);

        synchronized (plugins) {
            int i = Collections.binarySearch(plugins,plugin,PC);

            if (i >= 0) {
                throw new IllegalArgumentException("Duplicate registration of plugin for project ["+project+"], name ["+plugin.getName()+"] and priority ["+plugin.getPriority()+"]");
            }

            int insertion = -i-1;

            if (log.isDebugEnabled()) {
                log.debug("Inserted [{}] at index [{}]",plugin,insertion);
            }

            // insert either after exact match (i>=0)
            // or at insertion point (i<0).
            plugins.add(insertion,plugin);
        }
    }

    public void pluginUnbound(String key, PluginDescription plugin) {

        String project = plugin.getProject();

        List<PluginDescription> plugins =
            this.pluginsByProject.get(project);

        if (plugins == null) {

            return;
        }

        log.info("Unbound [{}] for key [{}]",plugin,key);

        synchronized (plugins) {

            int i = Collections.binarySearch(plugins,plugin,PC);

            if (i >= 0) {

                if (log.isDebugEnabled()) {
                    log.debug("Removing [{}] at index [{}]",plugin,i);
                }
                plugins.remove(i);
            }
            else {
                log.warn("Plugin [{}] not found.",plugin);
            }
        }
    }

    @Override
    public List<PluginDescription> getAllForProject(String project) {

        List<PluginDescription> plugins = this.pluginsByProject.get(project);

        if (plugins == null) {
            return List.of();
        }

        synchronized (plugins) {

            return new ArrayList<PluginDescription>(plugins);
        }
    }


}
