/***********************************************************
 *
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 17.04.2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.odf.util.text;

import org.odftoolkit.odfdom.dom.element.draw.DrawTextBoxElement;
import org.odftoolkit.odfdom.dom.element.text.TextAElement;
import org.odftoolkit.odfdom.dom.element.text.TextHElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.element.text.TextPageNumberElement;
import org.odftoolkit.odfdom.dom.element.text.TextSpanElement;
import org.odftoolkit.odfdom.dom.element.text.TextTabElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

public class TextFactory {

    public static TextPElement constructTextNode(Node parentNode, String textContent, String textStyle) {
        TextPElement textElement = (TextPElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextPElement.class);

        if (textContent != null) {
            textElement.setTextContent(textContent);
        }
        if (textStyle != null) {
            textElement.setTextStyleNameAttribute(textStyle);
        }

        parentNode.appendChild(textElement);
        return textElement;
    }

    public static TextAElement constructTextAElement(Node parentNode, String xlinkType, String href, String style, String visitedStyle) {
        TextAElement textElement = (TextAElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextAElement.class);
        if (xlinkType != null) {
            textElement.setXlinkTypeAttribute(xlinkType);
        }
        if (href != null) {
            textElement.setXlinkHrefAttribute(href);
        }
        if (style != null) {
            textElement.setStyleName(style);
        }
        if (visitedStyle != null) {
            textElement.setTextVisitedStyleNameAttribute(visitedStyle);
        }

        parentNode.appendChild(textElement);
        return textElement;
    }

    public static TextSpanElement constructTextSpan(Node parentNode, String textContent, String textStyle) {
        TextSpanElement textElement = (TextSpanElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextSpanElement.class);

        if (textContent != null) {
            textElement.setTextContent(textContent);
        }
        if (textStyle != null) {
            textElement.setTextStyleNameAttribute(textStyle);
        }

        parentNode.appendChild(textElement);
        return textElement;
    }

    public static TextHElement constructTextHeaderNode(Node parentNode, String textContent, String textStyle, Integer outlineLevel) {
        TextHElement hElement = (TextHElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextHElement.class);
        if (textContent != null) {
            hElement.setTextContent(textContent);
        }
        if (outlineLevel != null) {
            hElement.setTextOutlineLevelAttribute(outlineLevel);
        }
        if (textStyle != null) {
            hElement.setTextStyleNameAttribute(textStyle);
        }

        parentNode.appendChild(hElement);
        return hElement;

    }

    public static void appendEmptyTextNode(Node parentNode, String textStyle) {
        appendEmptyTextNodes(parentNode, textStyle, 1);
    }

    public static void appendEmptyTextNodes(Node parentNode, String textStyle, int count) {
        for (int n = 0; n < count; n++) {
            constructTextNode(parentNode, null, textStyle);
        }
    }

    public static void appendSimpleTextNode(Node parentNode, String text) {
        Text textNode = ((OdfFileDom)parentNode.getOwnerDocument()).createTextNode(text);
        parentNode.appendChild(textNode);
    }

    public static void appendTabNode(Node parentNode) {
        TextTabElement tabElement = (TextTabElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextTabElement.class);
        parentNode.appendChild(tabElement);
    }

    public static DrawTextBoxElement constructTextBox(Node parentNode, String minHeight) {
        DrawTextBoxElement textBoxElement = (DrawTextBoxElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(DrawTextBoxElement.class);

        if (minHeight != null) {
            textBoxElement.setFoMinHeightAttribute(minHeight);
        }

        parentNode.appendChild(textBoxElement);
        return textBoxElement;
    }

    public static void appendText(TextPElement textElement, String content) {
        Text textNode = textElement.getOwnerDocument().createTextNode(content);
        textElement.appendChild(textNode);
    }

    public static void appendText(TextSpanElement textElement, String content) {
        Text textNode = textElement.getOwnerDocument().createTextNode(content);
        textElement.appendChild(textNode);
    }

    /** Constructs a node containing the text <code>&lt;prefix&gt;&lt;pageNumber&gt;/&lt;totalNumberOfPages&gt;</code>, e.g. Seite 5/12.
     *
     * @param parentNode
     * @param prefix
     * @param textStyle
     * @return
     */
    public static TextPElement constructPageOfPagesNode(Node parentNode, String prefix, String textStyle) {
        TextPElement textElement = (TextPElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextPElement.class);

        TextFactory.appendText(textElement, prefix + " ");

        TextPageNumberElement pageNumberElement = textElement.newTextPageNumberElement("1");
        pageNumberElement.setTextSelectPageAttribute("current");

        TextFactory.appendText(textElement, "/");

        textElement.newTextPageCountElement("1");

        if (textStyle != null) {
            textElement.setTextStyleNameAttribute(textStyle);
        }

        parentNode.appendChild(textElement);
        return textElement;
    }

    public static TextSpanElement constructPageOfPagesSpan(Node parentNode, String prefix, String textStyle) {
        TextSpanElement textElement = (TextSpanElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextSpanElement.class);

        TextFactory.appendText(textElement, prefix + " ");

        TextPageNumberElement pageNumberElement = textElement.newTextPageNumberElement("1");
        pageNumberElement.setTextSelectPageAttribute("current");

        TextFactory.appendText(textElement, "/");

        textElement.newTextPageCountElement("1");

        if (textStyle != null) {
            textElement.setTextStyleNameAttribute(textStyle);
        }

        parentNode.appendChild(textElement);
        return textElement;
    }
}
