/***********************************************************
 * $Id: $
 * 
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 07.12.2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.odf.util.text;

import java.util.Map;
import java.util.ResourceBundle;

import org.odftoolkit.odfdom.dom.element.text.TextLineBreakElement;
import org.odftoolkit.odfdom.dom.element.text.TextSpanElement;
import org.odftoolkit.odfdom.dom.element.text.TextTabElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;

public class CompositeTextFactory {

    /** The purpose of this function is to internationalize whole sentences where certain words etc. need a certain format.
     *  However, the order of such words is not guaranteed to be equal in different languages, i.e. it cannot be done 
     *  by simply concatenating fixed strings.
     *  <p>
     *  This function works based on the string mapped to the given i18nPrefix in the given resourceBundle.
     *  It may e.g. look like
     *  <code>
     *  One,Data.challenge1.span.challengeStyle,Two,Data.date,Three,Four.span.cdesLabelStyle,Five
     *  </code>
     *  The major token delimiter is the comma.  Each of the major tokens represents one token to be added to the final string.
     *  The minor token delimiter is the dot.  Each of the minor tokens represents a piece of information specifying how the token
     *  at hand is to be added to the final string.
     *  <p>
     *  If the first minor token is the empty string, it will be ignored, and the second minor token is regraded as kind of tag.
     *  If the first minor token is the string <code>Data</code>, then the token will not be fetched from the resource bundle, but
     *  from the given dataMap.  In that case, the key is the second minor token.  
     *  For any other minor token value, the token is fetched from the resource bundle, by concatenating the given i18nPrefix with 
     *  the first minor token.
     *  <p>
     *  The next (third or second, depending on the case before) token is optional and specifies the kind of tag to be used, e.g.
     *  a text:span.  If omitted, the token will be appended as simple text node.  Spaces will be inserted between subsequent 
     *  simple text nodes automatically.
     *  <p>
     *  The next token is optional (and may only be specified if the tag token is given), and specifies a style to use.  Styles
     *  must be created by the calling code.  The token will be used as key to fetch the actual style name from the given style map.
     *  <p>
     *  Regardless of how the text tokens are generated (dataMap or resourceBundle), tokens will automatically be prefixed with a
     *  space.  Exceptions from that rules: The very first token, and tokens starting with dot, comma and semicolon.
     * 
     * @param parentNode The parent node, where all nodes described above will be appended as children
     * @param resourceBundle the resource bundle with i18n
     * @param i18nPrefix The i18n prefix as described above
     * @param dataMap The map where data strings will be fetched from
     * @param styleMap The map from style specifiers in the token string, to actual style names.
     */
    public static void appendCompositeText(Node parentNode, ResourceBundle resourceBundle, String i18nPrefix, Map<String, String> dataMap, Map<String, String> styleMap) {
        String specString = resourceBundle.getString(i18nPrefix);
        String[] majorTokens = specString.split(",");
        for (int majorTokenIndex = 0; majorTokenIndex < majorTokens.length; majorTokenIndex++) {
            String majorToken = majorTokens[majorTokenIndex];
            String[] minorTokens = majorToken.split("\\.");
            
            if (minorTokens.length >= 1) {
                String source = minorTokens[0];
                
                Integer currIndex = null; 
                String text = null;
                if (source.trim().length() == 0) {
                    text = "";
                    currIndex = 1;
                } else if ("Data".equals(source)) {
                    if (minorTokens.length >= 2) {
                        if (!(dataMap.containsKey(minorTokens[1]))) {
                            throw new IllegalArgumentException("Expected key [" + minorTokens[1] + "] in dataMap, didn't find it.");
                        }
                        
                        text = dataMap.get(minorTokens[1]);
                        
                        // Allow for unknown data values (i.e. not set in database).
                        if (text == null) {
                            text = "null";
                        }
                        
                        currIndex = 2;
                    }
                } else {
                    text = resourceBundle.getString(i18nPrefix + source);
                    currIndex = 1;

                    if (text == null) {
                        throw new IllegalArgumentException("Expected key [" + i18nPrefix + source + "] in resourceBundle, didn't find it.");
                    }
                }
                if (majorTokenIndex > 0 && text.length() > 0 && !(text.startsWith(".") || text.startsWith(",") || text.startsWith(";"))) {
                    text = " " + text;
                }
                
                if (text != null) {
                    if (currIndex < minorTokens.length) {
                        String tag = minorTokens[currIndex++];
                        String style = null;
                        if (currIndex < minorTokens.length) {
                            String styleKey = minorTokens[currIndex++];
                            if (!(styleMap.containsKey(styleKey))) {
                                throw new IllegalArgumentException("Expected key [" + styleKey + "] in styleMap, didn't find it.");
                            }
                            
                            style = styleMap.get(styleKey);
                        }
                        CompositeTextFactory.appendTextWithTagAndStyle(parentNode, text, tag, style);
                    } else {
                        TextFactory.appendSimpleTextNode(parentNode, text);
                    }
                }
            }
            
        }
    }
    
    public static void appendTextWithTagAndStyle(Node parentNode, String text, String tag, String style) {

        if ("span".equals(tag)) {
            TextSpanElement span = (TextSpanElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextSpanElement.class);
            if (style != null) {
                span.setTextStyleNameAttribute(style);
            }
            if (text.length() > 0) {
                TextFactory.appendSimpleTextNode(span, text);
            }
            parentNode.appendChild(span);
        } else if ("tab".equals(tag)) {
            TextTabElement tab = (TextTabElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextTabElement.class);
            if (text.length() > 0) {
                TextFactory.appendSimpleTextNode(tab, text);    
            }
            parentNode.appendChild(tab);
        } else if ("line-break".equals(tag)) {
            TextLineBreakElement lineBreak = (TextLineBreakElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TextLineBreakElement.class);
            if (text.length() > 0) {
                TextFactory.appendSimpleTextNode(lineBreak, text);    
            }
            parentNode.appendChild(lineBreak);            
        }
    }
}
