/***********************************************************
 * $Id: $
 * 
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 10.04.2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.odf.util.table;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

import org.apache.commons.lang3.time.FastDateFormat;
import org.clazzes.odf.util.style.Styles;
import org.clazzes.odf.util.text.TextFactory;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.datetime.UtcTimestamp;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class TableFactory {
    
    //public static StyleTableColumnPropertiesElement appendColumnStyle()
    
    public static TableTableElement constructTableElement(Node parentNode, String style) {
        TableTableElement tableElement = (TableTableElement)((OdfFileDom)parentNode.getOwnerDocument()).newOdfElement(TableTableElement.class);
        
        if (style != null) {
            tableElement.setTableStyleNameAttribute(style);
        }
        
        parentNode.appendChild(tableElement);
        
        return tableElement;
    }
    
    public static TableTableRowElement appendRow(TableTableElement tableElement, String rowStyle) {
        TableTableRowElement rowElement = tableElement.newTableTableRowElement();
        if (rowStyle != null) {
            rowElement.setTableStyleNameAttribute(rowStyle);    
        }        
        return rowElement;
    }
    
    public static TableTableCellElement appendCell(TableTableRowElement rowElement, String... values) {
        return TableFactory.appendCell(null, null, rowElement, values);
    }
    
    public static TableTableCellElement appendTextStyledCell(String textStyle, TableTableRowElement rowElement, String... values) {
        return TableFactory.appendCell(null, textStyle, rowElement, values);
    }
    
    public static TableTableCellElement appendCellStyledCell(String cellStyle, TableTableRowElement rowElement, String... values) {
        return TableFactory.appendCell(cellStyle, null, rowElement, values);
    }
    
    public static TableTableCellElement appendCell(String cellStyle, String textStyle, TableTableRowElement rowElement, String... values) {
        TableTableCellElement cellElement = rowElement.newTableTableCellElement(0, "string");
        
        // Unfortunately, newTableTableCellElement can only create cells with office:value set.  As we don´t want
        // office:value in a pure text cell, we remove it here.
        cellElement.removeAttribute("office:value");
        
        if (cellStyle != null) {
            cellElement.setTableStyleNameAttribute(cellStyle);    
        }        
        
        for (String value : values) {
            // value can be null (this will be adressed inside constructTextNode). 
            // Allowing to pass null values is convenient for the caller code
            TextFactory.constructTextNode(cellElement, value, textStyle);
        }
        
        return cellElement;
    }
    
    public static TableTableCellElement appendCell(String cellStyle, String textStyle, TableTableRowElement rowElement, List<String> values) {
        TableTableCellElement cellElement = rowElement.newTableTableCellElement(0, "string");
        
        // Unfortunately, newTableTableCellElement can only create cells with office:value set.  As we don´t want
        // office:value in a pure text cell, we remove it here.
        cellElement.removeAttribute("office:value");
        
        if (cellStyle != null) {
            cellElement.setTableStyleNameAttribute(cellStyle);    
        }        
        
        for (String value : values) {
            // value can be null (this will be adressed inside constructTextNode). 
            // Allowing to pass null values is convenient for the caller code
            TextFactory.constructTextNode(cellElement, value, textStyle);
        }
        
        return cellElement;        
    }  
    
    public static TableTableCellElement appendEmptyCell(String cellStyle, String textStyle, TableTableRowElement rowElement) {
        TableTableCellElement cellElement = rowElement.newTableTableCellElement(0, "string");
        
        // Unfortunately, newTableTableCellElement can only create cells with office:value set.  As we don´t want
        // office:value in a pure text cell, we remove it here.
        cellElement.removeAttribute("office:value");
        
        if (cellStyle != null) {
            cellElement.setTableStyleNameAttribute(cellStyle);    
        }        

        return cellElement;        
    }
    
    public static TableTableCellElement appendCell(String cellStyle, String textStyle, TableTableRowElement rowElement, int number) {
        return TableFactory.appendCell(cellStyle, textStyle, rowElement, new Double(number));
    }
        
    public static TableTableCellElement appendCell(String cellStyle, String textStyle, TableTableRowElement rowElement, Double number) {        
        if (number == null) {
            return TableFactory.appendEmptyCell(cellStyle, textStyle, rowElement);
        } else {
            TableTableCellElement cellElement = rowElement.newTableTableCellElement(number, "float");
            
            if (cellStyle != null) {
                cellElement.setTableStyleNameAttribute(cellStyle);    
            }
            
            return cellElement;            
        }
    }
    
    public static TableTableCellElement appendPageOfPagesCell(String cellStyle, String textStyle, TableTableRowElement rowElement, String prefix) {
        TableTableCellElement cellElement = rowElement.newTableTableCellElement(0, "string");
        
        // Unfortunately, newTableTableCellElement can only create cells with office:value set.  As we don´t want
        // office:value in a pure text cell, we remove it here.
        cellElement.removeAttribute("office:value");
        
        if (cellStyle != null) {
            cellElement.setTableStyleNameAttribute(cellStyle);
        }        
        
        TextFactory.constructPageOfPagesNode(cellElement, prefix, textStyle);
        return cellElement;
    }
    
    public static TableTableColumnElement appendColumnWithRelativeSize(Styles context, TableTableElement tableElement, String relativeSize) {
        String tableColumnStyle = context.getTableColumnStyle(context.constructTableColumnPropertiesWithRelativeWidth(relativeSize)); 
        
        TableTableColumnElement columnElement = tableElement.newTableTableColumnElement();
        columnElement.setStyleName(tableColumnStyle);
        
        return columnElement;
    }
    
    public static TableTableColumnElement appendColumnWithAbsoluteSize(Styles context, TableTableElement tableElement, String absoluteSize) {
        String tableColumnStyle = context.getTableColumnStyle(context.constructTableColumnPropertiesWithAbsoluteWidth(absoluteSize)); 
        
        TableTableColumnElement columnElement = tableElement.newTableTableColumnElement();
        columnElement.setStyleName(tableColumnStyle);
        
        return columnElement;        
    }
    
    public static TableTableColumnElement appendColumnWithOptimalSize(Styles context, TableTableElement tableElement) {
        String tableColumnStyle = context.getTableColumnStyle(context.constructTableColumnPropertiesWithOptimalWidth()); 
        
        TableTableColumnElement columnElement = tableElement.newTableTableColumnElement();
        columnElement.setStyleName(tableColumnStyle);
        
        return columnElement;        
    }
    
    public static TableTableColumnElement appendColumnWithNoSize(TableTableElement tableElement) {
        TableTableColumnElement columnElement = tableElement.newTableTableColumnElement();
        return columnElement;
    }
    
    public static void appendDateCellAsString(String cellStyle, String textStyle, TableTableRowElement rowElement, Double utcSeconds, String timeZone, String formatPattern) {
        UtcTimestamp utcTimeStamp = new UtcTimestamp(TimeZone.getTimeZone("Europe/Vienna"), (new Double(utcSeconds * 1000)).longValue());

        Calendar calendar = utcTimeStamp.toCalendar();
        Locale locale = ThreadLocalManager.getLoginLocale();
        
        FastDateFormat format = FastDateFormat.getInstance(formatPattern, locale);                         
        String formattedDateString = format.format(calendar);               
        
        TableFactory.appendCell(cellStyle, textStyle, rowElement, formattedDateString);        
    }
    
    public static List<TableTableColumnElement> getColumnElements(TableTableElement tableElement) {
        List<TableTableColumnElement> columnElements = new ArrayList<TableTableColumnElement>();
        
        NodeList childNodeList = tableElement.getChildNodes();
        for (int n = 0; n < childNodeList.getLength(); n++) {
            Node childNode = childNodeList.item(n);
            
        }
        
        return columnElements;
    }
    
    /** Cleans and prepares a TableTableElement loaded from a draft ods template file
     *  for usage e.g. with a FancyTable.
     *  Background: Such ods files can contain a single row and a single column, even
     *  if they look empty.  This function programmatically removes all TableTableRowElements
     *  and TableTableColumnElements of the TableTableElement, in order to avoid rows and
     *  columns showing up in an unexpected manner. 
     * 
     * @param tableElement some TableTableElement
     */
    public static void cleanTableElementBeforeUsage(TableTableElement tableElement) {
        NodeList children = tableElement.getChildNodes();
        List<Node> childrenToBeRemoved = new ArrayList<Node>();
        for (int n = 0; n < children.getLength(); n++) {
            Node child = children.item(n);
            if (child instanceof TableTableRowElement || child instanceof TableTableColumnElement) {
                childrenToBeRemoved.add(child);
            }
        }
        for (Node childToBeRemoved : childrenToBeRemoved) {
            tableElement.removeChild(childToBeRemoved); 
        }        
    }

}
