/*
 *  $Id: OdsHelper.java 1302 2013-06-14 10:55:23Z mda $
 *  
 *  Copyright (C) ITEG IT-Engineers GmbH 2012
 */
package org.clazzes.odf.util.table;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;

import org.clazzes.util.datetime.UtcTimestamp;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.incubator.doc.text.OdfWhitespaceProcessor;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * @author wpausch
 *
 */
public class OdsHelper {
	
	public static boolean isDateCell(TableTableCellElement cell) {
		String type = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value-type");		
		
		if (type == null || type.trim().length() == 0 || !(type.equals("date"))) {
			return false;
		} else {
			String dateAsString = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "date-value");
			if (dateAsString == null || dateAsString.trim().length() == 0) {
				return false;
			}
		}
		return true;
	}
	
	public static Long getUtcMillis(TableTableCellElement cell, TimeZone timeZone) throws ParseException {
		String dateAsString = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "date-value");
		
		UtcTimestamp utcTimestamp = new UtcTimestamp(dateAsString, timeZone);
		return utcTimestamp.getUtcMillis();
	}
	
	public static boolean isStringCell(TableTableCellElement cell) {
		String type = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value-type");
		return type != null && type.equals("string");
	}
	
	public static String getStringValue(TableTableCellElement cell) {
		OdfWhitespaceProcessor textProcessor = new OdfWhitespaceProcessor();
		return textProcessor.getText(cell);
	}
	
	public static boolean isFloatCell(TableTableCellElement cell) {
		String type = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value-type");
		return type != null && type.equals("float");
	}
	
	public static String getOfficeValue(TableTableCellElement cell) {
		return cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value");
	}
	
	public static int getNumberOfRowsRepeated(TableTableRowElement cell) {
		String rowsRepeated = cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeated");
		if (rowsRepeated == null || "".equals(rowsRepeated)) {
			return 1;
		} else {
			try {
				return Integer.parseInt(rowsRepeated);
			} catch (NumberFormatException e) {
				return 1;
			}
		}
	}
	
	public static int getNumberOfColumnsRepeated(TableTableCellElement cell) {
		String columnsRepeated = cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated");
		if (columnsRepeated == null || "".equals(columnsRepeated)) {
			return 1;
		} else {
			try {
				return Integer.parseInt(columnsRepeated);
			} catch (NumberFormatException e) {
				return 1;
			}
		}		
	}
	
	public static Double getFloatValue(TableTableCellElement cell) {
		String value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value");
		try {
			return Double.parseDouble(value);
		} catch (NumberFormatException e) {
			return null;
		}
	}
	
	public static boolean isEmptyCell(TableTableCellElement cell) {
		String type = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value-type");
		if (type == null || type.trim().length() == 0) {
			return true;
		}
		if (OdsHelper.isStringCell(cell)) {
			String text = OdsHelper.getStringValue(cell);
			return text == null || text.trim().length() == 0;
		}
		
		return false;
	}	
	
	public static List<List<TableTableCellElement>> getTableData(TableTableElement tableElement) {
		List<List<TableTableCellElement>> data = new ArrayList<List<TableTableCellElement>>();
		
		List<Integer> rowsRepeatedCounts = new ArrayList<Integer>();
		
		NodeList rowNodes = tableElement.getChildNodes();
		for (int n = 0; n < rowNodes.getLength(); n++) {
			Node rowNode = rowNodes.item(n);

			if (rowNode instanceof TableTableRowElement) {
				List<TableTableCellElement> currCells = new ArrayList<TableTableCellElement>();
				NodeList cellNodes = rowNode.getChildNodes();

				for (int z = 0; z < cellNodes.getLength(); z++) {
					Node cellNode = cellNodes.item(z);
					if (cellNode instanceof TableTableCellElement) {
						currCells.add((TableTableCellElement)cellNode);
					}
				}
				
				rowsRepeatedCounts.add(OdsHelper.getNumberOfRowsRepeated((TableTableRowElement)rowNode));
				data.add(currCells);
			}
		}
		
		for (int n = 0; n < data.size(); n++) {
			List<TableTableCellElement> line = data.get(n);
			
			boolean alreadyNonEmptyCellFound = false;
			for (int z = line.size() - 1; z >= 0; z--) {
				TableTableCellElement cell = line.get(z);
				int columnsRepeated = OdsHelper.getNumberOfColumnsRepeated(cell);
				alreadyNonEmptyCellFound |= !(OdsHelper.isEmptyCell(cell));
				
				// Only duplicate non-empty cells (or cells where right of them is a non-empty cell), 
				// to avoid a known problem related to very high repeated counts at the end of a line
				if (alreadyNonEmptyCellFound) {
					for (int i = 1; i < columnsRepeated; i++) {
						line.add(z, cell);
					}
				}
			}
		}
		
		// Get rid of trailing empty cells in lines, and of trailing empty lines.
		boolean onlyEmptyLinesSoFar = true;
		for (int n = data.size() - 1; n >= 0; n--) {
			List<TableTableCellElement> lineCells = data.get(n);
			for (int z = lineCells.size() - 1; z >= 0; z--) {
				if (OdsHelper.isEmptyCell(lineCells.get(z))) {
					lineCells.remove(z);
				} else {
					break;
				}
			}
			
			if (lineCells.size() == 0 && onlyEmptyLinesSoFar) {
				data.remove(n);
			} else {
				onlyEmptyLinesSoFar = false;
			}
		}
		
		for (int n = data.size() - 1; n >= 0; n--) {
			// So far, we only deleted lines *at the end* of the data array, thus the mapping from
			// indices in the rowsRepeatedCounts list to indices in the data list is intact at this point
			// Furthermore, the previous deletion step got rid of all potentially harmful lines with very huge
			// row count, which might occur at the end of the table.  Thus, we can simply clone all lines 
			// according to their row counts.
			int rowsRepeated = rowsRepeatedCounts.get(n);
			
			for (int z = 1; z < rowsRepeated; z++) {
				data.add(n, data.get(n));
			}
		}
		
		return data;
	}	

}
