/***********************************************************
 * $Id: $
 * 
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 11.04.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.odf.util.table;

import java.text.Collator;
import java.util.List;

import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class FancyOdfTableColumn<T> {
    
    private boolean headerColumn;
    private String name;
    private String caption;
    
    public FancyOdfTableColumn(String name, String caption) {
        this.headerColumn = false;
        this.name = name;
        this.caption = caption;
    }

    public FancyOdfTableColumn(boolean headerColumn, String name, String caption) {
        this.headerColumn = headerColumn;
        this.name = name;
        this.caption = caption;
    }    
    
    public boolean isHeaderColumn() {
        return this.headerColumn;
    }
    
    public String getName() {
        return this.name;
    }
    
    public String getCaption() {
        return this.caption;
    }
    
    public abstract void appendCell(TableTableRowElement rowElement, T data);
    
    public abstract int compare(T dataOne, T dataTwo, boolean descending);
    
    private static final Logger log = LoggerFactory.getLogger(FancyOdfTableColumn.class);
    
    /** Returns wether the given data item, with respect to this column, matches
     *  the given filter string.  Both arguments are guaranteed to be non-null.
     *  In case that no filter is given by the user, this function isn't called 
     *  at all by FancyOdfTable.
     * 
     * @param data data item, non-null
     * @param filter filter string, non-null
     * @return wether the cell matching column and data item matches the given filter string.
     */
    public boolean filter(T data, String filter) {
        return true;
    }
    
    public static int defaultStringCompare(Collator collator, String s1, String s2, boolean descending) {
        if (descending) {
            return -defaultStringCompare(collator, s1, s2, false);
        }
        
        if (s1 == null) {
            s1 = "";
        }
        if (s2 == null) {
            s2 = "";
        }
        
        return collator.compare(s1, s2);
    }
    
    public static int defaultStringCompare(Collator collator, String[] listOne, String[] listTwo, boolean descending) {
        String s1 = "";
        String s2 = "";
        for (String l1 : listOne) {
            s1 += l1 + ";";
        }
        for (String l2 : listTwo) {
            s2 += l2 + ";";
        }
        
        return defaultStringCompare(collator, s1, s2, descending);
    }
    
    public static int defaultStringCompare(Collator collator, List<String> listOne, List<String> listTwo, boolean descending) {
        String s1 = "";
        String s2 = "";
        for (String l1 : listOne) {
            s1 += l1 + ";";
        }
        for (String l2 : listTwo) {
            s2 += l2 + ";";
        }
        
        return defaultStringCompare(collator, s1, s2, descending);
    }
    
    public static int defaultLongCompare(Collator collator, Long l1, Long l2, boolean descending) {
        if (l1 == null) {
            return 1;
        } else if (l2 == null) {
            return -1;
        } else {
            long diff = descending ? -(l1 - l2) : l1 - l2;
            if (diff > 0) {
                return 1;
            } else if (diff == 0) {
                return 0;
            } else {
                return -1;
            }
        }
    }    
}
