/***********************************************************
 * $Id: $
 * 
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 11.04.2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.odf.util.table;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.clazzes.odf.util.style.Styles;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderRowsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Node;

public class FancyOdfTable<T> {
    
    private Styles contentAutomaticStyles;
    
    private String tableStyle;  
    public void setTableStyle(String tableStyle) {
        this.tableStyle = tableStyle;
    }
    
    private String headerCellStyle;
    private String headerTextStyle;
    private String dataCellStyle;
    private String dataTextStyle;
    private String dataRowStyle;
    
    private String tableName;
    private String filter;
    
    /** Width in centimeters.
     */
    private Double width;
    
    /** Relative column widths.  Each column will receive width "(its relative width / sum of all relative widths) * table width".
     */
    private Map<String, Double> relativeColumnWidths;
    
    /** If relativeColumnWidths == null, instead the absolute widths set from outside are used.
     */
    private Map<String, Double> absoluteColumnWidths;
    
    /** The table columns. 
     */
    private List<FancyOdfTableColumn<T>> columns;
    
    private boolean renderHeaderLine = true;
    
    private static final Logger log = LoggerFactory.getLogger(FancyOdfTable.class);
    
    private static final double EPSILON = 1e-7;
    
    public FancyOdfTable(String tableName, List<FancyOdfTableColumn<T>> columns) {
        this.contentAutomaticStyles = null;
        this.headerCellStyle = null;
        this.headerTextStyle = null;
        this.dataCellStyle = null;
        this.dataTextStyle = null;
        this.width = null;
        this.relativeColumnWidths = null;
        this.tableName = tableName;
        this.columns = columns;
    }
    
    public FancyOdfTable(Styles contentAutomaticStyles, String headerCellStyle, String headerTextStyle, String dataCellStyle, String dataTextStyle, Double width, Map<String, Double> relativeColumnWidths, List<FancyOdfTableColumn<T>> columns) {
        this.contentAutomaticStyles = contentAutomaticStyles;
        this.headerCellStyle = headerCellStyle;
        this.headerTextStyle = headerTextStyle;
        this.dataCellStyle = dataCellStyle;
        this.dataTextStyle = dataTextStyle;
        this.width = width;
        this.relativeColumnWidths = relativeColumnWidths;
        this.tableName = null;
        this.columns = columns;
    }
    
    public void setDataRowStyle(String dataRowStyle) {
        this.dataRowStyle = dataRowStyle;
    }
    
    public void setAbsoluteColumnWidths(Map<String, Double> absoluteColumnWidths) {
        this.absoluteColumnWidths = absoluteColumnWidths;
    }
    
    public void setFilter(String filter) {
        this.filter = filter;
    }
    
    public void setRenderHeaderLine(boolean renderHeaderLine) {
        this.renderHeaderLine = renderHeaderLine;
    }
    
    public void render(Node parentNode, List<T> data, List<ColumnSortSpec> columnSortSpecs) { 
        this.render(parentNode, data, columnSortSpecs, false);
    }
    
    public void render(Node parentNode, List<T> data, List<ColumnSortSpec> columnSortSpecs, boolean checkInterrupts) {
        // Prevent rendering empty tables.
        if (this.columns.isEmpty()) {
            return;
        }
        
        TableTableElement tableElement = ((OdfFileDom) parentNode.getOwnerDocument()).newOdfElement(TableTableElement.class);
        parentNode.appendChild(tableElement);
        
        this.renderIntoTableElement(tableElement, data, columnSortSpecs, checkInterrupts);
    }
    
    public void renderIntoTableElement(TableTableElement tableElement, List<T> data, List<ColumnSortSpec> columnSortSpecs) {
        this.renderIntoTableElement(tableElement, data, columnSortSpecs, false);
    }
    
    public void renderIntoTableElement(TableTableElement tableElement, List<T> data, List<ColumnSortSpec> columnSortSpecs, boolean checkInterrupts) {
        if (this.tableName != null) {
            tableElement.setTableNameAttribute(this.tableName);            
        }
        if (this.tableStyle != null) {
            tableElement.setTableStyleNameAttribute(this.tableStyle);
        }        

        // Based on this, calculate the absolute width in cm for each column
        if (this.absoluteColumnWidths == null) {
            this.absoluteColumnWidths = new HashMap<String, Double>();
            if (this.width != null && this.relativeColumnWidths != null) {
                // Calculate total sum of relative widths
                double relativeColumnWidthSum = 0d;
                for (Double relativeWidth : relativeColumnWidths.values()) {
                    relativeColumnWidthSum += relativeWidth;
                }
                


                for (String columnName : relativeColumnWidths.keySet()) {
                    double absoluteWidth = (relativeColumnWidths.get(columnName) / relativeColumnWidthSum) * this.width;
                    this.absoluteColumnWidths.put(columnName, absoluteWidth);
                }
                
                // Only consider columns with width greater than zero.  Columns with width == 0 are hidden at client side.
                
                List<FancyOdfTableColumn<T>> visibleColumns = new ArrayList<FancyOdfTableColumn<T>>();
                for (FancyOdfTableColumn<T> column : columns) {
                    String name = column.getName();
                    if (this.absoluteColumnWidths.containsKey(name) && this.absoluteColumnWidths.get(name) > EPSILON) {
                        visibleColumns.add(column);
                    }
                }
                
                columns = visibleColumns;
            }            
        }

        
        List<FancyOdfTableColumn<T>> headerColumns = new ArrayList<FancyOdfTableColumn<T>>();
        for (FancyOdfTableColumn<T> column : columns) {
            if (column.isHeaderColumn()) {
                headerColumns.add(column);
            }
        }
        
        if (headerColumns.size() > 0) {
            TableTableHeaderColumnsElement headerColumnsElement = tableElement.newTableTableHeaderColumnsElement();
            for (FancyOdfTableColumn<T> column : headerColumns) {
                Double absoluteWidth = this.absoluteColumnWidths.get(column.getName()); 
                if (absoluteWidth != null) {                    
                    String formattedWidth = String.format(Locale.ENGLISH, "%.3f", absoluteWidth);
                    log.info("Column [" + column.getName() + "] has width [" + formattedWidth + "]");
                    String tableColumnStyle = contentAutomaticStyles.getTableColumnStyle(contentAutomaticStyles.constructTableColumnPropertiesWithAbsoluteWidth(formattedWidth + "cm"));                     
                    TableTableColumnElement columnElement = headerColumnsElement.newTableTableColumnElement();
                    columnElement.setStyleName(tableColumnStyle);
                } else {
                    headerColumnsElement.newTableTableColumnElement();
                }                
            }            
        }
        
        // TODO: We generate something like
        //   <table:table table:name="local-table">
        //   <table:table-header-columns>
        //     <table:table-column/>
        //   </table:table-header-columns>
        //   <table:table-column/>
        // here --- should be put the lower table-column also into a <table:table-columns>?
        // Append columns with the specified sizes
        for (FancyOdfTableColumn<T> column : columns) {
            Double absoluteWidth = absoluteColumnWidths.get(column.getName());
            log.info("Column [" + column.getName() + "] has width [" + absoluteWidth + "]");
            if (absoluteWidth != null) {
                String formattedWidth = String.format(Locale.ENGLISH, "%.3f", absoluteWidth);
                TableFactory.appendColumnWithAbsoluteSize(contentAutomaticStyles, tableElement, formattedWidth + "cm");                
            } else {
                TableFactory.appendColumnWithNoSize(tableElement);
            }
        }            
        
        Map<String, FancyOdfTableColumn<T>> columnNameToColumn = new HashMap<String, FancyOdfTableColumn<T>>();
        for (FancyOdfTableColumn<T> column : columns) {
            columnNameToColumn.put(column.getName(), column);
        }

        // Set up header
        if (this.renderHeaderLine) {
            TableTableHeaderRowsElement headerRowsElement = tableElement.newTableTableHeaderRowsElement();
            TableTableRowElement headerRowElement = headerRowsElement.newTableTableRowElement();
            for (FancyOdfTableColumn<T> column : columns) {
                TableFactory.appendCell(this.headerCellStyle, this.headerTextStyle, headerRowElement, column.getCaption());
            }            
        }
        
        if (columnSortSpecs != null) {
            for (int z = columnSortSpecs.size() - 1; z >= 0; z--) {
                ColumnSortSpec sortSpec = columnSortSpecs.get(z);
                final FancyOdfTableColumn<T> column = columnNameToColumn.get(sortSpec.getProperty());
                final boolean descending = sortSpec.getDescending() != null ? sortSpec.getDescending().booleanValue() : false;
                if (column != null) {
                    Collections.sort(data, new Comparator<T>() {
                        public int compare(T o1, T o2) {
                            return column.compare(o1, o2, descending);
                        }
                    });                
                }
            }            
        }
        
        // Render data cells
        for (T t : data) {
            boolean add;
            if (this.filter != null) {
                add = false;
                for (FancyOdfTableColumn<T> column : columns) {
                    add |= column.filter(t, this.filter);
                }                
            } else {
                add = true;
            }            
            
            if (add) {
                TableTableRowElement rowElement = TableFactory.appendRow(tableElement, dataRowStyle);
                
                for (FancyOdfTableColumn<T> column : columns) {
                    column.appendCell(rowElement, t);
                }                
            }
            
            if (checkInterrupts && Thread.currentThread().isInterrupted()) {
                throw new RuntimeException("Table generation was interrupted.");
            }
        }
    }
}
