/***********************************************************
 * $Id: $
 * 
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 10.04.2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.odf.util.style;

import java.util.HashMap;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import org.clazzes.odf.util.voc.StyleFamily;
import org.odftoolkit.odfdom.dom.element.OdfStylePropertiesBase;
import org.odftoolkit.odfdom.dom.element.number.NumberTextElement;
import org.odftoolkit.odfdom.dom.element.style.StyleChartPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleGraphicPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleParagraphPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleStyleElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableCellPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableColumnPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTablePropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableRowPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTextPropertiesElement;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class Styles {
    
    private StyleRepository styleRepository;
    private String stylePrefix = null;
    private OdfElement styles = null;
    
    public Styles(StyleRepository styleRepository, OdfElement styles, String stylePrefix) {
        this.styleRepository = styleRepository;
        this.stylePrefix = stylePrefix;
        this.styles = styles;
    }
    
    public OdfElement getStylesElement() {
        return this.styles;
    }
    
    private String getStyleKey(String family, String className, String parent, Map<String, String> attributeNameToValue, OdfStylePropertiesBase... properties) {
        
        SortedMap<String, String> attributeToValue = new TreeMap<String, String>();
                
        attributeToValue.put("style:family", family);
        
        if (className != null) {
            attributeToValue.put("style:class", className);    
        }
        if (parent != null) {
            attributeToValue.put("style:parent-style-name", parent);    
        }        
        if (attributeNameToValue != null && attributeNameToValue.containsKey("data-style-name")) {
            attributeToValue.put("style:data-style-name", attributeNameToValue.get("data-style-name"));
        }
       
        
        for (OdfStylePropertiesBase property : properties) {
            this.registerAttributes(property, attributeToValue, null);
        }
        
        return this.getKeyFromSortedMap(attributeToValue);
    }    
    
    private String getStyleKey(String family, OdfElement parent, OdfElement... children) {
        SortedMap<String, String> attributeToValue = new TreeMap<String, String>();
        
        attributeToValue.put("style:family", family);
        this.registerAttributes(parent, attributeToValue, "parent");
        this.registerChildNodes(parent, attributeToValue);
        
        return this.getKeyFromSortedMap(attributeToValue);
    }
    
    private void registerChildNodes(OdfElement parentElement, SortedMap<String, String> attributeToValue) {
        NodeList nodeList = parentElement.getChildNodes();
        for (int n = 0; n < nodeList.getLength(); n++) {
            Node node = nodeList.item(n);
            if (node instanceof NumberTextElement) {
                NumberTextElement numberTextElement = (NumberTextElement)node;
                attributeToValue.put("child" + n + "__textcontent", numberTextElement.getTextContent());
            }
            this.registerAttributes(node, attributeToValue, "child" + n);
        }
    }
    
    /** Registers attributes for the given Node in the given map.  Lateron, based on the map
     *  contents, a unique key for exactly this styles (with its attribute values) will be 
     *  determined.  If the map lateron contains attributes from multiple Nodes (e.g. 
     *  because this function was called for multiple Nodes), and the danger of key conflicts
     *  exists, they need to have different keys.  For this case, an optional prefix can be 
     *  passed.  If the prefix is passed, all attribute keys will be determined using the
     *  rule "prefix" + "__" + "attribute-name"
     * 
     * @param odfElement
     * @param attributeToValue
     * @param prefix
     */
    private void registerAttributes(Node odfElement, SortedMap<String, String> attributeToValue, String prefix) {
        NamedNodeMap nodeMap = odfElement.getAttributes();
        for (int n = 0; n < nodeMap.getLength(); n++) {
            Node node = nodeMap.item(n);
            
            String name = (prefix != null ? prefix + "__" : "") + node.getNodeName();
            String value = node.getNodeValue();
            
            attributeToValue.put(name, value);
        }        
    }
    
    private String getKeyFromSortedMap(SortedMap<String, String> attributeToValue) {
        StringBuffer keyBuffer = new StringBuffer();
        for (String attribute : attributeToValue.keySet()) {
            String value = attributeToValue.get(attribute);
            keyBuffer.append(attribute + "=" + value + ";");
        }
        
        String key = keyBuffer.toString();
        return key;        
    }
    
    private StyleStyleElement constructStyleElement(String family, String styleName, String dataStyleName) {
        StyleStyleElement styleStyle = (StyleStyleElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleStyleElement.class);
        styleStyle.setStyleFamilyAttribute(family);
        styleStyle.setStyleNameAttribute(styleName);
        if (dataStyleName != null) {
            styleStyle.setStyleDataStyleNameAttribute(dataStyleName);
        }
        
        this.styles.appendChild(styleStyle);
        return styleStyle;        
    }
    
    public String getStyle(String family, String parentStyle, String givenStyleName, String styleClass, OdfStylePropertiesBase... properties) {
        
        String styleKey = this.getStyleKey(family, styleClass, null, null, properties);
        if (!this.styleRepository.containsKey(styleKey)) {
            String styleName = this.stylePrefix + "_" + (givenStyleName != null ? givenStyleName : family + this.styleRepository.size());            
            
            if (this.styleRepository.containsStyleName(styleName)) {
                throw new IllegalArgumentException("Trying to register style [" + styleName + "] (key [" + styleKey + "]) for the second time; "
                                                 + "most probably you explicitely specified a already used name.");
            }
            
            StyleStyleElement styleElement = this.constructStyleElement(family, styleName, null);
            if (styleClass != null) {
                styleElement.setStyleClassAttribute(styleClass);
            }
            if (parentStyle != null) {
                styleElement.setStyleParentStyleNameAttribute(parentStyle);
            }
            
            for (OdfStylePropertiesBase property : properties) {
                styleElement.appendChild(property);
            }
            
            this.styleRepository.register(styleKey, styleName);
        }
        
        return this.styleRepository.getStyleName(styleKey);
    }
    
    public String getStyle(String family, String parentStyle, String styleClass, Map<String, String> attributeNameToValue, OdfStylePropertiesBase... properties) {
        
        String styleKey = this.getStyleKey(family, styleClass, null, attributeNameToValue, properties);
        if (!this.styleRepository.containsKey(styleKey)) {
            String styleName = this.stylePrefix + "_" + family + this.styleRepository.size();         
            
            if (this.styleRepository.containsStyleName(styleName)) {
                throw new IllegalArgumentException("Trying to register style [" + styleName + "] (key [" + styleKey + "]) for the second time; "
                                                 + "most probably you explicitely specified a already used name.");
            }
            
            StyleStyleElement styleElement = this.constructStyleElement(family, styleName, attributeNameToValue.get("data-style-name"));
            if (styleClass != null) {
                styleElement.setStyleClassAttribute(styleClass);
            }
            if (parentStyle != null) {
                styleElement.setStyleParentStyleNameAttribute(parentStyle);
            }
            
            for (OdfStylePropertiesBase property : properties) {
                styleElement.appendChild(property);
            }
            
            this.styleRepository.register(styleKey, styleName);
        }
        
        return this.styleRepository.getStyleName(styleKey);
    }    
    
    public String getStyle(String family, OdfStylePropertiesBase... properties) {
        return this.getStyle(family, null, null, (String)null, properties);
    }
    
    public String getStyle(String family, String givenStyleName, OdfStylePropertiesBase... properties) {
        return this.getStyle(family, null, givenStyleName, (String)null, properties);
    }
    
    public String getStyle(String family, String givenStyleName, String styleClass, OdfStylePropertiesBase... properties) {
        return this.getStyle(family, null, givenStyleName, styleClass, properties);
    }
    
    public String getStyleWithClass(String family, String styleClass, OdfStylePropertiesBase... properties) {
        return this.getStyle(family, null, null, styleClass, properties);
    }
    
    public String getStyleWithParent(String family, String parentStyle, OdfStylePropertiesBase... properties) {
        return this.getStyle(family, parentStyle, null, (String)null, properties);
    }
    
    public String getStyleWithParent(String family, String parentStyle, String styleClass, OdfStylePropertiesBase... properties) {
        return this.getStyle(family, parentStyle, null, styleClass, properties);
    }
    
    public String getStyle(StyleFamily family, OdfElement parent, OdfElement... children) {
        String styleKey = this.getStyleKey(family.getValue(), parent, children);
        if (!this.styleRepository.containsKey(styleKey)) {
            String styleName = this.stylePrefix + "_" + family.getValue() + this.styleRepository.size();            
            
            if (this.styleRepository.containsStyleName(styleName)) {
                throw new IllegalArgumentException("Trying to register style [" + styleName + "] (key [" + styleKey + "]) for the second time");
            }
            
            this.styles.appendChild(parent);
            for (OdfElement child : children) {
                parent.appendChild(child);
            }
            
            this.styleRepository.register(styleKey, styleName);
        }
        
        String styleName = this.styleRepository.getStyleName(styleKey);        
        parent.setAttributeNS(null, "style:name", styleName);
        return styleName;
    }
    
    /************************************************************* Style Getters ***********************************************************/
    
    public String getChartStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.CHART.getValue(), properties);
    }
    
    public String getChartStyleWithDataStyle(String dataStyleName, OdfStylePropertiesBase... properties) {
        Map<String, String> attributeNameToValue = new HashMap<String, String>();
        attributeNameToValue.put("data-style-name", dataStyleName);
        return this.getStyle(StyleFamily.CHART.getValue(), null, null, attributeNameToValue, properties);
    }
    
    public String getGraphicStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.GRAPHIC.getValue(), properties);
    }
    
    public String getGraphicStyle(String givenStyleName, OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.GRAPHIC.getValue(), givenStyleName, properties);
    }    
    
    public String getGraphicStyleWithParent(String parentStyle, OdfStylePropertiesBase... properties) {
        return this.getStyleWithParent(StyleFamily.GRAPHIC.getValue(), parentStyle, properties);
    }   
    
    public String getPageLayoutStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.PAGE_LAYOUT.getValue(), properties);
    }
    
    public String getParagraphStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.PARAGRAPH.getValue(), properties);
    }
    
    public String getParagraphStyleWithClass(String styleClass, OdfStylePropertiesBase... properties) {
        return this.getStyleWithClass(StyleFamily.PARAGRAPH.getValue(), styleClass, properties);
    }
    
    public String getParagraphStyle(String givenStyleName, String styleClass, OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.PARAGRAPH.getValue(), givenStyleName, styleClass, properties);
    }
    
    public String getParagraphStyleWithParent(String parentStyle, String styleClass, OdfStylePropertiesBase... properties) {
        return this.getStyleWithParent(StyleFamily.PARAGRAPH.getValue(), parentStyle, styleClass, properties);
    }
    
    public String getParagraphStyleWithParent(String parentStyle, OdfStylePropertiesBase... properties) {
        return this.getStyleWithParent(StyleFamily.PARAGRAPH.getValue(), parentStyle, properties);
    }
    
    public String getTableStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.TABLE.getValue(), properties);
    }
    
    public String getTableCellStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.TABLE_CELL.getValue(), properties);
    }
    
    public String getTableColumnStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.TABLE_COLUMN.getValue(), properties);
    }
    
    public String getTableRowStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.TABLE_ROW.getValue(), properties);
    }
    
    public String getTextStyle(OdfStylePropertiesBase... properties) {
        return this.getStyle(StyleFamily.TEXT.getValue(), properties);
    }
    
    /**************************************************** Construction of *PropertiesElement ***********************************************/

    // StyleChartPropertiesElement
    public StyleChartPropertiesElement constructChartProperties() {
        return (StyleChartPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleChartPropertiesElement.class);
    }    
    
    public StyleChartPropertiesElement constructChartPropertiesWithAutoPosition(Boolean autoPosition) {
        StyleChartPropertiesElement chartProperties = this.constructChartProperties();
        if (autoPosition != null) {
            chartProperties.setChartAutoPositionAttribute(autoPosition);
        }
        return chartProperties;
    }
    
    // StyleParagraphPropertiesElement 
    public StyleParagraphPropertiesElement constructParagraphProperties() {
        return (StyleParagraphPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleParagraphPropertiesElement.class);
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithMargin(String marginLeft, String marginRight, String marginTop, String marginBottom) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        if (marginLeft != null) {
            paragraphProperties.setFoMarginLeftAttribute(marginLeft);
        }
        if (marginRight != null) {
            paragraphProperties.setFoMarginRightAttribute(marginRight);
        }
        if (marginTop != null) {
            paragraphProperties.setFoMarginTopAttribute(marginTop);
        }
        if (marginBottom != null) {
            paragraphProperties.setFoMarginBottomAttribute(marginBottom);
        }
        return paragraphProperties;
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithBackgroundColor(String backgroundColor) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        paragraphProperties.setFoBackgroundColorAttribute(backgroundColor);
        return paragraphProperties;
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithLineNumbers(Boolean numberLines, Integer lineNumber) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        paragraphProperties.setTextNumberLinesAttribute(numberLines);
        paragraphProperties.setTextLineNumberAttribute(lineNumber);
        return paragraphProperties;        
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithTextAlign(String textAlign) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        paragraphProperties.setFoTextAlignAttribute(textAlign);
        return paragraphProperties;        
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithTextAlign(String textAlign, Boolean justifySingleWords) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        if (textAlign != null) {
            paragraphProperties.setFoTextAlignAttribute(textAlign);    
        }
        if (justifySingleWords != null) {
            paragraphProperties.setStyleJustifySingleWordAttribute(justifySingleWords);
        }
        
        return paragraphProperties;        
    }    
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithTextAlignAndMargin(String textAlign, Boolean justifySingleWords, 
            String marginLeft, String marginRight, String marginTop, String marginBottom) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        if (textAlign != null) {
            paragraphProperties.setFoTextAlignAttribute(textAlign);    
        }
        if (justifySingleWords != null) {
            paragraphProperties.setStyleJustifySingleWordAttribute(justifySingleWords);
        }
        if (marginLeft != null) {
            paragraphProperties.setFoMarginLeftAttribute(marginLeft);
        }
        if (marginRight != null) {
            paragraphProperties.setFoMarginRightAttribute(marginRight);
        }
        if (marginTop != null) {
            paragraphProperties.setFoMarginTopAttribute(marginTop);
        }
        if (marginBottom != null) {
            paragraphProperties.setFoMarginBottomAttribute(marginBottom);
        }        
        
        return paragraphProperties;        
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithVerticalAlign(String verticalAlign, boolean snapToLayoutGrid) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        paragraphProperties.setStyleVerticalAlignAttribute(verticalAlign);
        paragraphProperties.setStyleSnapToLayoutGridAttribute(snapToLayoutGrid);
        return paragraphProperties;
    }
    
    public StyleParagraphPropertiesElement constructParagraphPropertiesWithBreakBefore(String breakBefore) {
        StyleParagraphPropertiesElement paragraphProperties = this.constructParagraphProperties();
        if (breakBefore != null) {
            paragraphProperties.setFoBreakBeforeAttribute(breakBefore);
        }
        return paragraphProperties;        
    }
    
    // StyleTableCellPropertiesElement
    public StyleTableCellPropertiesElement constructTableCellProperties() {
        return (StyleTableCellPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleTableCellPropertiesElement.class);
    }
    
    public StyleTableCellPropertiesElement constructTableCellPropertiesWithPaddingAndBorders(String padding, String border) {
        StyleTableCellPropertiesElement properties = this.constructTableCellProperties();
        properties.setFoPaddingAttribute(padding);
        properties.setFoBorderLeftAttribute(border);
        properties.setFoBorderRightAttribute(border);
        properties.setFoBorderBottomAttribute(border);
        properties.setFoBorderTopAttribute(border);
        return properties;
    }
    
    public StyleTableCellPropertiesElement constructTableCellPropertiesWithColorPaddingAndBorders(String backgroundColor, String padding, String border) {
        StyleTableCellPropertiesElement properties = this.constructTableCellProperties();
        properties.setFoBackgroundColorAttribute(backgroundColor);
        properties.setFoPaddingAttribute(padding);
        properties.setFoBorderLeftAttribute(border);
        properties.setFoBorderRightAttribute(border);
        properties.setFoBorderBottomAttribute(border);
        properties.setFoBorderTopAttribute(border);
        return properties;
    }    
    
    public StyleTableCellPropertiesElement constructTableCellPropertiesWithPaddingAndBorders(String paddingLeft, String borderLeft,
                                                                                             String paddingRight, String borderRight,
                                                                                             String paddingTop, String borderTop,
                                                                                             String paddingBottom, String borderBottom) {
        StyleTableCellPropertiesElement properties = this.constructTableCellProperties();
        if (paddingLeft != null) {
            properties.setFoPaddingLeftAttribute(paddingLeft);
        }
        if (borderLeft != null) {
            properties.setFoBorderLeftAttribute(borderLeft);
        }
        if (paddingRight != null) {
            properties.setFoPaddingRightAttribute(paddingRight);
        }
        if (borderRight != null) {
            properties.setFoBorderRightAttribute(borderRight);
        }
        if (paddingTop != null) {
            properties.setFoPaddingTopAttribute(paddingTop);
        }
        if (borderTop != null) {
            properties.setFoBorderTopAttribute(borderTop);
        }
        if (paddingBottom != null) {
            properties.setFoPaddingBottomAttribute(paddingBottom);
        }
        if (borderBottom != null) {
            properties.setFoBorderBottomAttribute(borderBottom);
        }
        properties.setStyleWritingModeAttribute("lr-tb");// TODO Test
        return properties;
    }
    
    // StyleGraphicPropertiesElement
    public StyleGraphicPropertiesElement constructGraphicProperties() {
        return (StyleGraphicPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleGraphicPropertiesElement.class);
    }
    
    public StyleGraphicPropertiesElement constructGraphicPropertiesWithColor(String strokeColor, String fillColor) {
        StyleGraphicPropertiesElement graphicProperties = this.constructGraphicProperties();
        
        if (strokeColor != null) {
            graphicProperties.setSvgStrokeColorAttribute(strokeColor);
        }        
        if (fillColor != null) {
            graphicProperties.setDrawFillColorAttribute(fillColor);
        }        
        
        return graphicProperties;
    }
    
    public StyleGraphicPropertiesElement constructGraphicPropertiesWithStrokeAndFill(String stroke, String fill) {
        StyleGraphicPropertiesElement graphicProperties = this.constructGraphicProperties();
        if (stroke != null) {
            graphicProperties.setDrawStrokeAttribute(stroke);
        }
        if (fill != null) {
            graphicProperties.setDrawFillAttribute(fill);
        }
        return graphicProperties;
    }
    
    public StyleGraphicPropertiesElement constructGraphicPropertiesWithStrokeAndFillAndColor(String stroke, String strokeColor, String fill, String fillColor) {
        StyleGraphicPropertiesElement graphicProperties = this.constructGraphicProperties();
        if (stroke != null) {
            graphicProperties.setDrawStrokeAttribute(stroke);
        }
        if (strokeColor != null) {
            graphicProperties.setSvgStrokeColorAttribute(strokeColor);
        }        
        if (fill != null) {
            graphicProperties.setDrawFillAttribute(fill);
        }
        if (fillColor != null) {
            graphicProperties.setDrawFillColorAttribute(fillColor);
        }
        return graphicProperties;
    }    
    
    public StyleGraphicPropertiesElement constructGraphicPropertiesWithAlignment(String verticalPos, String verticalRel,
            String horizontalPos, String horizontalRel, String textAreaVerticalAlign) {
        StyleGraphicPropertiesElement graphicProperties = this.constructGraphicProperties();
        if (verticalPos != null) {
            graphicProperties.setStyleVerticalPosAttribute(verticalPos);
        }
        if (verticalRel != null) {
            graphicProperties.setStyleVerticalRelAttribute(verticalRel);
        }
        if (horizontalPos != null) {
            graphicProperties.setStyleHorizontalPosAttribute(horizontalPos);
        }
        if (horizontalRel != null) {
            graphicProperties.setStyleHorizontalRelAttribute(horizontalRel);
        }
        if (textAreaVerticalAlign != null) {
            graphicProperties.setDrawTextareaVerticalAlignAttribute(textAreaVerticalAlign);
        }
        
        return graphicProperties;
    }    
    
    // StyleTablePropertiesElement
    public StyleTablePropertiesElement constructTableProperties() {
        return (StyleTablePropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleTablePropertiesElement.class);
    }   
    
    public StyleTablePropertiesElement constructTablePropertiesWithMargin(String marginLeft, String marginRight, String marginTop, String marginBottom) {
        StyleTablePropertiesElement tableProperties = this.constructTableProperties();
        if (marginLeft != null) {
            tableProperties.setFoMarginLeftAttribute(marginLeft);
        }
        if (marginRight != null) {
            tableProperties.setFoMarginRightAttribute(marginRight);
        }
        if (marginTop != null) {
            tableProperties.setFoMarginTopAttribute(marginTop);
        }
        if (marginBottom != null) {
            tableProperties.setFoMarginBottomAttribute(marginBottom);
        }
        return tableProperties;        
    }
    
    public StyleTablePropertiesElement constructTableProperties(String width, String marginLeft, String align, String writingMode) {
        StyleTablePropertiesElement tableProperties = this.constructTableProperties();
        if (width != null) {
            tableProperties.setStyleWidthAttribute(width);    
        }
        if (marginLeft != null) {
            tableProperties.setFoMarginLeftAttribute(marginLeft);    
        }
        if (align != null) {
            tableProperties.setTableAlignAttribute(align);    
        }
        if (writingMode != null) {
            tableProperties.setStyleWritingModeAttribute(writingMode);    
        }        
        return tableProperties;
    }
    
    // StyleTableColumnPropertiesElement
    public StyleTableColumnPropertiesElement constructTableColumnProperties() {
        return (StyleTableColumnPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleTableColumnPropertiesElement.class);
    }
    
    public StyleTableColumnPropertiesElement constructTableColumnPropertiesWithRelativeWidth(String relativeWidth) {
        StyleTableColumnPropertiesElement properties = this.constructTableColumnProperties();
        properties.setStyleRelColumnWidthAttribute(relativeWidth);
        return properties;
    }
    
    public StyleTableColumnPropertiesElement constructTableColumnPropertiesWithAbsoluteWidth(String absoluteWidth) {
        StyleTableColumnPropertiesElement properties = this.constructTableColumnProperties();
        properties.setStyleColumnWidthAttribute(absoluteWidth);
        return properties;
    }
    
    public StyleTableColumnPropertiesElement constructTableColumnPropertiesWithOptimalWidth() {
        StyleTableColumnPropertiesElement properties = this.constructTableColumnProperties();
        properties.setStyleUseOptimalColumnWidthAttribute(true);
        return properties;
    }
    
    // StyleTableRowPropertiesElement
    public StyleTableRowPropertiesElement constructTableRowProperties() {
        return (StyleTableRowPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleTableRowPropertiesElement.class);
    }    
    
    public StyleTableRowPropertiesElement constructTableRowPropertiesWithAlwaysKeepTogether() {
        StyleTableRowPropertiesElement properties = this.constructTableRowProperties();
        properties.setFoKeepTogetherAttribute("always");
        return properties;
    }
    
    public StyleTableRowPropertiesElement constructTableRowPropertiesWithOptimalRowHeight() {
        StyleTableRowPropertiesElement properties = this.constructTableRowProperties();
        properties.setStyleUseOptimalRowHeightAttribute(true);
        return properties;        
    }
    
    public StyleTableRowPropertiesElement constructTableRowPropertiesWithBackgroundColor(String backgroundColor, String keepTogether) {
        StyleTableRowPropertiesElement properties = this.constructTableRowProperties();
        properties.setFoBackgroundColorAttribute(backgroundColor);
        properties.setFoKeepTogetherAttribute(keepTogether);
        return properties;
    }
    
    public StyleTableRowPropertiesElement constructTableRowPropertiesWithMinHeight(String minHeight) {
        StyleTableRowPropertiesElement properties = this.constructTableRowProperties();
        properties.setStyleMinRowHeightAttribute(minHeight);
        return properties;        
    }
    
    // StyleTextPropertiesElement
    public StyleTextPropertiesElement constructTextProperties() {
        return (StyleTextPropertiesElement)((OdfFileDom)this.styles.getOwnerDocument()).newOdfElement(StyleTextPropertiesElement.class);
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithColor(String textColor) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        textProperties.setFoColorAttribute(textColor);
        return textProperties;
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithFontSize(String fontSize) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        textProperties.setFoFontSizeAttribute(fontSize);
        return textProperties;
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithFontSize(String fontSize, String fontWeight) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        textProperties.setFoFontSizeAttribute(fontSize);
        textProperties.setFoFontWeightAttribute(fontWeight);
        return textProperties;
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithColorAndFontSize(String textColor, String fontSize) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        textProperties.setFoColorAttribute(textColor);
        textProperties.setFoFontSizeAttribute(fontSize);
        return textProperties;        
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithColorAndFont(String textColor, String fontFamily, String fontSize, String fontWeight) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        if (textColor != null) {
            textProperties.setFoColorAttribute(textColor);    
        }
        if (fontFamily != null) {
            textProperties.setFoFontFamilyAttribute(fontFamily);    
        }
        if (fontSize != null) {
            textProperties.setFoFontSizeAttribute(fontSize);    
        }
        if (fontWeight != null) {
            textProperties.setFoFontWeightAttribute(fontWeight);    
        }        
        return textProperties;        
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithColorFontAndHyphenation(String textColor, String fontFamily, String fontSize, String fontWeight) {
        StyleTextPropertiesElement textProperties = this.constructTextPropertiesWithColorAndFont(textColor, fontFamily, fontSize, fontWeight);
        textProperties.setFoHyphenateAttribute(true);
        textProperties.setFoHyphenationRemainCharCountAttribute(2);
        textProperties.setFoHyphenationPushCharCountAttribute(2);
        return textProperties;
    }
    
    public StyleTextPropertiesElement constructStrikeThroughTextPropertiesWithColorAndFont(String textColor, String fontFamily, String fontSize, String fontWeight) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        if (textColor != null) {
            textProperties.setFoColorAttribute(textColor);    
        }
        if (fontFamily != null) {
            textProperties.setFoFontFamilyAttribute(fontFamily);    
        }
        if (fontSize != null) {
            textProperties.setFoFontSizeAttribute(fontSize);    
        }
        if (fontWeight != null) {
            textProperties.setFoFontWeightAttribute(fontWeight);    
        }        
        textProperties.setStyleTextLineThroughStyleAttribute("solid");
        textProperties.setStyleTextLineThroughTypeAttribute("single");
        return textProperties;        
    }    
    
    public StyleTextPropertiesElement constructStrikeThroughTextPropertiesWithColorFontAndHyphenation(String textColor, String fontFamily, String fontSize, String fontWeight) {
        StyleTextPropertiesElement textProperties = this.constructStrikeThroughTextPropertiesWithColorAndFont(textColor, fontFamily, fontSize, fontWeight);
        textProperties.setFoHyphenateAttribute(true);
        textProperties.setFoHyphenationRemainCharCountAttribute(2);
        textProperties.setFoHyphenationPushCharCountAttribute(2);
        return textProperties;
    }    
    
    public StyleTextPropertiesElement constructTextPropertiesWithFont(String fontName, String fontFamily, String fontSize, String fontWeight, String fontPitch) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        if (fontName != null) {
            textProperties.setStyleFontNameAttribute(fontName);
        }
        if (fontFamily != null) {
            textProperties.setFoFontFamilyAttribute(fontFamily);    
        }
        if (fontSize != null) {
            textProperties.setFoFontSizeAttribute(fontSize);    
        }
        if (fontWeight != null) {
            textProperties.setFoFontWeightAttribute(fontWeight);    
        }        
        if (fontPitch != null) {
            textProperties.setStyleFontPitchAttribute(fontPitch);
        }
        return textProperties;        
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithFontWeight(String fontWeight) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        textProperties.setFoFontWeightAttribute(fontWeight);
        return textProperties;        
    }
    
    public StyleTextPropertiesElement constructTextPropertiesWithFontWeightAndScale(String fontWeight, String scale) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        textProperties.setFoFontWeightAttribute(fontWeight);
        textProperties.setStyleTextScaleAttribute(scale);
        return textProperties;        
    }
    
    public StyleTextPropertiesElement constructUnderlineTextProperties(String underlineStyle, String underlineWidth, String underlineColor) {
        StyleTextPropertiesElement textProperties = this.constructTextProperties();
        if (underlineStyle != null) {
            textProperties.setStyleTextUnderlineStyleAttribute(underlineStyle);    
        }
        if (underlineWidth != null) {
            textProperties.setStyleTextUnderlineWidthAttribute(underlineWidth);
        }
        if (underlineColor != null) {
            textProperties.setStyleTextUnderlineColorAttribute(underlineColor);
        }        
        return textProperties;
    }
    
    /**************** Construction of style column / row elements ***************************/
    
    
}
