/***********************************************************
 *
 * Utility code for dealing with odf files using odfdom etc.
 * http://www.clazzes.org
 *
 * Created: 02.02.2018
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.odf.util.imp;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.clazzes.odf.util.table.OdsHelper;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.doc.table.OdfTable;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class XlsImporter {

    private static final Logger log = LoggerFactory.getLogger(XlsImporter.class);

    public static List<List<String>> getCellContents(File file) {
        List<List<String>> contents = new ArrayList<List<String>>();

        List<List<TableTableCellElement>> cells = XlsImporter.getCells(file);
        for (List<TableTableCellElement> row : cells) {
            List<String> rowContents = new ArrayList<String>();
            for (TableTableCellElement cell : row) {
                String value = XlsImporter.getCellString(cell);
                rowContents.add(value != null ? value.trim() : null);
            }
            contents.add(rowContents);
        }

        return contents;
    }

    /** Returns a two dimensional list of cells (first index rows, second index columns)
     *  containing all cells of first table of the given Ods file.
     *
     * @param file some Ods file
     * @return cells of the first table of the Ods file as described
     */
    public static List<List<TableTableCellElement>> getCells(File file) {
        OdfSpreadsheetDocument document;

        try {
            document = OdfSpreadsheetDocument.loadDocument(file);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        try {
            List<OdfTable> tables = document.getTableList();
            if (tables.size() == 0) {
                throw new IllegalArgumentException("Document needs to contain at least one table.");
            }

            OdfTable table = tables.get(0);
            TableTableElement tableElement = table.getOdfElement();

            return OdsHelper.getTableData(tableElement);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static String getCellString(TableTableCellElement cell) {
        String value = OdsHelper.getStringValue(cell);
        return value;
    }
}
